/*
 * This code was inspired by the Infinity Source code Marathon 2: register addresses and names were written down to paper.
 * This code was written independently from the Infinity Source and does not incorporate any portion of the original GPLv3-licensed code.
 * Therefore, it is not bound by the GPLv3 license.
 * 
 * This work is dedicated to the public domain under the Creative Commons CC0 1.0 Universal (CC0 1.0) Public Domain Dedication.
 * To view a copy of this dedication, visit http://creativecommons.org/publicdomain/zero/1.0/ or send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
 * 
 * Disclaimer: The author has made every effort to ensure that this code was developed independently and is not a derivative work of any GPLv3-licensed code. If any part of this code is found to be in violation of GPLv3, the author will comply with the requirements of the GPLv3 license.
 *
 * By Julien Aubert (a.k.a noglin)
 */
#include <stdint.h>
#include "mac_valkyrie.h"


void *valkyrie_buffer_video0 = (void *)0xf9300000;
void *valkyrie_buffer_video1 = (void *)0xf9340000;
void *valkyrie_buffer_screen = (void *)0xf9001000;

volatile uint8_t *valkyrie_reg_subsystem_config     = (volatile  uint8_t *)0x50f2a00c;
volatile uint8_t *valkyrie_reg_video_in_control     = (volatile  uint8_t *)0x50f2a020;
volatile uint16_t *valkyrie_reg_video_in_x0         = (volatile uint16_t *)0x50f2a060;
volatile uint16_t *valkyrie_reg_video_in_y0         = (volatile uint16_t *)0x50f2a064;
volatile uint16_t *valkyrie_reg_video_in_width      = (volatile uint16_t *)0x50f2a070;
volatile uint16_t *valkyrie_reg_video_in_height     = (volatile uint16_t *)0x50f2a074;
volatile uint16_t *valkyrie_reg_video_in_field_x0   = (volatile uint16_t *)0x50f2a080;
volatile uint16_t *valkyrie_reg_video_in_field_y0   = (volatile uint16_t *)0x50f2a084;
volatile uint8_t *valkyrie_reg_clut_color_key       = (volatile void *)0x50f2400c;

uint8_t valkyrie_reg_subsystem_config_initial;
uint8_t valkyrie_reg_video_in_control_initial;
uint16_t valkyrie_reg_video_in_x0_initial;
uint16_t valkyrie_reg_video_in_y0_initial;
uint16_t valkyrie_reg_video_in_width_initial;
uint16_t valkyrie_reg_video_in_height_initial;
uint16_t valkyrie_reg_video_in_field_x0_initial;
uint16_t valkyrie_reg_video_in_field_y0_initial;
uint8_t valkyrie_reg_clut_color_key_initial;

/* this 8bpp pixel value in screen will let current video through */
uint8_t valkyrie_screen_video_key = 0x0;
uint8_t valkyrie_screen_video_key_initial;
uint8_t valkyrie_screen_buffer_key_initial;
uint16_t valkyrie_backbuffer_value_initial_xy00;
int valkyrie_page = 0;


int valkyrie_available(void) {
	OSErr err;
	long gestalt;
	err = Gestalt(gestaltMachineType, &gestalt);
	if (err != noErr) {
		printf("Gestalt failed\n");
		exit(1);
	}
	/* gestaltTVAttr -> gestaltHasVidDecoderScaler ?? maybe all are same Valkyrie chip? */
    /* models with Valkyrie according to Appl Developer notes: 63X/52XX/53XX/62XX/63XX
       54XX/55XX seems to have it too but slightly modified perhaps? */
	switch (gestalt) {
		case gestaltQuadra630:
		case gestaltPowerMac5200:
/*		case gestaltPerforma5300: same gestalt value as 5200 */ 
		case gestaltPowerMac6200:
/*		case gestaltPerforma6300: same gestalt value as 6200 */
		case gestaltPowerMac5400: /* "Valkyrie-AR" same?? */
			return 1;
		default:
			return 0;
	}
}

void valkyrie_clear(uint16_t backbuffer_value, uint8_t screen_value) {
    {
        int x, y;
        for (y = 0; y < 240; ++y) {
            uint8_t *b0 = (uint8_t *)valkyrie_buffer_video0 + VALKYRIE_BUFFER_VIDEO_ROWBYTES * y;
            uint8_t *b1 = (uint8_t *)valkyrie_buffer_video1 + VALKYRIE_BUFFER_VIDEO_ROWBYTES * y;
            uint16_t *dst0 = (void *)b0;
            uint16_t *dst1 = (void *)b1;
            for (x = 0; x < 320; x++) {
                *dst0++ = backbuffer_value;
                *dst1++ = backbuffer_value;
            }
        }
    }
    {
        GDHandle mainDevice = GetMainDevice();
        PixMapHandle pixmap;
        long rowbytes;
        Ptr framebuffer;
        int bpp;
        int h = 480;
        int w = 640;
        int y;
        pixmap = (*mainDevice)->gdPMap;
        rowbytes = ((*pixmap)->rowBytes) & 0x3FFF;
        framebuffer = (*pixmap)->baseAddr;
        if (framebuffer != valkyrie_buffer_screen) {
            /* NOTE: main device is the primary device (not the one user selected in monitors & sound), if the computer has valkyrie it should be this device? */
            printf("huh, seems GetMainDevice is not the one that valkyrie is on??\n");
            exit(1);
        }
        bpp = (*pixmap)->pixelSize;
        valkyrie_screen_buffer_key_initial = *(uint8_t *)valkyrie_buffer_screen;
        for (y = 0; y < h; ++y) {
            uint8_t *dst8 = (uint8_t *)valkyrie_buffer_screen + y * rowbytes;
            int x;
            for (x = 0; x < w; ++x) {
                *dst8++ = screen_value;
            }
        }
    }
}

void valkyrie_init(uint8_t pixel_video_key) {
    valkyrie_page = 0;
    valkyrie_screen_video_key_initial = *valkyrie_reg_clut_color_key;
    valkyrie_reg_subsystem_config_initial = *valkyrie_reg_subsystem_config;
    valkyrie_reg_video_in_control_initial = *valkyrie_reg_video_in_control;
    valkyrie_reg_video_in_x0_initial = *valkyrie_reg_video_in_x0;
    valkyrie_reg_video_in_y0_initial = *valkyrie_reg_video_in_y0;
    valkyrie_reg_video_in_width_initial = *valkyrie_reg_video_in_width;
    valkyrie_reg_video_in_height_initial = *valkyrie_reg_video_in_height;
    valkyrie_reg_video_in_field_x0_initial = *valkyrie_reg_video_in_field_x0;
    valkyrie_reg_video_in_field_y0_initial = *valkyrie_reg_video_in_field_y0;
    valkyrie_screen_video_key = pixel_video_key;
    valkyrie_backbuffer_value_initial_xy00 = ((uint16_t *)valkyrie_buffer_video0)[0];
    valkyrie_clear(0, valkyrie_screen_video_key);
    *valkyrie_reg_subsystem_config      = 0; /* video buffers in rgb instead of yuv */
    *valkyrie_reg_video_in_control      = 0;
    *valkyrie_reg_video_in_field_x0     = 0;
    *valkyrie_reg_video_in_field_y0     = 0;
    *valkyrie_reg_video_in_x0           = 0;
    *valkyrie_reg_video_in_y0           = 0;
    *valkyrie_reg_video_in_width        = 640;
    *valkyrie_reg_video_in_height       = 480;
    /* valkyrie will draw video where main screen pixels has this set */
    *valkyrie_reg_clut_color_key = valkyrie_screen_video_key;
}

void *valkyrie_get_video_backbuffer(void) {
    return (valkyrie_page ? valkyrie_buffer_video0 : valkyrie_buffer_video1);
}

void valkyrie_pageflip_video(void) {
    *valkyrie_reg_video_in_control = (valkyrie_page ? VALKYRIE_VIDEO0 : VALKYRIE_VIDEO1) | VALKYRIE_WINDOW | VALKYRIE_2X;
    valkyrie_page = !valkyrie_page;
}

void valkyrie_stop(void) {
    valkyrie_clear(valkyrie_backbuffer_value_initial_xy00, valkyrie_screen_video_key_initial);
    *valkyrie_reg_subsystem_config = valkyrie_reg_subsystem_config_initial;
    *valkyrie_reg_video_in_control = valkyrie_reg_video_in_control_initial;
    *valkyrie_reg_video_in_x0 = valkyrie_reg_video_in_x0_initial;
    *valkyrie_reg_video_in_y0 = valkyrie_reg_video_in_y0_initial;
    *valkyrie_reg_video_in_width = valkyrie_reg_video_in_width_initial;
    *valkyrie_reg_video_in_height = valkyrie_reg_video_in_height_initial;
    *valkyrie_reg_video_in_field_x0 = valkyrie_reg_video_in_field_x0_initial;
    *valkyrie_reg_video_in_field_y0 = valkyrie_reg_video_in_field_y0_initial;
}

