﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.ComponentModel;

namespace BufferedGraphicsContextExample
{
    public partial class KnobControl : UserControl
    {
        private Image m_StripImage;
        private int m_ImagesInStrip;
        private Point m_MousePrevPos;
        private double m_MinValue;
        private double m_MaxValue = 100.0;
        private double m_Value;

        // Double buffering related
        private bool m_InitializationComplete;
        private bool m_isDisposing;
        private BufferedGraphicsContext m_backbufferContext;
        private BufferedGraphics m_backbufferGraphics;
        private Graphics m_drawingGraphics;

        public KnobControl()
        {
            InitializeComponent();

            // Set the control style to double buffer.
            this.SetStyle(ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint, true);
            this.SetStyle(ControlStyles.SupportsTransparentBackColor, false);
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);

            // Assign our buffer context.
            m_backbufferContext = BufferedGraphicsManager.Current;  
            m_InitializationComplete = true;

            RecreateBuffers();
        }

        public void DrawBackbuffer()
        {
            if (m_drawingGraphics == null)
                return;

            //m_drawingGraphics.Clear(Color.Black);

            PaintParentBackground(new PaintEventArgs(m_drawingGraphics, new Rectangle(0, 0, this.Width, this.Height)));

            if (IsValidStrip())
            {
                int imageIndex = (int)Math.Round((m_Value - m_MinValue) * (m_ImagesInStrip - 1) / (m_MaxValue - m_MinValue));
                int ImageStep = m_StripImage.Width / m_ImagesInStrip;
                m_drawingGraphics.DrawImage(m_StripImage, ClientRectangle, new Rectangle(imageIndex * ImageStep, 0, ImageStep, m_StripImage.Height), GraphicsUnit.Pixel);
            }
        }

        public EventHandler ValueChanged;

        [Bindable(true)]
        [Localizable(true)]
        public Image StripImage 
        { 
            get 
            {
                return m_StripImage; 
            } 

            set 
            {
                m_StripImage = value;
                ResizeAccordingtoStrip();
            }
        }

        public int ImagesInStrip
        {
            get 
            { 
                return m_ImagesInStrip; 
            }
            set
            {
                m_ImagesInStrip = value;
                ResizeAccordingtoStrip();
            }
        }

        public int ValueInt
        {
            get { return (int)m_Value; }
            set { m_Value = value; }
        }

        public double Value
        {
            get { return m_Value; }
            set 
            { 
                m_Value = value;
                Refresh();
            }
        }

        public double MinValue
        {
            get { return m_MinValue; }
            set { m_MinValue = value; }
        }

        public double MaxValue
        {
            get { return m_MaxValue; }
            set { m_MaxValue = value; }
        }

        private bool IsValidStrip()
        {
            if (m_ImagesInStrip == 0)
                return false;

            if (m_StripImage == null)
                return false;

            if (m_StripImage.Width == 0)
                return false;

            if (m_MaxValue <= m_MinValue)
                return false;

            return true;
        }

        private void ResizeAccordingtoStrip()
        {
            if (!IsValidStrip())
                return;

            SetBounds(Left, Top, m_StripImage.Height, m_StripImage.Width / m_ImagesInStrip);
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            // If we've initialized the backbuffer properly, render it on the control. 
            // Otherwise, do just the standard control paint.
            if (!m_isDisposing && m_backbufferGraphics != null)
            {
                DrawBackbuffer();
                m_backbufferGraphics.Render(e.Graphics);
            }
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);
            RecreateBuffers();
            Refresh();
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            m_MousePrevPos = new Point(e.X, e.Y);

            base.OnMouseDown(e);
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (IsValidStrip())
            {
                if ((e.Button & MouseButtons.Left) != 0)
                {
                    double delta = -1*(e.Y - m_MousePrevPos.Y) * m_ImagesInStrip / (m_MaxValue - m_MinValue) * 0.5;
                    m_Value += delta;
                    m_Value = Math.Max(m_Value, m_MinValue);
                    m_Value = Math.Min(m_Value, m_MaxValue);
                    if (ValueChanged != null)
                        ValueChanged(this, new EventArgs());

                    m_MousePrevPos = new Point(e.X, e.Y);
                    Refresh();

                }
            }
            
            base.OnMouseMove(e);
        }

        protected override void Dispose(bool disposing)
        {
            m_isDisposing = true;
            if (disposing)
            {
                m_drawingGraphics = null;
                
                if(components != null)
                    components.Dispose();

                // We must dispose of backbufferGraphics before we dispose of backbufferContext or we will get an exception.
                if (m_backbufferGraphics != null)
                    m_backbufferGraphics.Dispose();
                if (m_backbufferContext != null)
                    m_backbufferContext.Dispose();
            }
            base.Dispose(disposing);
        }

        private void OnMouseWheel(object sender, MouseEventArgs e)
        {
            int delta = e.Delta / 120;
        }

        private void PaintParentBackground(PaintEventArgs e)
        {
            if (Parent != null)
            {
                Rectangle rect = new Rectangle(Left, Top, Width, Height);

                e.Graphics.TranslateTransform(-rect.X, -rect.Y);

                try
                {
                    using (PaintEventArgs pea = new PaintEventArgs(e.Graphics, rect))
                    {
                        pea.Graphics.SetClip(rect);
                        InvokePaintBackground(Parent, pea);
                        InvokePaint(Parent, pea);
                    }
                }
                finally
                {
                    e.Graphics.TranslateTransform(rect.X, rect.Y);
                }
            }
            else
            {
                e.Graphics.FillRectangle(SystemBrushes.Control, ClientRectangle);
            }
        }

        private void RecreateBuffers()
        {
            // Check initialization has completed so we know backbufferContext has been assigned.
            // Check that we aren't disposing or this could be invalid.
            if (!m_InitializationComplete || m_isDisposing)
                return;

            // We recreate the buffer with a width and height of the control. The "+ 1" 
            // guarantees we never have a buffer with a width or height of 0. 
            m_backbufferContext.MaximumBuffer = new Size(this.Width + 1, this.Height + 1);

            // Dispose of old backbufferGraphics (if one has been created already)
            if (m_backbufferGraphics != null)
                m_backbufferGraphics.Dispose();


            // Create new backbufferGrpahics that matches the current size of buffer.
            var rect = new Rectangle(0, 0, Math.Max(this.Width, 1), Math.Max(this.Height, 1));
            m_backbufferGraphics = m_backbufferContext.Allocate(this.CreateGraphics(), rect);

            // Assign the Graphics object on backbufferGraphics to "drawingGraphics" for easy reference elsewhere.
            m_drawingGraphics = m_backbufferGraphics.Graphics;
            m_drawingGraphics.FillRectangle(new SolidBrush(Color.FromArgb(0)), rect);

            // This is a good place to assign drawingGraphics.SmoothingMode if you want a better anti-aliasing technique.
           
            // Invalidate the control so a repaint gets called somewhere down the line.
            this.Invalidate();
        }
    }
}
