﻿using System;
using System.Drawing;
using System.Windows.Forms;

namespace BufferedGraphicsContextExample
{
    public partial class EnvelopeEditor : UserControl
    {
        public class GraphPoint
        {
            public Color Color;
            public PointF Pos;
            public int Size;
            public int Shape;
            public bool IsMoveable;
            public float DragLimit_VMin;
            public float DragLimit_VMax;
            public float DragLimit_HMin;
            public float DragLimit_HMax;
        }

        public GraphPoint[] Points { get; set; }

        // Graph related
        private double m_ViewCenterX; // Time in center of view
        private double m_ViewCenterY; // center of view
        private double m_ViewWidth;   // Sec-to-Pixel ratio
        private double m_ViewHeight;  // Unit-to-Pixel ratio

        // Dragging
        private bool   m_Dragging;             // is dragging active?
        private PointF m_DragStartPos;         // used to store mousedown position in phy coords
        private int    m_DragPointIndex;       // point index that being dragged (-1 if no point is dragged)
        private PointF m_DragPointStartPos;

        // Double buffering related
        private bool m_InitializationComplete;
        private bool m_isDisposing;
        private BufferedGraphicsContext m_backbufferContext;
        private BufferedGraphics m_backbufferGraphics;
        private Graphics m_drawingGraphics;

        public PaintEventHandler OnCustomPaint;


        public double StartX
        {
            get { return V2P_TransX(0); }
        }

        public double EndX
        {
            get { return V2P_TransX(ClientSize.Width); }
        }

        public EnvelopeEditor()
        {
            InitializeComponent();

            // Set the control style to double buffer.
            this.SetStyle(ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint, true);
            this.SetStyle(ControlStyles.SupportsTransparentBackColor, false);
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);

            // Assign our buffer context.
            m_backbufferContext = BufferedGraphicsManager.Current;  
            m_InitializationComplete = true;

            m_ViewWidth = 2;
            m_ViewHeight = 1;
            m_ViewCenterX = m_ViewWidth / 2;
            m_ViewCenterY = m_ViewHeight / 2;

            var pnt = new GraphPoint();
            pnt.Color = Color.Red;
            pnt.Pos = new PointF(0.5f, 0.5f);
            pnt.Size = 2;
            Points = new GraphPoint[] { pnt };

            RecreateBuffers();

            Redraw();

            MouseWheel += OnMouseWheel;
            
            // Register to mouse wheel events
            //LowLevelMouseHook.DefaultInstance().MouseWheel += new MouseEventHandler(OnMouseWheel);
        }

        private void ConstrainZoom()
        {
            // constrain over zoom (less than 20ms)
            if (m_ViewWidth < 20.0 / 1000.0)
                m_ViewWidth = 20.0 / 1000.0;

            // constrain over zoom (over than 120s)
            if (m_ViewWidth > 120.0)
                m_ViewWidth = 120.0;
        }

        private void ConstrainPanning()
        {
            // constrain negative values
            if (V2P_TransX(0) < 0)
                m_ViewCenterX -= V2P_TransX(0);
        }

        private void OnMouseWheel(object sender, MouseEventArgs e)
        {
            // Get point in client space
            var pnt = /*PointToClient(*/new Point(e.X, e.Y);//);
            if (!ClientRectangle.Contains(pnt))
                return;

            // Store position of the mouse in physical coord
            double MousePrevPhyX = V2P_TransX(pnt.X);

            // Scale view
            int delta = e.Delta / 120;
            while (delta-- > 0) m_ViewWidth *= 1.07;
            while (++delta < 0) m_ViewWidth /= 1.07;

            // Enforce zoom constrains
            ConstrainZoom();


            // Corrent the center position (point under the mouse does not mode)
            double MouseNewPhyX = V2P_TransX(pnt.X);
            m_ViewCenterX += MousePrevPhyX - MouseNewPhyX;

            // Enforce panning constrains
            ConstrainPanning();

            Redraw();
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);
            RecreateBuffers();
            Redraw();
        }

        /// <summary> 
        /// Clean up any resources being used.
        /// </summary>
        /// <param name="disposing">true if managed resources should be disposed; otherwise, false.</param>
        protected override void Dispose(bool disposing)
        {
            m_isDisposing = true;
            if (disposing)
            {
                if(components != null)
                    components.Dispose();

                // We must dispose of backbufferGraphics before we dispose of backbufferContext or we will get an exception.
                if (m_backbufferGraphics != null)
                    m_backbufferGraphics.Dispose();
                if (m_backbufferContext != null)
                    m_backbufferContext.Dispose();
            }
            base.Dispose(disposing);
        } 

        private void RecreateBuffers()
        {
            // Check initialization has completed so we know backbufferContext has been assigned.
            // Check that we aren't disposing or this could be invalid.
            if (!m_InitializationComplete || m_isDisposing)
                return;

            // We recreate the buffer with a width and height of the control. The "+ 1" 
            // guarantees we never have a buffer with a width or height of 0. 
            m_backbufferContext.MaximumBuffer = new Size(this.Width + 1, this.Height + 1);

            // Dispose of old backbufferGraphics (if one has been created already)
            if (m_backbufferGraphics != null)
                m_backbufferGraphics.Dispose();

            // Create new backbufferGrpahics that matches the current size of buffer.
            m_backbufferGraphics = m_backbufferContext.Allocate(this.CreateGraphics(),
                new Rectangle(0, 0, Math.Max(this.Width, 1), Math.Max(this.Height, 1)));

            // Assign the Graphics object on backbufferGraphics to "drawingGraphics" for easy reference elsewhere.
            m_drawingGraphics = m_backbufferGraphics.Graphics;

            // This is a good place to assign drawingGraphics.SmoothingMode if you want a better anti-aliasing technique.
           
            // Invalidate the control so a repaint gets called somewhere down the line.
            this.Invalidate();
        }

        public double P2V_ScaleX(double value)
        {
            return value * ClientSize.Width / m_ViewWidth;
        }

        public double P2V_TransX(double value)
        {
            return (value - m_ViewCenterX) * ClientSize.Width / m_ViewWidth + ClientSize.Width / 2;
        }

        public double V2P_ScaleX(double value)
        {
            return value * m_ViewWidth / ClientSize.Width;
        }

        public double V2P_TransX(double value)
        {
            return (value - ClientSize.Width / 2) * m_ViewWidth / ClientSize.Width + m_ViewCenterX;
        }

        public double P2V_ScaleY(double value)
        {
            return (m_ViewHeight - value) * ClientSize.Height / m_ViewHeight;
        }

        public double P2V_TransY(double value)
        {
            return (m_ViewHeight - value - m_ViewCenterY) * ClientSize.Height / m_ViewHeight + ClientSize.Height / 2;
        }

        public double V2P_ScaleY(double value)
        {
            return (ClientSize.Height - value) * m_ViewHeight / ClientSize.Height;
        }

        public double V2P_TransY(double value)
        {
            return (ClientSize.Height / 2 - value) * m_ViewHeight / ClientSize.Height + m_ViewCenterY;
        }

        private Color ColorMul(Color c1, Color c2)
        {
            return Color.FromArgb((int)c1.A * c2.A / 255, (int)c1.R * c2.R / 255, (int)c1.G * c2.G / 255, (int)c1.B * c2.B / 255);
        }

        private void DrawGrid()
        {
            var GridColor = Color.FromArgb(0xFF, 0x50, 0x50, 0x50);

            // Set format of string.
            StringFormat drawFormat = new StringFormat();
            drawFormat.Alignment = StringAlignment.Center;
            drawFormat.LineAlignment = StringAlignment.Center;

            m_drawingGraphics.Clear(Color.Black);

            // Find grid resultion
            string LabelFormatting = "";
            double Major = 10; // seconds
            double Minor = 10; // seconds
            int MajorMinorRatio = 0;
            int RequiredWidth   = 0;
            while (P2V_ScaleX(Minor) > RequiredWidth)
            {
                // Compute Ticks resolutons
                MajorMinorRatio = (MajorMinorRatio == 2) ? 5 : 2;
                Major = Minor;
                Minor /= MajorMinorRatio;

                // Count digits left to point
                int DigitCounter = 1;
                int iTempValue = (int)V2P_TransX(ClientSize.Width);
                while (iTempValue != 0)
                {
                    iTempValue /= 10;
                    DigitCounter++;
                }

                // Count digits right to point
                LabelFormatting = "0.";
                double fTempValue = Math.IEEERemainder(Minor, 1.0);
                while (fTempValue != 0)
                {
                    fTempValue = Math.IEEERemainder(fTempValue * 10, 1.0);
                    LabelFormatting += "0";
                    DigitCounter++;
                }

                // Measure label
                RequiredWidth = (int)(m_drawingGraphics.MeasureString(new String('0', DigitCounter), Font).Width * 1.2);
            }

            // Compute general drawing points
            int SYMax = ClientSize.Height;
            int SYMid = SYMax / 2;
            double SXRight = ClientSize.Width;

            // Setup pen for Minor Grid
            int MinorGridFactor;
            double MinorSpace = P2V_ScaleX(Minor);
            double MajorSpace = P2V_ScaleX(Major);
            MinorGridFactor = (int)(255 * (MajorSpace - RequiredWidth) / (MajorSpace - MinorSpace));
            MinorGridFactor = Math.Max(0, Math.Min(255, MinorGridFactor));
            var minorPen = new Pen(ColorMul(Color.FromArgb(0xFF, MinorGridFactor, MinorGridFactor, MinorGridFactor), GridColor));
            var majorPen = new Pen(GridColor);

            // Setup brush for Minor font
            int MinorTextFactor = Math.Max(0, Math.Min(255, (MinorGridFactor - 192) * 4));
            var minorBrush = new SolidBrush(ColorMul(Color.FromArgb(0xFF, MinorTextFactor, MinorTextFactor, MinorTextFactor), GridColor));
            var majorBrush = new SolidBrush(GridColor);

            // Compute Minor vertical spacing factor
            int MinorVSpaceFactor =  15 * Math.Max(0, Math.Min(255, (MinorGridFactor - 192) * 4)) / 255;

            // Draw grid
            double VXLeft = Math.Floor(V2P_TransX(0) / Major) * Major;
            double SXLeft = P2V_TransX(VXLeft);
            for (int i = (int)((SXRight - SXLeft) / MinorSpace) + 1; i >= 0; i--)
            {
                // Select brushes and pens
                bool bMajor = (i % MajorMinorRatio == 0);
                var TextBrush = bMajor ? majorBrush : minorBrush;
                var LinePen   = bMajor ? majorPen   : minorPen;

                // Compute vertical endings
                int SYMid1 = SYMid - (bMajor ? 15 : MinorVSpaceFactor);
                int SYMid2 = SYMid + (bMajor ? 15 : MinorVSpaceFactor);

                // Compute horivontal Phy and view positions
                double SXPos = SXLeft + MinorSpace * i;
                double VXPos = VXLeft + Minor      * i;

                // Draw Lines
                m_drawingGraphics.DrawLine(LinePen, (int)SXPos, 0, (int)SXPos, SYMid1);
                m_drawingGraphics.DrawLine(LinePen, (int)SXPos, SYMid2, (int)SXPos, SYMax);

                // Draw Text
                if (bMajor || (MinorTextFactor > 0))
                    m_drawingGraphics.DrawString(VXPos.ToString(LabelFormatting), Font, TextBrush, new RectangleF((float)(SXPos - MajorSpace), SYMid - 15, (float)MajorSpace * 2, 30), drawFormat);
            }
        }

        private void DrawPoints()
        {
            // Exit if no points 
            if (Points == null)
                return;

            foreach (var pnt in Points)
            {
                // Convert point into view coordinate system
                var view_pos = new Point((int)P2V_TransX(pnt.Pos.X), (int)P2V_TransY(pnt.Pos.Y));

                // Skip out-of-view poitns
                if (!ClientRectangle.Contains(view_pos))
                    continue;

                // Draw point
                m_drawingGraphics.DrawRectangle(new Pen(pnt.Color), view_pos.X - pnt.Size+1, view_pos.Y - pnt.Size+1, pnt.Size * 2 + 1, pnt.Size * 2 + 1);
            }

        }

        public void Redraw()
        {
            if (m_drawingGraphics == null)
                return;

            DrawGrid();

            DrawPoints();

            // Custom draw step
            if (OnCustomPaint != null)
                OnCustomPaint(this, new PaintEventArgs(m_drawingGraphics, Rectangle.Empty));

            // Force the control to both invalidate and update. 
            this.Refresh();
        }

        protected override void OnPaint(PaintEventArgs e)
        {           
            // If we've initialized the backbuffer properly, render it on the control. 
            // Otherwise, do just the standard control paint.
            if (!m_isDisposing && m_backbufferGraphics != null)
                m_backbufferGraphics.Render(e.Graphics);  
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            // Get mouse coords
            var mouse = new Point(e.X, e.Y);

            // Setup Drag
            m_Dragging = true;
            m_DragStartPos = new PointF((float)V2P_TransX(e.X), (float)V2P_TransY(e.Y));

            // Search points for hit
            m_DragPointIndex = -1;
            if (Points != null)
            {
                for (int i = 0; i < Points.Length; i++) 
                {
                    // Convert point into view coordinate system
                    var pnt = Points[i];
                    var view_pos = new Point((int)P2V_TransX(pnt.Pos.X), (int)P2V_TransY(pnt.Pos.Y));

                    // check if mouse is in point
                    var pointRect = new Rectangle(view_pos.X - pnt.Size, view_pos.Y - pnt.Size, pnt.Size * 2 + 2, pnt.Size * 2 + 2);
                    if (pointRect.Contains(mouse))
                    {
                        m_DragPointIndex = i;
                        m_DragPointStartPos = m_DragStartPos;
                        Redraw();
                        break;
                    }
                }
            }

        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);

            m_Dragging = false;
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            // Get mouse coords
            var mouse = new Point(e.X, e.Y);

            if (m_Dragging)
            {
                if (m_DragPointIndex == -1)
                {
                    // Normal dragging - Correct offset by panning shift
                    m_ViewCenterX -= V2P_TransX(e.X) - m_DragStartPos.X;

                    // Enforce panning constrains
                    ConstrainPanning();
                }
                else
                {
                    // Shift Point
                    Points[m_DragPointIndex].Pos = new PointF((float)(m_DragStartPos.X + V2P_TransX(e.X) - m_DragStartPos.X),
                                                              (float)(m_DragStartPos.Y + V2P_TransY(e.Y) - m_DragStartPos.Y));

                }
            }
            Redraw();
        }
    }
}
