﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.Serialization;

namespace TinyFM8
{
    [DataContract]
    public class PatchData
    {
        [DataMember]
        public string DisplayName;
        [DataMember]
        public string AuthorName;
        [DataMember]
        public string Comment;

        /// <summary>
        /// Defines how big is the "voice bank" for that instrument.
        /// A large voice bank can utilized to: 
        ///   1. Generate a deeper/thicker sound when used by unison voices  (ie, Master_Unison_Voices > 1)
        ///   2. Allow old note to property decay when a new note is pressed (ie, Master_Polyphony_Voices / Master_Unison_Voices > 1)
        /// </summary>
        [DataMember]
        public byte Master_Polyphony_Voices;

        /// <summary>
        /// Define how many voices will be played each time a note is started
        /// </summary>
        [DataMember]
        public byte Master_Unison_Voices;

        /// <summary>
        /// Define how diffrent is the pitch scaling for each of the played voices
        ///   0   => Pitch scaling does not change across voices
        ///   100 => Pitch scaling of each voice incresase by 6%
        /// </summary>
        [DataMember]
        public byte Master_Unison_Detune;

        /// <summary>
        /// Define how paning is diffrent between each played voice 
        ///   0   => Voice panning is not effected
        ///   100 => Each voice is played entierly on left or right side
        /// </summary>
        [DataMember]
        public byte Master_Unison_Pan;

        /// <summary>
        /// Operator X, Noise generator cut off frequancy (Low Pass cutoff)
        ///   0   => Cutoff at 0Hz
        ///   100 => Cutoff at 12Khz
        ///   Note: Cutoff frequancy is not linear and computed by power(10, CutOff / 24.51)
        /// </summary>
        [DataMember]
        public byte OPX_NoiseCutOff;

        /// <summary>
        /// Operator X, Noise generator resonance amount
        ///   0   => resonance is turned off
        ///   100 => High amount of resonance
        /// </summary>
        [DataMember]
        public byte OPX_NoiseReso;

        /// <summary>
        /// Operator X, Noise generator gain
        ///   0   => No noise
        ///   100 => Lots of noise
        /// </summary>
        [DataMember]
        public byte OPX_NoiseAmp;

        /// <summary>
        /// Operator X, Limiter: pre-limiter gain
        ///   0   => Signal is reduced to 20%
        ///   100 => Signal is amplified by 325%
        /// </summary>
        [DataMember]
        public byte OPX_SatGain;

        /// <summary>
        /// Operator X, Saturator/limiter limit value
        ///   0   => Signal is trimmed low to it's zero position
        ///   100 => Signal is not trimmed
        /// </summary>
        [DataMember]
        public byte OPX_SatLimit;

        /// <summary>
        /// Operator Z, Multimode filter - 1st filter cutoff frequancy 
        ///   0   => 0Hz
        ///   100 => 13Mhz
        /// </summary>
        [DataMember]
        public byte OPZ_CutOff;

        /// <summary>
        /// Operator Z, Multimode filter - 2nd filter cutoff frequancy (relative to 1st filter frequancy)
        ///   0   => 0Hz offset
        ///   100 => 12.5Khz Offset
        /// </summary>
        [DataMember]
        public byte OPZ_Spread;

        /// <summary>
        /// Operator Z, Multimode filter - 1nd filter resonance gain
        ///   0   => No resonance
        ///   100 => Lots of resonance
        /// </summary>
        [DataMember]
        public byte OPZ_Reso1;

        /// <summary>
        /// Operator Z, Multimode filter - 2nd filter resonance gain
        ///   0   => No resonance
        ///   100 => Lots of resonance
        /// </summary>
        [DataMember]
        public byte OPZ_Reso2;

        /// <summary>
        /// Operator Z, Multimode filter - 1st Filter mode
        ///   0   => Low Pass
        ///   50  => Band Pass
        ///   100 => High Pass
        /// </summary>
        [DataMember]
        public byte OPZ_Mode1;

        /// <summary>
        /// Operator Z, Multimode filter - 1st Filter mode
        ///   0   => Low Pass
        ///   50  => Band Pass
        ///   100 => High Pass
        /// </summary>
        [DataMember]
        public byte OPZ_Mode2;

        /// <summary>
        /// Operator Z, Multimode filter - filters combination
        ///   0   => two filters are parellel to each other (2 x 6db filters)
        ///   100 => two filters are serial to each other (12db filter)
        /// </summary>
        [DataMember]
        public byte OPZ_Routing;

        /// <summary>
        /// Operator Z, Multimode filter - filters mixing
        ///   0   => Output is purly the 1st filter only
        ///   100 => Output is purly the filter combination
        /// </summary>
        [DataMember]
        public byte OPZ_Mix;

        /// <summary>
        /// Defines a global picth envlope scaling
        ///   0   => Pitch envelope is ignored
        ///  100  => Pitch envelope is 100% active
        /// </summary>
        [DataMember]
        public byte Pitch_Envelope;

        /// <summary>
        /// Defines a constant offset to the played note
        ///  +24 => 2 Octaves higher
        ///  -24 => 2 Octaves lower
        /// </summary>
        [DataMember]
        public sbyte Pitch_Transp;

        /// <summary>
        /// Defines the main FM modulation matrix connection
        /// Size should be 8x8
        /// </summary>
        [DataMember]
        public byte[][] FM_Matrix = new byte[8][];

        /// <summary>
        /// Define if operator is enabled
        /// Size should be 8 (one-per-oscillator + 1 z-op + 1 x-op)
        /// </summary>
        [DataMember]
        public byte[] OP_Active = new byte[8];

        /// <summary>
        /// What kind of waveform is used by each operator
        /// Size should be 6 (one-per-oscillator)
        /// </summary>
        [DataMember]
        public byte[] OP_Waveform = new byte[6];

        /// <summary>
        /// Defines what is the effect of note velocity has on oscillator output power
        ///   0   => No effect, the operator does not react to velocity
        ///  +100 => Operator is amplified when a high velocity note is played
        ///  -100 => Operator is attenuated when a high velocity note is played
        /// Size should be 8 (one-per-oscillator + 1 z-op + 1 x-op)
        /// </summary>
        [DataMember]
        public sbyte[] OP_Velocity = new sbyte[8];

        /// <summary>
        /// Defines if oscillator gets resets (zeroed) when ever a new note is played
        /// Size should be 6 (one-per-oscillator)
        /// </summary>
        [DataMember]
        public byte[] OP_KeySync = new byte[6];

        /// <summary>
        /// Defines the output volume for each operator
        ///   0   => operator output is not included in output channel
        ///   100 => operator output is fully included in output channel
        /// Size should 8 (one-per-oscillator + 1 z-op + 1 x-op)
        /// </summary>
        [DataMember]
        public byte[] OP_OutVol = new byte[8];

        /// <summary>
        /// Defines the output panning for each operator
        ///   -100 => All to the left
        ///      0 => Both
        ///   +100 => All to the right
        /// Size should 8 (one-per-oscillator + 1 z-op + 1 x-op)
        /// </summary>
        [DataMember]
        public sbyte[] OP_OutPan = new sbyte[8];

        /// <summary>
        /// Defines how envelope time gets scaled according to note 
        ///   0   => No scaling, envelope time is 1:1 
        ///   100 => envelope playing time increases as note pitch is higher
        /// Size should be 9 (one-per-oscillator + 1 z-op + 1 x-op + 1 pitch-env)  
        /// </summary>
        [DataMember]
        public byte[] OP_KeyScale = new byte[9];

        /// <summary>
        /// Define sustain loop start poisition inside the operator envelope.
        /// Value is an envelope point index. -1==No sustain loop
        /// Size should be 9 (one-per-oscillator + 1 z-op + 1 x-op + 1 pitch-env)  
        /// </summary>
        [DataMember]
        public sbyte[] OP_Env_SustainLoopStart = new sbyte[9];

        /// <summary>
        /// Define sustain loop end poisition inside the operator envelope.
        /// Value is an envelope point index. -1==No sustain loop
        /// Size should be 9 (one-per-oscillator + 1 z-op + 1 x-op + 1 pitch-env)  
        /// </summary>
        [DataMember]
        public sbyte[] OP_Env_SustainLoopEnd = new sbyte[9];

        /// <summary>
        /// Operator envelopes, Time deltas between each key (in seconds)
        /// Size should be 9 (one-per-oscillator + 1 z-op + 1 x-op + 1 pitch-env)  
        /// </summary>
        [DataMember]
        public float[][] OP_Env_RelTime = new float[9][];

        /// <summary>
        /// Operator envelopes, value of each key
        /// Range is 0..1
        /// Size should be 9 (one-per-oscillator + 1 z-op + 1 x-op + 1 pitch-env)  
        /// </summary>
        [DataMember]
        public float[][] OP_Env_Level = new float[9][];

        /// <summary>
        /// Operator envelopes, slope of each key
        ///   0.0 => Early changing envelope
        ///   0.5 => Linear changing envelope
        ///   1.0 => Late changing envelope
        /// Size should be 9 (one-per-oscillator + 1 z-op + 1 x-op + 1 pitch-env)  
        /// </summary>
        [DataMember]
        public float[][] OP_Env_Slope = new float[9][];

        /// <summary>
        /// Key scaling graph, Note index for each key 
        /// Size should be 9 (one-per-oscillator + 1 z-op + 1 x-op )  
        /// </summary>
        [DataMember]
        public float[][] OP_KeySc_Note = new float[8][];

        /// <summary>
        /// Key scaling graph, operators output scaling value
        /// Range is 0..1
        /// Size should be 8 (one-per-oscillator + 1 z-op + 1 x-op)  
        /// </summary>
        [DataMember]
        public float[][] OP_KeySc_Level = new float[8][];

        /// <summary>
        /// Key scaling graph, slope of each key
        ///   0.0 => Early changing envelope
        ///   0.5 => Linear changing envelope
        ///   1.0 => Late changing envelope
        /// </summary>
        [DataMember]
        public float[][] OP_KeySc_Slope = new float[8][];

        /// <summary>
        /// operator frequnacy scaling
        /// For instance, playing "A-4" note, oscillator's frequancy is calculated by: FreqOSC = 440 * OP_Ratio + OP_Offset
        /// Size should be 6 (one-per-oscillator)  
        /// </summary>
        [DataMember]
        public float[] OP_Ratio = new float[6];

        /// <summary>
        /// operator frequnacy offset 
        /// For instance, playing "A-4" note, oscillator's frequancy is calculated by: FreqOSC = 440 * OP_Ratio + OP_Offset
        /// Size should be 6 (one-per-oscillator)  
        /// </summary>
        [DataMember]
        public float[] OP_Offset = new float[6];
    }

    [DataContract]
    public class ChannelData 
    {
        [DataMember]
        public int       TrackIndex;

        [DataMember]
        public float     Gain;

        [DataMember]
        public PatchData patch; 
    }

}
