﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using KSDManager;
using BufferedGraphicsContextExample;

namespace TinyFM8
{
    public partial class MainUserInterface : UserControl
    {
        public IntPtr Voice;
        public Plugin _plugin;
        public InstrumentLibrary InstLib = new InstrumentLibrary();

        public MainUserInterface()
        {
            InitializeComponent();

            panelOpZ1.Bounds = pnlControlsPage.ClientRectangle;
            panelOpX1.Bounds = pnlControlsPage.ClientRectangle;
            panelOpOsc1.Bounds = pnlControlsPage.ClientRectangle;

            panelOpZ1.UIChanged += OnPanelChangeEvent;
            panelOpX1.UIChanged += OnPanelChangeEvent;
            panelOpOsc1.UIChanged += OnPanelChangeEvent;

            InstLib.LoadLibrary();

            InstLib.Settings.InstrumentList.Select(t => new ListViewItem(new string[] { t.Name }));
            foreach (var item in InstLib.Settings.InstrumentList)
            {
                var Lstitem = new ListViewItem(new string[] { item.Name });
                Lstitem.Group = listView1.Groups[0];
                Lstitem.Tag = item;

                listView1.Items.Add(Lstitem);
            }
            listView1.AutoResizeColumns(ColumnHeaderAutoResizeStyle.HeaderSize);

            fmMatrixControl1.SelectionChanged += OnMatrixSelectionChanged;
            fmMatrixControl1.MatrixChanged += OnMatrixChanged;
            envelopeOscEnv.OnCustomPaint += OnPaint_OscEnv;
            envelopePitch.OnCustomPaint += OnPaint_PitchEnv;
        }

        private void OnPaint_PitchEnv(object sender, PaintEventArgs e)
        {
            if (Voice == IntPtr.Zero)
                return;

            CustomPaintEnv(e.Graphics, 8, envelopePitch);
        }

        private void OnPaint_OscEnv(object sender, PaintEventArgs e)
        {
            if (Voice == IntPtr.Zero)
                return;

            CustomPaintEnv(e.Graphics, fmMatrixControl1.SelectedOP, envelopeOscEnv);
        }

        private void CustomPaintEnv(Graphics g, int OpIndex, EnvelopeEditor envControl)
        {

            var points = envControl.ClientSize.Width;
            var buf = new float[points];
            SoundRenderInterface.RenderEnvelope(Voice, OpIndex, (float)envControl.StartX, (float)envControl.EndX, points, buf);

            var line = new Point[points + 2];
            for (int i = 0; i < points; i++)
            {
                if (float.IsNaN(buf[i]))
                    return;
                line[i + 1] = new Point(i, (int)envControl.P2V_TransY(buf[i]));
            }

            // Add first and last point
            line[0] = new Point(line[0].X, (int)envControl.P2V_TransY(0));
            line[points + 1] = new Point(line[points].X, (int)envControl.P2V_TransY(0));
            
            g.FillPolygon(new SolidBrush(Color.FromArgb(0x50, Color.LightBlue)), line);
            g.DrawLines(Pens.LightBlue, line);

            // start loop marker
            if (_plugin.ChannelData.patch.OP_Env_SustainLoopStart[OpIndex] != -1)
            {
                var SegIdx = _plugin.ChannelData.patch.OP_Env_SustainLoopStart[OpIndex];
                var Level = _plugin.ChannelData.patch.OP_Env_Level[OpIndex][SegIdx];

                // Compute Time
                var Time   = 0.0f;
                for (int i = 0; i <= SegIdx; i++)
                    Time += _plugin.ChannelData.patch.OP_Env_RelTime[OpIndex][i];

                var marker = new Point[2];
                marker[0] = new Point((int)envControl.P2V_TransX(Time), (int)envControl.P2V_TransY(Level));
                marker[1] = new Point((int)envControl.P2V_TransX(Time), (int)envControl.P2V_TransY(0));
                g.DrawLines(Pens.White, marker);
            }

            // End loop marker
            if (_plugin.ChannelData.patch.OP_Env_SustainLoopEnd[OpIndex] != -1)
            {
                var SegIdx = _plugin.ChannelData.patch.OP_Env_SustainLoopEnd[OpIndex];
                var Level  = _plugin.ChannelData.patch.OP_Env_Level[OpIndex][SegIdx];

                // Compute Time
                var Time = 0.0f;
                for (int i = 0; i <= SegIdx; i++)
                    Time += _plugin.ChannelData.patch.OP_Env_RelTime[OpIndex][i];

                var marker = new Point[2];
                marker[0] = new Point((int)envControl.P2V_TransX(Time), (int)envControl.P2V_TransY(Level));
                marker[1] = new Point((int)envControl.P2V_TransX(Time), (int)envControl.P2V_TransY(0));
                g.DrawLines(Pens.White, marker);
            }
        }

        private void OnMatrixChanged(object sender, EventArgs e)
        {
            SaveUI();
        }

        private void OnMatrixSelectionChanged(object sender, EventArgs e)
        {
            LoadUI();
        }

        private void button1_Click(object sender, EventArgs e)
        {
            listView1.AutoResizeColumns(ColumnHeaderAutoResizeStyle.HeaderSize);
        }

        private void btnImportFM8_Click(object sender, EventArgs e)
        {
            InstLib.ParentControl = this;
            InstLib.CheckForFM8(true);
        }

        private void listView1_SelectedIndexChanged(object sender, EventArgs e)
        {
            if ((listView1.SelectedItems != null) && (listView1.SelectedItems.Count != 0))
            {
                var inst = listView1.SelectedItems[0].Tag as InstrumentDetails;
                _plugin.LoadPatch(inst.File.Filename);
                LoadUI();
            }
        }

        private void OnPanelChangeEvent(object sender, EventArgs args)
        {
            SaveUI();
        }

        private void SaveUI()
        {
            // Load Settings
            _plugin.ChannelData.patch.DisplayName               = txtName.Text                  ;
            _plugin.ChannelData.patch.AuthorName                = txtAuthor.Text                ;
            _plugin.ChannelData.patch.Comment                   = txtComment.Text               ;
            _plugin.ChannelData.patch.Master_Polyphony_Voices   = (byte)updwnNrOfVoices.Value   ;
            _plugin.ChannelData.patch.Master_Unison_Voices      = (byte)updwnUnisonVoices.Value ;
            _plugin.ChannelData.patch.Master_Unison_Detune      = (byte)knobUnisonDetune.Value  ;
            _plugin.ChannelData.patch.Master_Unison_Pan         = (byte)knobUnisonPan.Value     ;
            _plugin.ChannelData.patch.Pitch_Transp              = (sbyte)updwnPitchTransp.Value ;
            _plugin.ChannelData.patch.Pitch_Envelope            = (byte)knobPitchScaling.Value  ;

            // Load Matrix
            for (var iCol = 0; iCol < 8; iCol++)
            {
                for (var iRow = 0; iRow < 8; iRow++)
                    _plugin.ChannelData.patch.FM_Matrix[iRow][iCol] = (byte)fmMatrixControl1.Matrix[iCol, iRow];

                _plugin.ChannelData.patch.OP_OutVol[iCol] = (byte)fmMatrixControl1.Matrix[iCol, 8];
                _plugin.ChannelData.patch.OP_OutPan[iCol] = (sbyte)fmMatrixControl1.Matrix[iCol, 9];
                _plugin.ChannelData.patch.OP_Active[iCol] = fmMatrixControl1.OpActive[iCol] ? (byte)1 : (byte)0;
            }

            if (fmMatrixControl1.SelectedOP <= 5)
                panelOpOsc1.SaveToData();

            if (fmMatrixControl1.SelectedOP == 6)
                panelOpX1.SaveToData();

            if (fmMatrixControl1.SelectedOP == 7)
                panelOpZ1.SaveToData();

            // Load Patch
            _plugin.LoadPatch(_plugin.ChannelData.patch);
        }

        private void LoadUI()
        {
            txtName.Text            = _plugin.ChannelData.patch.DisplayName;
            txtAuthor.Text          = _plugin.ChannelData.patch.AuthorName;
            txtComment.Text         = _plugin.ChannelData.patch.Comment;
            updwnNrOfVoices.Value   = _plugin.ChannelData.patch.Master_Polyphony_Voices;
            updwnUnisonVoices.Value = _plugin.ChannelData.patch.Master_Unison_Voices;
            knobUnisonDetune.Value  = _plugin.ChannelData.patch.Master_Unison_Detune;
            knobUnisonPan.Value     = _plugin.ChannelData.patch.Master_Unison_Pan;
            updwnPitchTransp.Value  = _plugin.ChannelData.patch.Pitch_Transp;
            knobPitchScaling.Value  = _plugin.ChannelData.patch.Pitch_Envelope;

            for (var iCol = 0; iCol < 8; iCol++)
            {
                for (var iRow = 0; iRow < 8; iRow++)
                    fmMatrixControl1.Matrix[iCol, iRow] = _plugin.ChannelData.patch.FM_Matrix[iRow][iCol];

                fmMatrixControl1.Matrix[iCol, 8] = _plugin.ChannelData.patch.OP_OutVol[iCol];
                fmMatrixControl1.Matrix[iCol, 9] = _plugin.ChannelData.patch.OP_OutPan[iCol];
                fmMatrixControl1.OpActive[iCol] = _plugin.ChannelData.patch.OP_Active[iCol] != 0;
            }

            fmMatrixControl1.Redraw();

            // Update selected waveform
            if (fmMatrixControl1.SelectedOP <= 5)
            {
                panelOpOsc1.pd = _plugin.ChannelData.patch;
                panelOpOsc1.OpIndex = fmMatrixControl1.SelectedOP;
                panelOpOsc1.LoadFromData();
                panelOpOsc1.Visible = true;
            }
            else
                panelOpOsc1.Visible = false;


            if (fmMatrixControl1.SelectedOP == 6)
            {
                panelOpX1.pd = _plugin.ChannelData.patch;
                panelOpX1.LoadFromData();
                panelOpX1.Visible = true;
            }
            else
                panelOpX1.Visible = false;

            if (fmMatrixControl1.SelectedOP == 7)
            {
                panelOpZ1.pd = _plugin.ChannelData.patch;
                panelOpZ1.LoadFromData();
                panelOpZ1.Visible = true;
            }
            else
                panelOpZ1.Visible = false;

            // get a voice
            Voice = (_plugin.AudioProcessor.OffVoices.Count > 0) ? _plugin.AudioProcessor.OffVoices[0] : _plugin.AudioProcessor.ActiveVoices[0];

            // Make the envelope editor refresh
            envelopeOscEnv.Redraw();
            envelopePitch.Redraw();

            var s = "    Time  Value   Slope\r\n";
            var InsideLoop = false;
            var OpIdx = fmMatrixControl1.SelectedOP;
            for (int iPnt = 0; iPnt < _plugin.ChannelData.patch.OP_Env_Level[OpIdx].Length; iPnt++)
            {
                if (_plugin.ChannelData.patch.OP_Env_SustainLoopStart[OpIdx] == iPnt) InsideLoop = true;

                s += InsideLoop ? "*" : " ";
                s += _plugin.ChannelData.patch.OP_Env_RelTime[OpIdx][iPnt].ToString("0.00").PadLeft(7) +
                     _plugin.ChannelData.patch.OP_Env_Level[OpIdx][iPnt].ToString("0.00").PadLeft(7) +
                     _plugin.ChannelData.patch.OP_Env_Slope[OpIdx][iPnt].ToString("0.000").PadLeft(8) + "\r\n";

                if (_plugin.ChannelData.patch.OP_Env_SustainLoopEnd[OpIdx] == iPnt) InsideLoop = false;
            }
            textOscEnv.Text = s;


            s = "    Time  Value   Slope\r\n";
            InsideLoop = false;
            OpIdx = fmMatrixControl1.SelectedOP;
            for (int iPnt = 0; iPnt < _plugin.ChannelData.patch.OP_Env_Level[8].Length; iPnt++)
            {
                if (_plugin.ChannelData.patch.OP_Env_SustainLoopStart[8] == iPnt) InsideLoop = true;

                s += InsideLoop ? "*" : " ";
                s += _plugin.ChannelData.patch.OP_Env_RelTime[8][iPnt].ToString("0.00").PadLeft(7) +
                     _plugin.ChannelData.patch.OP_Env_Level[8][iPnt].ToString("0.00").PadLeft(7) +
                     _plugin.ChannelData.patch.OP_Env_Slope[8][iPnt].ToString("0.000").PadLeft(8) + "\r\n";

                if (_plugin.ChannelData.patch.OP_Env_SustainLoopEnd[8] == iPnt) InsideLoop = false;
            }
            textPitchEnv.Text = s;
        }
    }
}
