﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Xml;
using System.Runtime.Serialization;
using KSDManager;
using System.Windows.Forms;
using System.Drawing;

namespace TinyFM8
{
    [DataContract]
    public class FileInfo
    {
        [DataMember]
        public string Filename;
        
        [DataMember]
        public DateTime Timestamp;
    }

    [DataContract]
    [KnownType(typeof(FileInfo))]
    public class InstrumentDetails
    {
        [DataMember]
        public FileInfo File;
        
        [DataMember]
        public string Name;
        
        [DataMember]
        public string Authour;

        [DataMember]
        public string Comment;
    }

    [DataContract]
    [KnownType(typeof(InstrumentDetails))]
    public class SettingsStorage
    {
        [DataMember]
        public bool FM8FolderDetected;

        [DataMember]
        public List<InstrumentDetails> InstrumentList = new List<InstrumentDetails>();
    }

    public class InstrumentLibrary
    {
        public string LibraryPath;
        public string FM8ImportedPath;
        public SettingsStorage Settings = new SettingsStorage();
        public Control ParentControl;

        public InstrumentLibrary()
        {
            LibraryPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData), "TinyFM8");
            Directory.CreateDirectory(LibraryPath);

            // Set TinyFM8 FM8 imports folder
            FM8ImportedPath = Path.Combine(LibraryPath, "FM8Import");
            Directory.CreateDirectory(FM8ImportedPath);
        }

        public static void WriteObject(object obj, string filename)
        {
            using (var fs = new FileStream(filename, FileMode.Create))
            {
                using (var writer = XmlDictionaryWriter.CreateTextWriter(fs))
                {
                    DataContractSerializer ser = new DataContractSerializer(obj.GetType());
                    ser.WriteObject(writer, obj);
                }
            }
        }

        public static T ReadObject<T>(string path)
        {
            using (var fs = new FileStream(path, FileMode.OpenOrCreate))
            {
                XmlDictionaryReader reader = XmlDictionaryReader.CreateTextReader(fs, new XmlDictionaryReaderQuotas());

                // Create the DataContractSerializer instance.
                DataContractSerializer ser = new DataContractSerializer(typeof(T));

                // Deserialize the data and read it from the instance.
                return (T)ser.ReadObject(reader);
            }
        }

        public void ScanFolders(string baseFolder, string fileMask, List<string> files)
        {
            // Add current folder files
            files.AddRange(Directory.GetFiles(baseFolder, fileMask));

            // Get sub folders files
            var folders = Directory.GetDirectories(baseFolder);
            foreach (var subFolder in folders)
                ScanFolders(subFolder, fileMask, files);        
        }

        public void LoadLibrary()
        {
            // Load Settings
            try
            {
                Settings = ReadObject<SettingsStorage>(Path.Combine(LibraryPath, "Library.xml"));
            }
            catch
            {
            }

            // Get file list
            var files = new List<string>();
            ScanFolders(LibraryPath, "*.tfm8", files);

            // Build up dictionaries
            var FilesDic = files.ToDictionary(t => t, t => File.GetLastWriteTime(t));
            var InstLDic = Settings.InstrumentList.ToDictionary(t => t.File.Filename, t => t);

            // Scan files for new or changed files
            var ReadList = new List<string>();
            foreach (var file in FilesDic)
            {
                // Check if Library already contains files
                if (InstLDic.ContainsKey(file.Key))
                {
                    // Check if file was changed
                    if (InstLDic[file.Key].File.Timestamp != file.Value)
                        ReadList.Add(file.Key);

                    // Remove from check list
                    InstLDic.Remove(file.Key);
                }
                else
                {
                    // File does not exists in library, add it to the reading list
                    ReadList.Add(file.Key);
                }
            }

            // Read all pending files
            foreach (var Filename in ReadList)
            {
                try
                {
                    // Load file
                    var inst = ReadObject<PatchData>(Filename);

                    // Add To instrument library
                    Settings.InstrumentList.Add(new InstrumentDetails()
                    {
                        File = new FileInfo()
                        {
                            Filename = Filename,
                            Timestamp = File.GetLastWriteTime(Filename)
                        },
                        Name = inst.DisplayName,
                        Authour = inst.AuthorName,
                        Comment = inst.Comment
                    });
                }
                catch (Exception ex)
                {
                    MessageBox.Show(null, "Failed to load the file: " + Filename + "\n[" + ex.Message + "]", "TinyFM8 Load Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }

            // Check if need to remove dead library files
            foreach (var Inst in InstLDic)
                if (!File.Exists(Inst.Key))
                    Settings.InstrumentList.Remove(Inst.Value);

            // Write library
            WriteObject(Settings, Path.Combine(LibraryPath, "Library.xml"));
        }

        public void CheckForFM8(bool bAskUser)
        {
            // Get TinyFM8 imports list 
            Directory.CreateDirectory(FM8ImportedPath);
            var TFM8Files = Directory.GetFiles(FM8ImportedPath, "*.tfm8");

            // Exit if there are imported files
            if (TFM8Files.Length != 0)
                return;

            // Try to import
            var FM8BasePath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.CommonProgramFiles), "Native Instruments");

            // Create ksd and nfm8 file list
            var FM8files = new List<string>();
            ScanFolders(FM8BasePath, "*.nfm8", FM8files);
            ScanFolders(FM8BasePath, "*.ksd", FM8files);

            // Do nothing if there are not too many files
            if (FM8files.Count < 100)
                return;

            // Ask user to import
            if (bAskUser)
                if (MessageBox.Show(ParentControl, "FM8 instruments were found. Do you wish to import them?", "FM8 Importer", MessageBoxButtons.YesNo, MessageBoxIcon.Question) != DialogResult.Yes)
                    return;


            // Open Progress form
            if (ParentControl != null)
                Utils.SetNativeEnabled(ParentControl.Handle, false);
            var Progress = new ProgressDialog();
            Progress.StartPosition = FormStartPosition.Manual;
            Progress.Location = new Point(ParentControl.PointToScreen(ParentControl.Location).X + (ParentControl.Width - Progress.Width) / 2, ParentControl.PointToScreen(ParentControl.Location).Y + (ParentControl.Height - Progress.Height) / 2);
            Progress.Text = "FM8 Imports";
            Progress.Show();

            // Import files
            int i = 0;
            foreach (var file in FM8files)
            {
                Application.DoEvents();
                var inst = ImportFM8(file);
                Progress.Step(i++, FM8files.Count, inst != null ? inst.DisplayName : "[Unknown]");
            }

            // Save library
            WriteObject(Settings, Path.Combine(LibraryPath, "Library.xml"));

            // Close progress dialog
            Progress.Close();
            if (ParentControl != null)
                Utils.SetNativeEnabled(ParentControl.Handle, true);
        }

        public PatchData ImportFM8(string file)
        {
            // Load KSD
            PatchData inst = null;
            try
            {
                // load KSD file
                var fm8_inst = FM8Instrument.Load(file);

                // create instrument data
                inst = fm8_inst.ToPatchData();

                // Compose output file name
                var Filename = Path.Combine(FM8ImportedPath, Utils.SafeFilename(inst.DisplayName) + ".tfm8");

                // Check if already exists
                if (Settings.InstrumentList.FirstOrDefault(t => t.File.Filename == Filename) != null)
                    return inst;

                // Write instrument
                InstrumentLibrary.WriteObject(inst, Filename);

                // Update Cached list
                Settings.InstrumentList.Add(new InstrumentDetails()
                {
                    File = new FileInfo()
                    {
                        Filename = Filename,
                        Timestamp = File.GetLastWriteTime(Filename)
                    }, 
                    Name = inst.DisplayName, 
                    Authour = inst.AuthorName, 
                    Comment = inst.Comment 
                });
            }
            catch
            {
                // Do nothing - skip file
            }

            return inst;
        }
    }
}
