﻿using Jacobi.Vst.Core;
using Jacobi.Vst.Framework;
using Jacobi.Vst.Framework.Plugin;
using System;
using KSDManager;
using System.IO;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Text;

namespace TinyFM8
{
    /// <summary>
    /// Implements the audio processing of the plugin using the <see cref="SampleManager"/>.
    /// </summary>
    public class AudioProcessor : VstPluginAudioProcessorBase
    {
        StreamWriter log = null;

        [DllImport("Kernel32.dll")]
        private static extern bool QueryPerformanceCounter(out long lpPerformanceCount);

        [DllImport("Kernel32.dll")]
        private static extern bool QueryPerformanceFrequency(out long lpFrequency);

        [DllImport("user32.dll", CharSet = CharSet.Auto, ExactSpelling = true, CallingConvention = CallingConvention.Winapi)]
        private static extern short GetKeyState(int keyCode);

        public List<IntPtr> ActiveVoices = new List<IntPtr>();
        public List<IntPtr> OffVoices = new List<IntPtr>();
        public bool VoicesReady = false;
        private Plugin _plugin;
        public int SampleCounter;
        public float Track1;
        public float Track2;
        public float Track3;
        public string PrefStr;
        public float CSPref;
        public float NativePref;


        public bool ScrollLock
        {
            get {return (((ushort)GetKeyState(0x91)) & 0xffff) != 0;}
        }

        /// <summary>
        /// Constructs a new instance.
        /// </summary>
        /// <param name="plugin">Must not be null.</param>
        public AudioProcessor(Plugin plugin)
            : base(2, 2, 0)
        {
            _plugin = plugin;
        }


        public void LoadChannelData(ChannelData inst)
        {
            try
            {
                var memStream1 = new MemoryStream();
                var binWriter1 = new BinaryWriter(memStream1);
                
                var list1 = new List<ChannelData>();
                list1.Add(inst);
                SoundRenderInterface.SerilaizeInstruments(list1, binWriter1);
                binWriter1.Flush();


                // Create Instrument
                var InstArray = new IntPtr[1];
                var buffer = memStream1.GetBuffer();
                SoundRenderInterface.CreateInstruments(buffer, InstArray);

                // Load Instrument
                lock (_plugin.AudioProcessor.ActiveVoices)
                {
                    OffVoices.Clear();
                    ActiveVoices.Clear();
                    OffVoices.Add(InstArray[0]);
                    for (int i = 1; i < inst.patch.Master_Polyphony_Voices; i++)
                        OffVoices.Add(SoundRenderInterface.DuplicateInstrument(InstArray[0]));
                }

                VoicesReady = true;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString());
            }
        }


        public float InEnvLevel;
        public int SilentCounter = 0;
        public int SignalStart = 0;
        public int SignalEnd = 0;
        public float SignalLength = 0;
        public bool GetSample = false;

        float PrevSample;
        int WaveSamples = 0;
        float[] WaveSample = new float[1024 * 1024];

        public void ComputeEnvelopeLevel(int SampleCounter, VstAudioBuffer[] inChannels)
        {
            float IIR_Factor = 0.00001f;
            for (int index = 0; index < inChannels[0].SampleCount; index++)
            {
                // Wave Sampler
                float Sample = inChannels[0][index];
                if ((Sample > 0) && (PrevSample < 0))
                {
                    if (GetSample)
                    {
                        var sb = new StringBuilder(1024 * 1024);
                        sb.AppendLine("Data");
                        for (int i = 0; i < WaveSamples; i++)
                            sb.AppendLine(WaveSample[i].ToString());
                        Clipboard.SetText(sb.ToString());
                    }
                    GetSample = false;
                    WaveSamples = 0;
                }
                PrevSample = Sample;
                WaveSample[WaveSamples++] = Sample;
                if (WaveSamples >= 1024 * 1024) WaveSamples = 0;


                Sample = Math.Abs(Sample);
                if (Sample > InEnvLevel)
                    InEnvLevel = Sample;
                else
                    InEnvLevel = InEnvLevel * (1f-IIR_Factor) + Sample * IIR_Factor;

                // Check Singal length
                if (Sample < 0.01)
                {
                    // In Silence
                    SilentCounter++;
                    SignalEnd = SampleCounter + index;
                }
                else
                {
                    // Some signal found, check if this is a trigger
                    if (SilentCounter > SampleRate / 20)
                        SignalLength = (SignalEnd - SignalStart) / SampleRate;

                    SignalStart = SampleCounter + index;
                    SilentCounter = 0;
                }
            }
        }


        public Single[] EnvelopeBuffer = new float[1024*1024];
        public int EnvelopeBufferSize = 0;
        void CollectEnvelope(int SampleCounter, VstAudioBuffer[] inChannels)
        {
            // Handle Data out
            if (GetSample)
            {
                var sb = new StringBuilder(1024 * 1024);
                sb.AppendLine("Data");
                for (int i = 0; i < EnvelopeBufferSize; i++)
                    sb.AppendLine(EnvelopeBuffer[i].ToString());
                Clipboard.SetText(sb.ToString());
            }
            GetSample = false;

            float IIR_Factor = 0.1f;
            for (int index = 0; index < inChannels[0].SampleCount; index++)
            {
                // Rectify input
                float Sample = Math.Abs(inChannels[0][index]);

                // Update IIR
                if (Sample > InEnvLevel)
                    InEnvLevel = Sample;
                //else
                //    InEnvLevel = InEnvLevel * (1f - IIR_Factor) + Sample * IIR_Factor;

                // Check Singal length
                if (Sample < 0.0001)
                {
                    // In Silence
                    SilentCounter++;
                }
                else
                {
                    // Some signal found, check if this is a trigger
                    if (SilentCounter > SampleRate)
                        EnvelopeBufferSize = 0;

                    SilentCounter = 0;
                }


                // Add sumple to buffer
                if ((((SampleCounter + index) & 15) == 0) && (SilentCounter < SampleRate / 50))
                {
                    if (EnvelopeBufferSize < EnvelopeBuffer.Length)
                        EnvelopeBuffer[EnvelopeBufferSize++] = InEnvLevel;
                    InEnvLevel = 0;
                }
            }
        }
        

        public override void Process(VstAudioBuffer[] inChannels, VstAudioBuffer[] outChannels)
        {
            // Synth render
            if (VoicesReady)
            {
                try
                {
                    // Start performance measure
                    long Freq, s1, s2;
                    QueryPerformanceFrequency(out Freq);
                    QueryPerformanceCounter(out s1);

                    // Render all sample to buffer
                    float[] Buffer2 = new float[outChannels[0].SampleCount * 2];
                    lock (_plugin.AudioProcessor.ActiveVoices)
                    {
                        foreach (var voice in ActiveVoices)
                            SoundRenderInterface.SynthRender(voice, Buffer2, Buffer2.Length / 2);
                        foreach (var voice in OffVoices)
                            SoundRenderInterface.SynthRender(voice, Buffer2, Buffer2.Length / 2);
                    }

                    // Stop performance measure
                    QueryPerformanceCounter(out s2);

                    // Place data in output buffer
                    for (int i = 0; i < Buffer2.Length / 2; i++)
                    {
                        outChannels[0][i] = Buffer2[i * 2 + 0];
                        outChannels[1][i] = Buffer2[i * 2 + 1];
                    }
                    
                    // Report perf data
                    double Factor = SampleRate / outChannels[0].SampleCount;
                    NativePref = NativePref*0.95f + 0.05f * (float)(Factor * 1000.0 * (s2 - s1) / Freq);
                    PrefStr = String.Format("{0}", NativePref);
                }
                catch (Exception ex)
                {
                    System.Diagnostics.Debugger.Break();
                }
            }

            SampleCounter += inChannels[0].SampleCount;

            // Debuging
            // ComputeEnvelopeLevel(SampleCounter, outChannels);
        }
    }
}
