(** Converts programs to intermediate bytecode. *)

(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

include Program
include Imbc

(** Collects an opcode to the imbc program. *)
let collect_opcode imbc opcode =
	let is_label =
		match opcode with Label _ -> true | _ -> false
	in
	if (List.exists (fun o -> opcode_eq opcode o) imbc.imbc_opcodes) || is_label then
		imbc
	else
		{ imbc with imbc_opcodes = opcode :: imbc.imbc_opcodes }

(** Collects a constant to the imbc program. Returns a (imbc,int) -pair (the constant index). *)
let collect_constant imbc const =
	let imbc =
		if List.exists (fun v -> v = const) imbc.imbc_constants then
			imbc
		else
			{ imbc with imbc_constants = imbc.imbc_constants @ [const] }
	in

	let idx = Util.list_idx imbc.imbc_constants const in
	(imbc, idx)

(** Collects a string constant to the imbc program. Returns a (imbc,ivalue) -pair. *)
let collect_string imbc str =
	let imbc =
		if List.exists (fun v -> v = str) imbc.imbc_strings then
			imbc
		else
			{ imbc with imbc_strings = imbc.imbc_strings @ [str] }
	in

	let v = Val_String_Const (Util.list_idx imbc.imbc_strings str) in
	(imbc, v)

(** Collects an external reference to the imbc program. Returns a (imbc,ivalue) -pair. *)
let collect_external imbc ext =
	let imbc =
		if List.exists (fun v -> v = ext) imbc.imbc_externals then
			imbc
		else
			{ imbc with imbc_externals = imbc.imbc_externals @ [ext] }
	in

	let v = Val_External ext in
	(imbc, v)

(** Converts a program to the intermediate bytecode. *)
let convert prg =
	(* Start with an empty program. *)
	let imbc =
		{
			imbc_heapsize = prg.prg_heap_size;
			imbc_opcodes = [];
			imbc_constants = [];
			imbc_strings = [];
			imbc_externals = [];
			imbc_funs = [];
			imbc_preset_globals = [];
			imbc_uninit_globals = [];
		}
	in

	(* Collect all globals. *)
	let collect_global imbc (name,glb) =
		let preset imbc v =
			{ imbc with imbc_preset_globals = (name,v) :: imbc.imbc_preset_globals }
		in

		match glb with
		| Global_Empty r -> { imbc with imbc_uninit_globals = name :: imbc.imbc_uninit_globals }
		| Global_Int (v,r) -> preset imbc (Val_Int v)
		| Global_Float (v,r) -> preset imbc (Val_Float v)
		| Global_Raw_Array (v,r) ->
			let (imbc, sval) = collect_string imbc v in
			preset imbc sval
		| Global_Uninitialized_Array (v,r) -> failwith "No support yet!"
	in
	let imbc = List.fold_left collect_global imbc prg.prg_globals in

	(* Unique id generator. *)
	let cur_unique_id = ref 0 in
	let uid () = incr cur_unique_id; "@" ^ (string_of_int !cur_unique_id) in

	(* Collect all functions. *)
	let collect_fun imbc (name,func) =
		(* Code collector. *)
		let rev_code = ref [] in
		let add_opcode imbc op =
			(* Add to the code. *)
			rev_code := op :: !rev_code;
			(* Add to the opcode list. *)
			collect_opcode imbc op
		in

		(* Collects a constant ivalue. *)
		let add_constant_opcode imbc ival =
			let byte_min = Int32.of_int (-128) in
			let byte_max = Int32.of_int 127 in

			(* Small enough to go without the constant table entry? *)
			match ival with
			| Val_Int v when ((v >= byte_min) && (v <= byte_max)) ->
				add_opcode imbc (PushConstantByte (Int32.to_int v))
			| Val_Float f when (f > (-0.0000001) && f < 0.0000001) ->
				add_opcode imbc (PushConstantByte 0)
			| _ ->
				let (imbc,idx) = collect_constant imbc ival in
				add_opcode imbc (PushConstant idx)
		in

		(* Returns the local/parameter stack index. *)
		let get_local_var_idx name =
			-1 - (Util.list_idx func.func_locals name)
		in
		let get_param_var_idx name =
			5 + (Util.list_idx func.func_params name)
		in

		(* Returns the globals index and a true if it's preset. (int,bool) -pair. *)
		let get_global_idx imbc name =
			if List.exists (fun s -> s = name) imbc.imbc_uninit_globals then
				(Util.list_idx imbc.imbc_uninit_globals name, false)
			else
				(Util.list_find_idx (fun (n,_) -> n = name) imbc.imbc_preset_globals, true)
		in

		(* Loop break points. *)
		let loop_break = ref None in
		let loop_continue = ref None in

		(* Create the code. *)
		let rec gen_code imbc stmt =
			match stmt with
			| Nop r -> add_constant_opcode imbc (Val_Int Int32.zero)
			| Block (stmts,r) ->
				let rec do_block_item imbc stmts =
					match stmts with
					| [] -> add_constant_opcode imbc (Val_Int Int32.zero)
					| hd :: tl ->
						(* Generate code for this item. *)
						let imbc = gen_code imbc hd in
						(* Need to pop a value? (from any item "in between" we do) *)
						match tl with
						| [] -> imbc
						| tl ->
							let imbc = add_opcode imbc Pop in
							do_block_item imbc tl
				in
				do_block_item imbc stmts
			| ConstInt (v,r) -> add_constant_opcode imbc (Val_Int v)
			| ConstFloat (v,r) -> add_constant_opcode imbc (Val_Float v)
			| ConstRawArray (v,r) -> let (imbc,v2) = collect_string imbc v in add_constant_opcode imbc v2
			| GetGlobal (name,r) ->
				let (idx,is_preset) = get_global_idx imbc name in
				if is_preset then
					add_opcode imbc (PushPresetGlobal idx)
				else
					add_opcode imbc (PushUninitGlobal idx)
			| GetLocal (name,r) -> add_opcode imbc (PushLocal (get_local_var_idx name))
			| GetParam (name,r) -> add_opcode imbc (PushLocal (get_param_var_idx name))
			| GetExtSymbol (name,r) ->
				let (imbc,v) = collect_external imbc name in
				add_constant_opcode imbc v
			| SetGlobal (name,stmt,r) ->
				let imbc = gen_code imbc stmt in

				let (idx,is_preset) = get_global_idx imbc name in
				if is_preset then
					add_opcode imbc (StorePresetGlobal idx)
				else
					add_opcode imbc (StoreUninitGlobal idx)
			| SetLocal (name,stmt,r) ->
				let imbc = gen_code imbc stmt in
				add_opcode imbc (StoreLocal (get_local_var_idx name))
			| SetParam (name,stmt,r) ->
				let imbc = gen_code imbc stmt in
				add_opcode imbc (StoreLocal (get_param_var_idx name))
			| Program.Call (name,params,r) ->
				let imbc = List.fold_left (fun imbc s -> gen_code imbc s) imbc (List.rev params) in
				add_opcode imbc (Imbc.Call name)
			| CallAsm (name,params,r) ->
				let (asm_params,asm_code,asm_attrs,_) = List.assoc name prg.prg_asmfun_list in
				let op = AsmFun (name,asm_params,asm_code,asm_attrs) in

				let imbc = List.fold_left (fun imbc s -> gen_code imbc s) imbc params in
				add_opcode imbc op
			| CallC (ct,params,r) ->
				let imbc = List.fold_left (fun imbc s -> gen_code imbc s) imbc (List.rev params) in
				(match ct with
				| Calltype_Cdecl -> add_opcode imbc (Imbc.Call_Cdecl ((List.length params) - 1))
				| Calltype_Cdecl_FP -> add_opcode imbc (Imbc.Call_CdeclFP ((List.length params) - 1))
				| Calltype_Stdcall -> add_opcode imbc Imbc.Call_Stdcall
				| Calltype_Stdcall_FP -> add_opcode imbc Imbc.Call_StdcallFP)
			| If (condstmt,truestmt,falsestmt,r) ->
				let lbl_false = uid () in
				let lbl_end = uid () in
				
				let imbc = gen_code imbc condstmt in
				let imbc = add_opcode imbc (JumpIfNot lbl_false) in
				let imbc = gen_code imbc truestmt in
				let imbc = add_opcode imbc (Jump lbl_end) in
				let imbc = add_opcode imbc (Label lbl_false) in
				let imbc = gen_code imbc falsestmt in
				add_opcode imbc (Label lbl_end)
			| While (condstmt,codestmt,r) ->
				let lbl_loop = uid () in
				let lbl_end = uid () in

				let orig_loop_break = !loop_break in
				let orig_loop_continue = !loop_continue in
				
				(* The default return value. *)
				let imbc = add_constant_opcode imbc (Val_Int Int32.zero) in

				(* Start of loop (label + test *)
				let imbc = add_opcode imbc (Label lbl_loop) in
				let imbc = gen_code imbc condstmt in
				let imbc = add_opcode imbc (JumpIfNot lbl_end) in

				(* The looping code; first clear the return value, then run the code (which will be the next return value). *)
				loop_break := Some lbl_end;
				loop_continue := Some lbl_loop;
				let imbc = add_opcode imbc (Pop) in
				let imbc = gen_code imbc codestmt in
				loop_break := orig_loop_break;
				loop_continue := orig_loop_continue;

				(* Loop back. *)
				let imbc = add_opcode imbc (Jump lbl_loop) in
				add_opcode imbc (Label lbl_end)
			| Break r ->
				(* The breaks return value. *)
				let imbc = add_constant_opcode imbc (Val_Int Int32.zero) in

				(match !loop_break with
				| None -> failwith ("Break encountered outside of a loop in function '" ^ name ^ "' at " ^ (Pos_obj.range_desc r));
				| Some dst -> add_opcode imbc (Jump dst))
			| Continue r ->
				(* The continues return value. *)
				let imbc = add_constant_opcode imbc (Val_Int Int32.zero) in

				(match !loop_continue with
				| None -> failwith ("Continue encountered outside of a loop in function '" ^ name ^ "' at " ^ (Pos_obj.range_desc r));
				| Some dst -> add_opcode imbc (Jump dst))
			| Program.Return (s,r) -> add_opcode (gen_code imbc s) Imbc.Return
		in
		let imbc = gen_code imbc func.func_code in
		
		let imbc = add_opcode imbc Return in

		(* Add the function. *)
		let new_fun =
			{
				ifun_params = (List.length func.func_params);
				ifun_stack_entries = (List.length func.func_locals);
				ifun_code = List.rev !rev_code;
			}
		in
		{ imbc with imbc_funs = (name,new_fun) :: imbc.imbc_funs }
	in
	let imbc = List.fold_left collect_fun imbc prg.prg_fun_list in

	(* Done. *)
	imbc
