(** The data structures and the generator for generating whole programs. *)

(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(** Identifier name. *)
type id_name = string

(** Function attributes. *)
type fun_attr =
	| FunAttr_UseVmFloatTest
type fun_attrs = fun_attr list

(** Raw assembly function. Contains the raw code and the number of parameters. *)
type asm_fun = int * (string list) * fun_attrs * Pos_obj.range

(** Normal program function. *)
type func =
{
	func_params: string list;
	func_locals: string list;
	func_code: stmt;
	func_range: Pos_obj.range;
}
and stmts = stmt list
and stmt =
	| Nop of Pos_obj.range
	| Block of stmts * Pos_obj.range
	| ConstInt of int32 * Pos_obj.range
	| ConstFloat of float * Pos_obj.range
	| ConstRawArray of string * Pos_obj.range
	| GetGlobal of string * Pos_obj.range
	| GetLocal of string * Pos_obj.range
	| GetParam of string * Pos_obj.range
	| GetExtSymbol of string * Pos_obj.range
	| SetGlobal of string * stmt * Pos_obj.range
	| SetLocal of string * stmt * Pos_obj.range
	| SetParam of string * stmt * Pos_obj.range
	| Call of string * stmts * Pos_obj.range
	| CallAsm of string * stmts * Pos_obj.range
	| CallC of call_type * stmts * Pos_obj.range
	| If of stmt * stmt * stmt * Pos_obj.range
	| While of stmt * stmt * Pos_obj.range
	| Break of Pos_obj.range
	| Continue of Pos_obj.range
	| Return of stmt * Pos_obj.range
and call_type =
	| Calltype_Cdecl
	| Calltype_Cdecl_FP
	| Calltype_Stdcall
	| Calltype_Stdcall_FP

(** Global variable types. *)
type global_var =
	| Global_Empty of Pos_obj.range
	| Global_Int of int32 * Pos_obj.range
	| Global_Float of float * Pos_obj.range
	| Global_Raw_Array of string * Pos_obj.range
	| Global_Uninitialized_Array of int * Pos_obj.range

(** The main program type. *)
type prg =
{
	prg_heap_size: int;
	prg_constants: (id_name * stmt) list;
	prg_globals: (id_name * global_var) list;
	prg_asmfun_list: (id_name * asm_fun) list;
	prg_fun_list: (id_name * func) list;
}

(** Converts a function attribute to a string. *)
val string_of_fun_attr : fun_attr -> string

(** Converts a string to a function attribute. Throws Not_found if it's an invalid string. *)
val fun_attr_of_string : string -> fun_attr

(** Tests if the attribute is in the in the list. *)
val has_attr : fun_attrs -> fun_attr -> bool

(** Generates a program from the IL4 tree. *)
val generate : Il4_reader.item_list -> prg
