#include "textdata.h"
#include <afxmt.h>


CTextData::CTextData() :
	CImageData(),
	m_iceColors(false),
	m_list(0),
	m_userFont(0),
	m_userPalette(0)
{
	m_list = new CLineList();
}


CTextData::~CTextData() {

	delete m_list;
	m_list = 0;

	m_buffer.clear();

	if (m_userFont) {
		delete m_userFont;
		m_userFont = 0;
	}
	
	if (m_userPalette) {
		delete m_userPalette;
		m_userPalette = 0;
	}
}


/*! large switch statement that will execute the command with the given argument */
void CTextData::executeCommand(AnsiCommandData *command) {

	switch(command->command) {
		case ANSI_STORE:	
			store(command->argbuf);
			break;
		case ANSI_SET_ATTRIBUTES:
			setAttributes(command->argbuf);
			break;
		case ANSI_SET_POSITION:
			setPosition(command->argbuf);
			break;
		case ANSI_MOVE_UP:
			moveUp(command->argbuf);
			break;
		case ANSI_MOVE_DOWN:
			moveDown(command->argbuf);
			break;
		case ANSI_MOVE_FORWARD:
			moveForward(command->argbuf);
			break;
		case ANSI_MOVE_BACKWARD:
			moveBackward(command->argbuf);
			break;
		case ANSI_SAVE_POSITION:
			savePosition();
			break;
		case ANSI_RESTORE_POSITION:
			restorePosition();
			break;
		case ANSI_CLEAR_SCREEN:
			clearScreen(command->argbuf);
			break;
		case ANSI_CLEAR_LINE:
			clearLine(command->argbuf);
			break;
	}
}

/*
 *			ANSI Commands Helpers
 */

void get1arg( char *args, int *a1 ) {
	*a1 = atoi( args );
}

void get2args( char *args, int *a1, int *a2 ) {
	char *tmp;
	
	*a1 = atoi( args );
	tmp = (char *) strchr( args, ';' );
	if (!tmp) *a2 = 0;
	else *a2 = atoi( tmp+1 );
}

/*! Generates an array of integers for the arguments, terminated by a -1. */
int *getargv( char *args, int &index) {

	static int argv[20];
	index = 0;

	if (args[0]>='0' && args[0]<='9') {
		memset(argv, 0xff, 20*4);
		argv[index++] = atoi(args);
		char *this_arg = args;	
		do {
			this_arg = (char *) strchr(this_arg, ';');
			if (this_arg!=NULL) {
				this_arg++;
				int i = atoi(this_arg);
				if (i) {
					argv[index++] = i;
				} else {
					i=0;
				}
			}
		}while(this_arg!=NULL);
	}

	return argv;
}

/*
 *			ANSI Commands
 */

/*! Stores a byte in the current position, wrapping if necessary, and increments the position. */
void CTextData::storeImp(char c ) {
		
	m_line[m_x*2] = c; 
	m_line[m_x*2 + 1] = m_attribute; 
	m_x++;
}

/*! thin wrapper for storeImp which handles the very few exceptions */
void CTextData::store(char *s) {
	int count;
	int i=0;
	static char lastChar = 0;
	
	while(s[i]) {
		
		switch(s[i]) {
	/*
			case '\n':
				m_line = m_list->forwardCat(); 
				m_x = 0;
				break;
	*/
			case '\n':
			case '\r':
				break;
			case '\t':
				count = m_x % TABSTOP;
				if (count) {
					count = TABSTOP - count;
				}
				//gz added
				else {
					count = TABSTOP;
				}
				//
				for (i=0; i<count; i++) {
					storeImp(' '); 
				} 
				break;		

			//new 6.1
			//backspace
			case 8:
				if (m_x>0)
					m_x--;
				break;
			default:			
				storeImp(s[i]);
				break;
		}

		if (m_x==WRAP || (s[i]=='\n' && lastChar!='\n')) { 
			m_x = 0; 
			m_line = m_list->forwardCat(); 
		}

		lastChar = s[i];
		i++;
	}	
}

/*	Moves the active position upward without altering the column position. The number of lines moved 
	is determined by the parameter. A parameter value of zero or one moves the active position one 
	line upward. A parameter value of n moves the active position n lines upward. If an attempt is 
	made to move the cursor above the top margin, the cursor stops at the top margin. */
void CTextData::moveUp( char *args ) {
/*
	int i;
	int num_move;
	get1arg( args, &num_move );
	if (!num_move) num_move = 1;
	for (i=0; i<num_move; i++) {
		m_line = m_list->backwardCat();
	}
*/
/*
	int numArgs;
	int *nums = getargv(args, numArgs);

	if (numArgs==0) {
		m_line = m_list->backwardCat();		
	} else {
		for (int i=0; i<nums[0]; i++) {
			m_line = m_list->backward();
		}
	}
*/
	int numArgs;
	int *nums = getargv(args, numArgs);
	if (!numArgs || nums[0]==0) nums[0]=1;
	for (int i=0; i<nums[0]; i++) {
		m_line = m_list->backward();
	}
}
/*	The CUD sequence moves the active position downward without altering the column position. 
	The number of lines moved is determined by the parameter. If the parameter value is zero or one, 
	the active position is moved one line downward. If the parameter value is n, the active position 
	is moved n lines downward. In an attempt is made to move the cursor below the bottom margin, 
	the cursor stops at the bottom margin. */
void CTextData::moveDown( char *args ) {
/*
	int i;
	int num_move;
	
	get1arg( args, &num_move );
	if (!num_move) num_move = 1;
	
	for (i=0; i<num_move; i++) {
		m_line = m_list->forwardCat();
	}
//*/

	int numArgs;
	int *nums = getargv(args, numArgs);

	if (!numArgs || nums[0]==0) nums[0] = 1;
	for (int i=0; i<nums[0]; i++) 
		m_line = m_list->forward();	
}

/*	The CUF sequence moves the active position to the right. The distance moved is determined by the parameter. 
	A parameter value of zero or one moves the active position one position to the right. 
	A parameter value of n moves the active position n positions to the right. If an attempt is made to move 
	the cursor to the right of the right margin, the cursor stops at the right margin. */
void CTextData::moveForward( char *args ) {
/*
	int num_move;
	get1arg( args, &num_move );
	if (!num_move) num_move = 1;
	m_x += num_move;
	if ( m_x > WRAP-1 ) m_x = WRAP-1;
//*/

/*	
	int numArgs;
	int *nums = getargv(args, numArgs);
	if (numArgs==0) {
		if ((m_x+1)<WRAP) {
			m_x++;
		}
	} else if ((m_x+nums[0])<WRAP) {
		m_x += nums[0];
	} else {
		m_x = 0;
	}
*/

	int numArgs;
	int *nums = getargv(args, numArgs);
	if (!numArgs || nums[0]==0) nums[0]=1;
	m_x = __min(WRAP-1, m_x+nums[0]);
}

/* The CUB sequence moves the active position to the left. The distance moved is determined by the parameter. 
	If the parameter value is zero or one, the active position is moved one position to the left. 
	If the parameter value is n, the active position is moved n positions to the left. 
	If an attempt is made to move the cursor to the left of the left margin, the cursor stops at the left margin */
void CTextData::moveBackward( char *args ) {
/*
	int num_move;
	get1arg( args, &num_move );
	if (!num_move) num_move = 1;
	m_x -= num_move;
	if ( m_x < 0 ) m_x = 0;
//*/

	int numArgs;
	int *nums = getargv(args, numArgs);
	if (!numArgs || nums[0]==0) nums[0] = 1;
	m_x = __max(0, m_x-nums[0]);
}

void CTextData::savePosition() {
	m_list->savePosition(m_x);
}

void CTextData::restorePosition() {
	m_x	   = m_list->loadSavedPosition();
	m_line = m_list->getContents();
}

/*	The CUP sequence moves the active position to the position specified by the parameters. 
	This sequence has two parameter values, the first specifying the line position and the second specifying 
	the column position. A parameter value of zero or one for the first or second parameter moves the 
	active position to the first line or column in the display, respectively. 
	The default condition with no parameters present is equivalent to a cursor to home action. 
	In the VT100, this control behaves identically with its format effector counterpart, HVP.*/
void CTextData::setPosition( char *args ) {
	int pos_x, pos_y;
	int i;
	
	get2args( args, &pos_y, &pos_x );
	// adjust if y specified
	if (!pos_y || pos_y==0) pos_y = 1;
	if (!pos_x || pos_x==0) pos_x = 1;
	
	m_list->rewind(); 
	for (i=0; i<pos_y; i++ ) {
		m_line = m_list->forwardCat();
	}
	m_x = pos_x - 1;
}

/*	This sequence erases some or all of the characters in the display according to the parameter. Any complete line erased by this sequence will return that line to single width mode. Editor Function
	Parameter 	Parameter Meaning
	0 	Erase from the active position to the end of the screen, inclusive (default)
	1 	Erase from start of the screen to the active position, inclusive
	2 	Erase all of the display -- all lines are erased, changed to single-width, and the cursor does not move.*/
void CTextData::clearScreen(char *args) {
/*
	m_list->init(WRAP*2);
	m_line = m_list->forwardCat();	
//*/

//6.1
/*
	char *last = 0;
	m_line = m_list->top();
	while(last!=m_line) {
		m_list->initLine(m_line);
		last = m_line;
		m_line = m_list->forward();
	}
	m_list->reset();
	m_line = m_list->top();
//*/

	int numArgs;
	int *nums = getargv(args, numArgs);
	if (!numArgs) nums[0] = 0;

	char *last = 0;	
	switch(nums[0]) {

		case 0:	
			do {
				m_list->initLine(m_line);
				last = m_line;
				m_line = m_list->forward();
			} while(m_line!=last);
			break;

		case 1:

			last   = m_line;
			m_line = m_list->top();
			do {
				m_list->initLine(m_line);				
				m_line = m_list->forward();
			} while(m_line!=last);			
			m_list->initLine(m_line);				
			break;

		case 2:
			m_line = m_list->top();
			do {
				m_list->initLine(m_line);
				last = m_line;
				m_line = m_list->forward();
			}while(last!=m_line);
			break;
	}

	/*! \note setting the line to the top which is wrong */
	m_list->reset();
	m_line = m_list->top();
}


/*	Erases some or all characters in the active line according to the parameter. Editor Function
	Parameter	Parameter Meaning
	0 			Erase from the active position to the end of the line, inclusive (default)
	1 			Erase from the start of the screen to the active position, inclusive
	2 			Erase all of the line, inclusive */
void CTextData::clearLine(char *args) {

	/*! \note prev acidview implementation used just 1 and then set m_x = 0 */

	int numArgs;
	int *nums = getargv(args, numArgs);
	if (!numArgs) nums[0] = 0;

	int i;
	switch(nums[0]) {
		case 0:
			for(i=m_x; i<WRAP; i++) {
				m_line[i*2]   = ' ';
				m_line[i*2+1] = m_attribute; 
			} 
			break;

		case 1:
			for(i=m_x; i>=0; i--) {
				m_line[i*2]   = ' ';
				m_line[i*2+1] = m_attribute; 
			} 
			break;

		case 2:
			for(i=0; i<WRAP; i++) {
				m_line[i*2]   = ' ';
				m_line[i*2+1] = m_attribute; 
			} 
			break;
	}
}

void CTextData::setAttributes(char *args) {
/*
	int i;
	int *args;
	args = getargv(argbuf);
	for ( i=0; args[i] != -1; i++ ) {
		if ( args[i] == 0 ) {  
			// reset attribute 
			m_attribute = BIN_WHITE; 
		} else if ( args[i] == 1 ) {
			// set bold on 
			m_attribute |= 0x08;
		} else if ( args[i] == 5 ) {
			// set blink on
			m_attribute |= 0x80;
		//!! Used to be just one & in the next line
		} else if ( args[i] >= 30 && args[i] <= 37 ) {
			// foreground
			m_attribute &= 0xf8;
			m_attribute |= color_conversion_table[ args[i] - 30 ];
		} else if ( args[i] >= 40 && args[i] <= 47 ) {
			m_attribute &= 0x8f;
			// background
			m_attribute |= ( color_conversion_table[ args[i] - 40 ] << 4 );
		}
	}
*/	
	int numArgs;
	int *nums = getargv(args, numArgs);	
	if (numArgs==0) {
		m_attribute = BIN_WHITE;
	} else {

		for (int i=0; i<numArgs; i++) {

			if ( nums[i] == 0) {  
				// reset attribute 
				m_attribute = BIN_WHITE; 
			} else if ( nums[i] == 1 ) {
				// set bold on 
				m_attribute |= 0x08;
			} else if ( nums[i] == 5 ) {
				// set blink on
				m_attribute |= 0x80;
			//!! Used to be just one & in the next line
			} else if ( nums[i] >= 30 && nums[i] <= 37 ) {
				// foreground
				m_attribute &= 0xf8;
				m_attribute |= color_conversion_table[ nums[i] - 30 ];
			} else if ( nums[i] >= 40 && nums[i] <= 47 ) {
				m_attribute &= 0x8f;
				// background
				m_attribute |= ( color_conversion_table[ nums[i] - 40 ] << 4 );

			//new 6.1
			} else if (nums[i]==7) {
				m_attribute = BIN_WHITE;
			} else if (nums[i]==8) {
				m_attribute = 0;
			}
/*
#ifdef _DEBUG
			else {

				::MessageBox(NULL, args, "attribute not found", MB_OK);
			}		
#endif
*/

		}
	}
}

void CTextData::addCommand(AnsiCommandData *command) {	

	CSingleLock singleLock(m_semaphore);
	singleLock.Lock();
	
	m_buffer.push_back(*command);

	singleLock.Unlock();
	//Sleep(0);
}


int CTextData::getNextCommand(AnsiCommandData &a) {	

	if (m_buffer.empty())
		return -1;

	CSingleLock singleLock(m_semaphore);
	singleLock.Lock();
	
	a = (AnsiCommandData)m_buffer.front();
	m_buffer.pop_front();

	singleLock.Unlock();
	//Sleep(0);

	return m_buffer.size();
}

bool CTextData::hasCommands() {
	return (m_buffer.size()!=0);
}