#include "imagefactory.h"
#include "ImageLoader.h"
#include "JpegLoader.h"
#include "AnsiLoader.h"
#include "BinLoader.h"
#include "XBinLoader.h"
#include "AdfLoader.h"
#include "GifLoader.h"
#include "PngLoader.h"
#include "RipLoader.h"
#include "BmpLoader.h"


#include "ImageRenderer.h"
#include "TextRenderer.h"
#include "BinaryRenderer.h"
#include "RipRenderer.h"
#include "AcidImage.h"

/*! Can change this string in the binary to track leaks */
char distributionString[] = "nicknamexxxxxxxxxx";

using namespace std;


CImageFactory::CImageFactory() :
	m_enableRip(false),
	m_ripRenderer(0)
{

	m_textRenderer   = new CTextRenderer();
	m_binaryRenderer = new CBinaryRenderer();

	m_loaders.push_back(new CJpegLoader(m_binaryRenderer));
	m_loaders.push_back(new CGifLoader(m_binaryRenderer));
	m_loaders.push_back(new CPngLoader(m_binaryRenderer));
	m_loaders.push_back(new CBmpLoader(m_binaryRenderer));

	m_loaders.push_back(new CBinLoader(m_textRenderer));
	m_loaders.push_back(new CXbinLoader(m_textRenderer));
	m_loaders.push_back(new CAdfLoader(m_textRenderer));
	
#ifdef ENABLE_RIP
	m_ripRenderer	 = new CRipRenderer();
	m_enableRip = true;
	m_loaders.push_back(new CRipLoader(m_ripRenderer));
#endif

	/*! \note Last loader must be the AnsiLoader*/
	m_loaders.push_back(new CAnsiLoader(m_textRenderer));
}


CImageFactory::~CImageFactory() {

	for(vector<CImageLoader *>::iterator i=m_loaders.begin(); i!=m_loaders.end(); i++)
		delete (*i);

	m_loaders.clear();

	delete m_textRenderer;
	delete m_binaryRenderer;
	delete m_ripRenderer;
}


CAcidImage *CImageFactory::loadFile(const char *filename) {

	/*! \todo if the filename contains | the then file is to be
		extracted from an archive file */

	//open file and read the first four signature bytes
	FILE *fp = fopen(filename, "rb");
	if (!fp) {
#ifdef _DEBUG
		OutputDebugString("File Not Found\n");
#endif
		return 0;
	}

	fseek(fp, 0, SEEK_END);
	if (ftell(fp)<4)
		return 0;
	fseek(fp, 0, SEEK_SET);

	char signature[4];
	fread(signature, 4, 1, fp);

	/*! \note extension is needed for files without a signature (bin, rip)*/
	//read extension... seek the period from the back
	int ext = strlen(filename)-1;
	while(filename[ext]!='.' && ext>0) {
		ext--;
	}
	ext++;

	//iterate through all the loaders to see what can load the file
	for (vector<CImageLoader *>::iterator i=m_loaders.begin(); i!=m_loaders.end(); i++) {

		if ((*i)->isFileType(signature, filename+ext)) {
			return (new CAcidImage(filename, (*i)));
		}
	}

	//if no specific loader is found, use the text loader 
	return (new CAcidImage(filename, m_loaders.back()));
}


CAcidImage *CImageFactory::loadFileFromData(const char *data, int length, const char *filename) {

	
	//open file and read the first four signature bytes
	if (!data) {
#ifdef _DEBUG
		OutputDebugString("no data located in ImageFactory::loadFileFromData\n");
#endif
		return 0;
	}

	if (length<4)
		return 0;
	char signature[4];
	memcpy(signature, data, 4);

	/*! \note extension is needed for files without a signature (bin, rip)*/
	//read extension... seek the period from the back
	int ext = 0;
	if (filename) {
		ext = strlen(filename)-1;
		while(filename[ext]!='.' && ext>0) {
			ext--;
		} 
		ext++;
	}

	//iterate through all the loaders to see what can load the file
	for (vector<CImageLoader *>::iterator i=m_loaders.begin(); i!=m_loaders.end(); i++) {

		if ((*i)->isFileType(signature, filename+ext)) {
			return (new CAcidImage(data, length, (*i)));
		}
	}

	//if no specific loader is found, use the text loader 
	return (new CAcidImage(data, length, m_loaders.back()));
}

CAcidImage *CImageFactory::loadFileFromResource(int resource) {

	HRSRC hrsrc = FindResource(NULL, MAKEINTRESOURCE(resource), "Splash");
	if (hrsrc)
	{
		HGLOBAL hglobal = LoadResource(NULL, hrsrc);
		if (hglobal)
		{
			int length = SizeofResource(NULL, hrsrc);
			LPVOID buf = LockResource(hglobal);		
			return loadFileFromData((const char *)buf, length);
			/*! \note says you don't need to free the resources used by this function, but 
				i really don't believe big g */
		}//if loaded resource
	}//if found resource

	return 0;

}

void CImageFactory::enableRip() {
		
	if (!m_enableRip) {
		m_enableRip = true;
		m_ripRenderer	 = new CRipRenderer();
		m_loaders.insert(m_loaders.begin(), new CRipLoader(m_ripRenderer));
	}
}