/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes
#include <caosGL/gfx/cScene.h>

// extern includes
#include <caosGL/core/cParser.h>
#include <caosGL/core/cRegistry.h>
#include <caosGL/core/cUtil.h>

using namespace caosGL::gfx;
using namespace caosGL::core;

#define cRandomSet_attribsFile <caosGL/effects/cRandomSet.attribs>

namespace caosGL {
	namespace effects {
		static vector <tAttr> attributeNames;
		static log4cpp::Category& cat = log4cpp::Category::getInstance ("caosGL::effects::cRandomSet");

			/**
			 *<br> class:		cRandomSet
			 *<br> namespace:	caosGL::effects
			 *<br> inherits:	caosGL::gfx::cScene
			 *<br> implements:	<none>
			 *<br> purpose:		Used to randomize a numeric atribute of an object over a range, with
			 *					a given permutation.
			 *
			 */

		class CAOSGL_API cRandomSet : public cScene {
			typedef cScene super;
		private:
			cBaseNode * _bn;
			#define ATTRIB(n,t,v,d) ATTRIB_DEFINE_VAR(n,t)
			#include cRandomSet_attribsFile
			#undef ATTRIB
			tDWord _attr;
			tFloat _last;

		public:
			/********************************************************************************************/
			cRandomSet (const string n, cGroup * f) : super (n,f) {
				init ();
			}

			/********************************************************************************************/
			~cRandomSet () {
			}

			/********************************************************************************************/
			tFloat ran (tFloat range) const {
				return (static_cast <tFloat> (rand ())/0x7fff)*range;
			}

			// from iDrawable
			/********************************************************************************************/
			tBool draw (const tFloat time) {
				return true;
			}

			// from iDrawable
			/********************************************************************************************/
			tBool transparent () const {
				return false;
			}

			// from iDrawable
			/********************************************************************************************/
			tVoid transparent (const tBool trasnp) {
			}

			// from cBaseNode
			/********************************************************************************************/
			tBool compile () {
				_bn = cRegistry::getNode (_obj);
				if (!_bn) {
					cat.error ("%s::%s sorry, cannot find obj %s to randomize...",
						getTypeName().c_str(), _name.c_str(), _obj.c_str());
					return false;
				}
				if (_last < _min)
					_last = _min;
				if (_last > _max)
					_last = _max;
				if (_perm>1||_perm<0)
					cat.warn ("%s::%s permutation value is out of the recomended range %f", 
					getTypeName().c_str(), _name.c_str(), _perm);
				return true;
			}

			// from cBaseNode
			/********************************************************************************************/
			tBool visit (tFloat t) {
				if (!cBaseNode::visit (t))
					return false;

				if (!_bn)
					return false;

				tFloat d0 = (_max-_min)*_perm;
				tFloat d1 = ran (d0)-d0/2;
				d1 = _last + d1;
				if (d1<_min)
					d1 = _min;
				if (d1>_max)
					d1 = _max;

				_bn->set (_attr, d1);
				//cat.info ("random number (%s.%s) [%f,%f]: %f (last: %f)", _obj.c_str(), _attr.rcharData(), _min, _max, d1, _last);
				_last = d1;

				return true;
			}

			// from cBaseNode
			/********************************************************************************************/
			tVoid leave () {
				return;
			}

			// from cBaseNode
			/********************************************************************************************/
			tBool init () {
				#define ATTRIB(n,t,v,d) ATTRIB_INIT_VAR(n,v)
				#include cRandomSet_attribsFile
				#undef ATTRIB
				_bn = cNULL;
				_last = -1;
				return true;
			}

			// from cBaseNode
			/********************************************************************************************/
			const tBool set (const tDWord key, const string & value) {
				if (super::set (key, value)) return true;
				switch (key) {
					#define ATTRIB(n,t,v,d) ATTRIB_SET_S(n)
					#include cRandomSet_attribsFile
					#undef ATTRIB
				case 'attr': { _attr = cUtil::doFli (value); } return true;
				case '    ': return false;
				default: return false;
				}
				return false;
			}

			// from cBaseNode
			/********************************************************************************************/
			const tBool set (const tDWord key, const tFloat & value) {
				if (super::set (key, value)) return true;
				switch (key) {
					#define ATTRIB(n,t,v,d) ATTRIB_SET_N(n)
					#include cRandomSet_attribsFile
					#undef ATTRIB
				case '    ': return false;
				default: return false;
				}
				return false;
			}

			// from cBaseNode
			/********************************************************************************************/
			const tBool get (const tDWord key, string & value) const {
				if (super::get (key, value)) return true;
				switch (key) {
					#define ATTRIB(n,t,v,d) ATTRIB_GET(n)
					#include cRandomSet_attribsFile
					#undef ATTRIB
				case 'attr': { value = cUtil::doFli (_attr); } return true;
				case '    ': return false;
				default: return false;
				}
				return false;
			}

			// from cBaseNode
			/********************************************************************************************/
			const vector <tAttr> * getAttributeNames () const {
				if (attributeNames.size () == 0) {
					const vector <tAttr> * v = super::getAttributeNames ();
					attributeNames.insert (attributeNames.begin (), v->begin (), v->end ());
					#define ATTRIB(n,t,v,d) ATTRIB_ATTRIBNAMES(n,d)
					#include cRandomSet_attribsFile
					#undef ATTRIB
					attributeNames.push_back(make_pair('attr',string("The attribute to randomize")));
				}
				return &attributeNames;
			}

			// from cBaseNode
			/********************************************************************************************/
			const string getTypeName () const { return "caosGL::effects::cRandomSet"; }
		};
	}
}

// for node creation
#include <caosGL/core/cRegistry.h>
#include <caosGL/gfx/cNodeCreator.h>

class cRandomSetNodeCreator : public cNodeCreator {
public:
	cRandomSetNodeCreator () {
		name ("caosGL::effects::cRandomSet");
	}
	cBaseNode * createNode (const string n, cGroup * f) {
		return new caosGL::effects::cRandomSet (n,f);
	}
};
caosGL::core::cRegisterNodeCreator <cRandomSetNodeCreator> cRandomSetNodeCreatorInstance;

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof