/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

#ifndef CAOSGL_GFX_CVECTOR
#define CAOSGL_GFX_CVECTOR

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes

// extern includes
#include <caosGL/gfx/caos3dtypes.h>
#include <math.h>

namespace caosGL { 
	namespace gfx {
		/**
		 *<br> class:		cVector
		 *<br> namespace:	caosGL::gfx
		 *<br> inherits:	<none>
		 *<br> implements:	<none>
		 *<br> purpose:		Handy 3D vector class, integrates with the cMatrix class.
		 *
		 */

		class CAOSGL_API cVector {
		public:
			/********************************************************************************************/
			cVector () {}
			
			/********************************************************************************************/
			cVector (const tVector & v) {
				x=v.x; y=v.y; z=v.z;
			}
			
			/********************************************************************************************/
			cVector (const cVector & v) {
				x=v.x; y=v.y; z=v.z;
			}
			
			/********************************************************************************************/
			cVector (tFloat _x, tFloat _y, tFloat _z) {
				x=_x; y=_y; z=_z;
			}
			
			/********************************************************************************************/
			~cVector () {}

			/********************************************************************************************/
			tFloat x,y,z,w;

			/********************************************************************************************/
			tVoid angles (tFloat & alpha, tFloat & theta) {
				tFloat lenXY = sqrt (x*x+y*y); if (lenXY==0) return;
				tFloat lenXZ = sqrt (x*x+z*z); if (lenXZ==0) return;
				alpha = asin (x/lenXY);
				theta = asin (x/lenXZ);
			}
			
			/********************************************************************************************/
			inline const bool operator == (const cVector& v) const { return x==v.x && y==v.y && z==v.z; }
			
			/********************************************************************************************/
			inline const bool operator < (const cVector& v) const	{
				if (x<v.x) return true;
				else if (x>v.x) return false;
				else if (y<v.y) return true;
				else if (y>v.y) return false;
				else return (z<v.z);
			}


			/********************************************************************************************/
			inline void set (tFloat _x, tFloat _y, tFloat _z) {
				x=_x; y=_y; z=_z;
			}

			/********************************************************************************************/
			/// dot product
			inline tFloat operator * (const cVector& rhs) const {
				return x*rhs.x+y*rhs.y+z*rhs.z;
			}

			/********************************************************************************************/
			/// cross product
			inline const cVector operator ^ (const cVector& rhs) const {
				return cVector (y*rhs.z-z*rhs.y,
					z*rhs.x-x*rhs.z ,
					x*rhs.y-y*rhs.x);
			}

			/********************************************************************************************/
			/// multiply by scalar
			inline const cVector operator * (const tFloat& rhs) const {
				return cVector (x*rhs, y*rhs, z*rhs);
			}

			/********************************************************************************************/
			/// unary multiply by scalar
			inline cVector& operator *= (const tFloat& rhs) {
				x*=rhs;
				y*=rhs;
				z*=rhs;
				return *this;
			}

			/********************************************************************************************/
			/// divide by scalar
			inline const cVector operator / (const tFloat& rhs) const {
				return cVector (x/rhs, y/rhs, z/rhs);
			}

			/********************************************************************************************/
			/// unary divide by scalar
			inline cVector& operator /= (const tFloat& rhs) {
				x/=rhs;
				y/=rhs;
				z/=rhs;
				return *this;
			}

			/********************************************************************************************/
			/// binary vector add
			inline const cVector operator + (const cVector& rhs) const {
				return cVector (x+rhs.x, y+rhs.y, z+rhs.z);
			}

			/********************************************************************************************/
			/** unary vector add.  Slightly more efficient because no temporary
				intermediate object*/
			inline cVector& operator += (const cVector& rhs) {
				x += rhs.x;
				y += rhs.y;
				z += rhs.z;
				return *this;
			}

			/********************************************************************************************/
			/// binary vector subract
			inline const cVector operator - (const cVector& rhs) const {
				return cVector (x-rhs.x, y-rhs.y, z-rhs.z);
			}

			/********************************************************************************************/
			/// unary vector subract
			inline cVector& operator -= (const cVector& rhs) {
				x-=rhs.x;
				y-=rhs.y;
				z-=rhs.z;
				return *this;
			}

			/********************************************************************************************/
			/// negation operator.  Returns the negative of the Vec3
			inline const cVector operator - () const {
				return cVector (-x, -y, -z);
			}

			/********************************************************************************************/
			/// Length of the vector = sqrt (vec . vec)
			inline const tFloat length () const {
				return sqrtf (x*x + y*y + z*z);
			}

			/********************************************************************************************/
			/// Length squared of the vector = vec . vec
			inline const tFloat length2 () const {
				return x*x + y*y + z*z;
			}

			/********************************************************************************************/
			/** normalize the vector so that it has length unity
				returns the previous length of the vector*/
			inline const tFloat normalize () {
				tFloat norm = cVector::length ();
				x /= norm;
				y /= norm;
				z /= norm;
				return (norm);
			}
		};
	}
}

#endif // CAOSGL_GFX_CVECTOR
/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof