/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

#ifndef CAOSGL_GFX_CMATRIX
#define CAOSGL_GFX_CMATRIX

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes
#include <caosGL/gfx/cVector.h>

// extern includes
//#include <3dtypes.h>
#include <math.h>

using namespace caosGL::gfx;

namespace caosGL { 
	namespace gfx {
		/**
		 *<br> class:		cMatrix
		 *<br> namespace:	caosGL::gfx::threeD
		 *<br> inherits:	<none>
		 *<br> implements:	<none>
		 *<br> purpose:		4x4 cMatrix for storage & manipulation of transformations in scene graph. 
		 *					Provides basic maths operations, IO and via osg::Object reference counting.
		 *					You can directly load the cMatrix with OpenGL's LoadcMatrixf () function via
		 *					the public member _mat as the cMatrix is stored in the OpenGL format.
		 *					Caution: The disadvantage of this feature is, that the cMatrix access is
		 *					'transposed' if you compare it with the standard C/C++ 2d-array-access
		 *					convention . I.e. _mat[i][j] accesses the ith column of the jth row in the
		 *					4x4 cMatrix.
		 */

		class CAOSGL_API cMatrix {
		public:
			cMatrix ();
			cMatrix (const cMatrix& cMatrix);
			cMatrix (tFloat a00, tFloat a01, tFloat a02, tFloat a03,
				tFloat a10, tFloat a11, tFloat a12, tFloat a13,
				tFloat a20, tFloat a21, tFloat a22, tFloat a23,
				tFloat a30, tFloat a31, tFloat a32, tFloat a33);

			cMatrix& operator = (const cMatrix& cMatrix);

			virtual ~cMatrix ();

			void makeIdent ();

			void set (const tFloat* m);
			void set (const tDouble* m);

			void set (tFloat a00, tFloat a01, tFloat a02, tFloat a03,
					  tFloat a10, tFloat a11, tFloat a12, tFloat a13,
					  tFloat a20, tFloat a21, tFloat a22, tFloat a23,
					  tFloat a30, tFloat a31, tFloat a32, tFloat a33);

			void copy (const cMatrix& cMatrix);

			void makeScale (tFloat sx, tFloat sy, tFloat sz);
			void preScale (tFloat sx, tFloat sy, tFloat sz, const cMatrix& m );
			void postScale (const cMatrix& m, tFloat sx, tFloat sy, tFloat sz );

			void preScale (tFloat sx, tFloat sy, tFloat sz );
			void postScale (tFloat sx, tFloat sy, tFloat sz );

			void makeTrans (tFloat tx, tFloat ty, tFloat tz );
			void preTrans (tFloat tx, tFloat ty, tFloat tz, const cMatrix& m );
			void postTrans (const cMatrix& m, tFloat tx, tFloat ty, tFloat tz );

			void preTrans (tFloat tx, tFloat ty, tFloat tz );
			void postTrans (tFloat tx, tFloat ty, tFloat tz );


			/**
			 * Calc the rotation cMatrix which aligns vector \a old_vec with
			 * vector \a new_vec.  Both \a old_vec and \a new_vec must have
			 * length 1.0.
			 */
			void makeRot (const cVector& old_vec, const cVector& new_vec );

			void makeRot (tFloat deg, tFloat x, tFloat y, tFloat z );
			void preRot (tFloat deg, tFloat x, tFloat y, tFloat z, const cMatrix& m  );
			void postRot (const cMatrix& m, tFloat deg, tFloat x, tFloat y, tFloat z );

			void preRot (tFloat deg, tFloat x, tFloat y, tFloat z );
			void postRot (tFloat deg, tFloat x, tFloat y, tFloat z );

			void setTrans (tFloat tx, tFloat ty, tFloat tz );
			void setTrans (const cVector& v );
			cVector getTrans () const { return cVector (_mat[3][0],_mat[3][1],_mat[3][2]); } 

			void preMult (const cMatrix& m);
			void postMult (const cMatrix& m);
			void mult (const cMatrix& lhs,const cMatrix& rhs);

			cMatrix operator * (const cMatrix& m) const;

    		/** apply apply an 3x3 transform of v*M[0..2,0..2]  */
    		inline static cVector transform3x3 (const cVector& v,const cMatrix& m);
    		/** apply apply an 3x3 transform of M[0..2,0..2]*v  */
    		inline static cVector transform3x3 (const cMatrix& m,const cVector& v);

    		/** post multipy v. ie. (m*v) */
			inline cVector operator * (const cVector& v) const;

    		/** pre multipy v. ie. (v*m) */
			friend inline cVector operator * (const cVector& v,const cMatrix& m);

			bool invert (const cMatrix& m);

		public :
			tFloat _mat[4][4];

		protected:
	};

	inline cVector cMatrix::operator * (const cVector& v) const {
		tFloat d = 1.0f/ (_mat[3][0]*v.x+_mat[3][1]*v.y+_mat[3][2]*v.z+_mat[3][3]) ;
		return cVector ((_mat[0][0]*v.x + _mat[0][1]*v.y + _mat[0][2]*v.z + _mat[0][3])*d,
			 (_mat[1][0]*v.x + _mat[1][1]*v.y + _mat[1][2]*v.z + _mat[1][3])*d,
			 (_mat[2][0]*v.x + _mat[2][1]*v.y + _mat[2][2]*v.z + _mat[2][3])*d) ;
	}


	inline cVector operator * (const cVector& v,const cMatrix& m) {
		tFloat d = 1.0f/ (m._mat[0][3]*v.x+m._mat[1][3]*v.y+m._mat[2][3]*v.z+m._mat[3][3]) ;
		return cVector ((m._mat[0][0]*v.x + m._mat[1][0]*v.y + m._mat[2][0]*v.z + m._mat[3][0])*d,
			 (m._mat[0][1]*v.x + m._mat[1][1]*v.y + m._mat[2][1]*v.z + m._mat[3][1])*d,
			 (m._mat[0][2]*v.x + m._mat[1][2]*v.y + m._mat[2][2]*v.z + m._mat[3][2])*d);
	}

	inline cVector cMatrix::transform3x3 (const cVector& v,const cMatrix& m) {
		return cVector ((m._mat[0][0]*v.x + m._mat[1][0]*v.y + m._mat[2][0]*v.z),
					 (m._mat[0][1]*v.x + m._mat[1][1]*v.y + m._mat[2][1]*v.z),
					 (m._mat[0][2]*v.x + m._mat[1][2]*v.y + m._mat[2][2]*v.z));
	}

	inline cVector cMatrix::transform3x3 (const cMatrix& m,const cVector& v) {
		return cVector ((m._mat[0][0]*v.x + m._mat[0][1]*v.y + m._mat[0][2]*v.z),
					 (m._mat[1][0]*v.x + m._mat[1][1]*v.y + m._mat[1][2]*v.z),
					 (m._mat[2][0]*v.x + m._mat[2][1]*v.y + m._mat[2][2]*v.z) ) ;
	}

	}
}

#endif // CAOSGL_GFX_CMATRIX
/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof