#include <math.h>
#include <conio.h>
#include <dos.h>
#include <time.h>
#include <mem.h>
#include <malloc.h>

#include <wgt5.h>

/* Heightfield Renderer
   Written by Chris Egerter
   Placed in the public domain on September 29, 1995 */


//#define USE_VIRTUAL 1
/* Uncomment to use a virtual screen instead of writing directly onto
   the VGA ram. */

/* Keyboard defines */
#define KEY_UP 0x48
#define KEY_DOWN 0x50
#define KEY_LEFT 0x4B
#define KEY_RIGHT 0x4D
#define KEY_W 0x11
#define KEY_S 0x1F
#define KEY_E 0x12
#define KEY_D 0x20
#define KEY_R 0x13
#define KEY_F 0x21
#define KEY_F1 0x3B
#define KEY_F2 0x3C
#define KEY_F3 0x3D
#define KEY_CTRL 0x1D


/* General graphics info */
block virt;                     /* A virtual screen */
block ground_image;             /* Contains the colors of the heightfield */
block ground_data;              /* ground_image+4 */
block color_image;              /* Contains the heightfield data */
block color_data;               /* color_image+4 */
block cloud_image;              /* Background clouds bitmap */
color pal[256];                 /* Palette for all of the above */


/* View Variables */
unsigned short playerx;         /* Viewer's coordinates */
unsigned short playery;
int playerangle;                /* Viewer's direction (0-1279) */
int playerspeed;                /* Viewer's speed */
int elevation = 10170;
int RAYDISTANCE = 64;           /* Length of rays */
int cloudx = 0;                 /* Cloud position */
int eyedistance = 1;            /* Changes perspective */
int tiltx = 0;                  /* Amount of tilt */
unsigned long ydrawcoord;       /* Keeps track of current y coordinate when 
                                   drawing each column */


/* Heightfield Tables */
int isin[1280];
int icos[1280];
int persptable [1024];
int xanglestep[1280];
int yanglestep[1280];
int perscalc[1024];


/* Detail levels */
#define LOW_DETAIL 0
#define LOW_WIDTH 80
#define MEDIUM_DETAIL 1
#define MEDIUM_WIDTH 160
#define HIGH_DETAIL 2
#define HIGH_WIDTH 320
int draw_routine = MEDIUM_DETAIL;
int view_width = MEDIUM_WIDTH;
int pixelcount = 64;            /* Length of rays (can change) */
#define PIXPREC2 256L


int ticker_count;
int frames;


void timer_proc (void)
{
 ticker_count++;
}




/* The following inline assembly segments draw vertical lines of a solid
   color, in either 1, 2 or 4 pixel widths. */
unsigned long wvlineh (unsigned long scrcoord, unsigned short length, unsigned char col);
#pragma aux wvlineh = \
  "vloop: mov [ebx], al" \
  "sub ebx, 320" \
  "dec cx" \
  "jnz vloop" \
parm [ebx] [cx] [al]\
value [ebx]\
modify exact [ax cx ebx] nomemory;


unsigned long wvlinem (unsigned long scrcoord, unsigned short length, unsigned char col);
#pragma aux wvlinem = \
  "mov ah, al" \
  "vloop: mov [ebx], ax" \
  "sub ebx, 320" \
  "dec cx" \
  "jnz vloop" \
parm [ebx] [cx] [al]\
value [ebx]\
modify exact [ax cx ebx] nomemory;

unsigned long wvlinel (unsigned long scrcoord, unsigned short length, unsigned char col);
#pragma aux wvlinel = \
  "mov ah, al" \
  "mov dx, ax"\
  "shl eax, 16"\
  "mov ax, dx"\
  "vloop: mov [ebx], eax" \
  "sub ebx, 320" \
  "dec cx" \
  "jnz vloop" \
parm [ebx] [cx] [al]\
value [ebx]\
modify exact [ax cx ebx] nomemory;


/* The following inline assembly segments draw vertical lines copied from
   a bitmap, in either 1, 2 or 4 pixel widths. */
void wclineh (unsigned long scrcoord, unsigned long cldcoord, unsigned short length);
#pragma aux wclineh = \
  "mov esi,cloud_image" \
  "add esi,4" \
  "add esi,ebx" \
  "vloop: mov al, [esi]" \
  "mov [edi], al" \
  "sub edi, 320" \
  "sub esi, 320" \
  "dec cx" \
  "jg vloop" \
parm [edi] [ebx] [cx] \
modify exact [ax cx edi esi] nomemory;


void wclinem (unsigned long scrcoord, unsigned long cldcoord, unsigned short length);
#pragma aux wclinem = \
  "mov esi,cloud_image" \
  "add esi,4" \
  "add esi,ebx" \
  "vloop: mov ax, [esi]" \
  "mov [edi], ax" \
  "sub edi, 320" \
  "sub esi, 320" \
  "dec cx" \
  "jg vloop" \
parm [edi] [ebx] [cx] \
modify exact [ax cx edi esi] nomemory;

void wclinel (unsigned long scrcoord, unsigned long cldcoord, unsigned short length);
#pragma aux wclinel = \
  "mov esi,cloud_image" \
  "add esi,4" \
  "add esi,ebx" \
  "vloop: mov eax, [esi]" \
  "mov [edi], eax" \
  "sub edi, 320" \
  "sub esi, 320" \
  "dec cx" \
  "jg vloop" \
parm [edi] [ebx] [cx] \
modify exact [ax cx edi esi] nomemory;



/* Table creation */
void h_create_sincos (void)
{
int i;
float realangle;

 for (i = 0; i < 1280; i++)
   {
    realangle = i / 640.0 * 3.14153254234216;
    icos[i] = cos(realangle) * PIXPREC2;
    isin[i] =  sin(realangle) * PIXPREC2;
   }
}


void h_calculate_perspective (void)
{
int i;

 for (i = 0; i < 1024; i++)
    persptable [i] = (2024L / (i + 1));
}




unsigned char col;
unsigned char hgt;

/* Reads the color and height from the two heightfield bitmaps */
void h_read_heightfield (unsigned short ax, unsigned short ay, 
                        block height, block ground);
#pragma aux h_read_heightfield = \
  "mov ebx, 0" \
  "mov bh, dh" \
  "mov bl, ch" \
  "mov al, [esi + ebx]" \
  "mov col, al" \
  "mov al, [edi + ebx]" \
  "mov hgt, al" \
parm [cx] [dx] [edi] [esi]  \
modify exact [eax ecx ebx edx edi esi] nomemory;



void h_draw_heightfield (void)
{
int pixel_width;                /* Width of pixels (1,2,4) */
int leftangle;                  /* Angle of left ray */
int cloudcalc;                  /* Position of cloud bitmap */
int x, y;
unsigned short ax, ay;          /* Position of initial rays */
int gxstep, gystep;             /* Fixed point step value of ray */
int top_mountain;               /* Tells where the lowest pixel has been
                                   drawn (starts at 199) */
int oldpos, newpos;



int tiltfactor;
int linelength;
int height;
int *perspptr;
int perscalc2;
block colptr, hgtptr;
int scraddr;
int colstep;



 pixel_width = 320 / view_width;
 
 leftangle = (playerangle - 160);
 if (leftangle < 0)         
     leftangle += 1280;

 cloudcalc = cloudx;

 #ifdef VIRTUAL_SCREEN
   wsetscreen (virt);
 #else
   wnormscreen ();
 #endif


 for (y = 0; y < pixelcount; y++)
   perscalc[y] = 50 + ((elevation * persptable[y]) >> 12);


 for (x = 0; x < view_width; x++)
   {
    ax = playerx + (icos [leftangle] * eyedistance);    
    ay = playery + (isin [leftangle] * eyedistance);
    /* Get initial position of ray */   

    gxstep = xanglestep [leftangle];            /* Get ray step */
    gystep = yanglestep [leftangle];
  
    top_mountain = 199;

    tiltfactor = (tiltx * (x * pixel_width - 160)) >> 8;
    oldpos = 199;

    ydrawcoord = abuf + 63680L + x * pixel_width;
    perspptr = &persptable[1];


    for (y = 1; y < pixelcount; y++)
      {
       h_read_heightfield (ax, ay, ground_data, color_data);

       ax += gxstep;
       ay += gystep;
    
       perscalc2 = perscalc[y] + tiltfactor;
       newpos = perscalc2 - (((int)hgt * (int)(persptable[y])) >> 5);
  
       if ((newpos < oldpos) && (newpos < top_mountain))
         {
          if (oldpos > top_mountain)
            oldpos = top_mountain;
	
          if (newpos < 0)
            newpos = 0;
	
          linelength = oldpos - newpos;
	
          /* Draw each vertical line */
          if (linelength > 0)
            switch (draw_routine)
              {        
               case 0: ydrawcoord = wvlinel (ydrawcoord, linelength, col);
                       break;
               case 1: ydrawcoord = wvlinem (ydrawcoord, linelength, col);
                       break;
               case 2: ydrawcoord = wvlineh (ydrawcoord, linelength, col);
                       break;
              }
	
          top_mountain = newpos;
         }

      oldpos = newpos;
     }
   

   /* We need to draw the clouds */
   if ((top_mountain > linelength) && (top_mountain > 0))
     {
      linelength = top_mountain - linelength + 1;
 
      switch (draw_routine)
        {
         case 0: wclinel (ydrawcoord, cloudcalc + 
                         (top_mountain - tiltfactor + 39) * 320, top_mountain);
                 break;
         case 1: wclinem (ydrawcoord, cloudcalc + 
                         (top_mountain - tiltfactor + 39) * 320, top_mountain);
                 break;
         case 2: wclineh (ydrawcoord, cloudcalc + 
                         (top_mountain - tiltfactor + 39) * 320, top_mountain);
                 break;
        }
     }

    cloudcalc += pixel_width;
    if (cloudcalc > 320)
      cloudcalc -= 320;

    leftangle = (leftangle + pixel_width) % 1280;
    /* Advance to next angle */
   }
}



void h_calculate_distance_table (void)
{
int i;
int ax, ay, bx, by;
int xdiff, ydiff;

 for (i = 0; i < 1280; i++)
   {
    ax = icos [i] * eyedistance;
    ay = isin [i] * eyedistance;
    bx = icos [i] * RAYDISTANCE;
    by = isin [i] * RAYDISTANCE;
   
    xdiff = bx - ax;
    ydiff = by - ay;
    
    xanglestep[i] = (xdiff / pixelcount);
    yanglestep[i] = (ydiff / pixelcount);
   }
}



void rotate_player (int cenx, int ceny, int angle)
{
int x;   
int y;
int x2, y2;
int newx, newy;

 if (angle < 0)
   angle += 1280;
   
  x = (playerx - cenx);
  y = (playery - ceny);
   
  x2 = (x * icos[angle] - y * isin[angle]) >> 8;
  y2 = (x * isin[angle] + y * icos[angle]) >> 8;

  newx = x2;
  playerx = newx + cenx;
  newy = y2;
  playery = newy + ceny;
}





void process_keyboard (void)
{

 if (kbdon[KEY_LEFT])
   {
    rotate_player ((playerx + icos[playerangle] * 15),
                   (playery + isin[playerangle] * 15), -10);

    playerangle -= 10;          /* Update the viewer's angle */
    if (playerangle < 0)
      playerangle += 1280;

    cloudx -= 8;                /* Update the cloud position */
    if (cloudx < 0)
      cloudx += 320;

    if (tiltx < 64)             /* Tilt the screen */
      tiltx  += 8;
   }
 else if (kbdon[KEY_RIGHT])
   {
    rotate_player ( (playerx + icos[playerangle] * 15),
                    (playery + isin[playerangle] * 15), 10);

    playerangle += 10;          /* Update the viewer's angle */
    if (playerangle > 1279)
      playerangle -= 1280;

    cloudx += 8;                /* Update the cloud position */
    if (cloudx > 319)
      cloudx -= 320;

    if (tiltx > -64)            /* Tilt the screen */
      tiltx -= 8;

   }
 else
   {
    if (tiltx > 0)              /* Slowly remove tilt */
      tiltx -= 8;
    if (tiltx < 0)
      tiltx += 8;
   }

 if ((kbdon[KEY_UP]) && (playerspeed < 500))
   playerspeed += 40;
         
 if ((kbdon[KEY_DOWN]) && (playerspeed > -500))
   playerspeed -= 40;
	
 if (playerspeed > 0)   /* Add friction */
   playerspeed--;
 if (playerspeed < 0)
   playerspeed++;
	

 playerx += (icos[playerangle] * playerspeed) >> 8;
 playery += (isin[playerangle] * playerspeed) >> 8;
 /* Move the player based on angle and speed */



 if (kbdon[KEY_W])                      /* Increase height */
   {
    if (elevation < 14370)
      elevation += 5 * 60;  
   }

 if (kbdon[KEY_S])                      /* Decrease height */
   {
    if (elevation > 5)
      elevation -= 5 * 60;          
   }
	
 if (kbdon[KEY_E])
   pixelcount += 1;                     /* Increase length of ray */

 if (kbdon[KEY_D])                      /* Decrease length of ray */
   {
    pixelcount -= 1;
    if (pixelcount < 1)
      pixelcount = 1;
   }



 if (kbdon[KEY_R])
    eyedistance += 1;
 if (kbdon[KEY_F])
    eyedistance -= 1;



 /* Detail settings */
 if (kbdon[KEY_F1])  /* Low detail */
   {
    draw_routine = LOW_DETAIL;
    view_width = LOW_WIDTH;
   }
 if (kbdon[KEY_F2])  /* Medium detail */
   {
    draw_routine = MEDIUM_DETAIL;
    view_width = MEDIUM_WIDTH;
   }
 if (kbdon[KEY_F3])  /* High detail */
   {
    draw_routine = HIGH_DETAIL;
    view_width = HIGH_WIDTH;
   }
     
}



void main (void)
{
int i;

 printf ("Height field demo\n");
 printf ("Copyright 1995 Chris Egerter\n");
 printf ("Written with WGT 5.1 for Watcom C/C++\n");
 printf ("Released on September 29, 1995\n\n");
 printf ("Keys\n");
 printf ("ESC  - quits\n");
 printf ("W    - Increase height\n");
 printf ("S    - Decrease height\n");
 printf ("E    - Inc Range\n");
 printf ("D    - Dec Range\n");
 printf ("R    - Inc eye distance\n");
 printf ("F    - Dec eye distance\n");
 printf ("F1   - Low detail\n");
 printf ("F2   - Medium detail\n");
 printf ("F3   - High detail\n");
 printf ("Press any key to begin...\n");

 getch ();

 vga256 ();

 cloud_image = wloadpcx ("cloud.pcx", pal);
 ground_image = wloadpcx ("height.pcx", pal);
 color_image = wloadpcx ("ground.pcx",pal);
 ground_data = ground_image + 4;
 color_data = color_image + 4;
 wsetpalette (0, 255, pal);

 wnormscreen ();

 #ifdef VIRTUAL_SCREEN
   virt = wnewblock (0, 0, 319, 199);
 #endif

 h_create_sincos ();
 h_calculate_perspective ();
 h_calculate_distance_table ();

 playerx = 100L << 8;
 playery = 100L << 8;
 playerangle = 0;

 installkbd ();
   
 ticker_count = 0;
 frames = 0;

 winittimer();
 wstarttimer (timer_proc, TICKS(64));

 do
  {
   process_keyboard ();
   h_draw_heightfield ();
   frames++;

   #ifdef VIRTUAL_SCREEN
     wcopyscreen (0, 0, 319, 199, virt, 0, 0, NULL);
   #endif
  } while (!kbdon[1]);

 wstoptimer ();
 wdonetimer ();
 uninstallkbd ();

 wsetmode (3);   
 printf("Ticks: %lu\n", ticker_count);
 printf("Frames: %lu\n", frames);
 printf("   FPS: %3.2f\n", (double) frames / ((double)ticker_count / 64.0));
}

