///////////////////////////////////////////////
// Copyright
///////////////////////////////////////////////
//
// TextFX4
// Copyright (c) 1995-2001 Jari Komppa
//
//
///////////////////////////////////////////////
// License
///////////////////////////////////////////////
// 
//     This software is provided 'as-is', without any express or implied
//     warranty.    In no event will the authors be held liable for any damages
//     arising from the use of this software.
// 
//     Permission is granted to anyone to use this software for any purpose,
//     including commercial applications, and to alter it and redistribute it
//     freely, subject to the following restrictions:
// 
//     1. The origin of this software must not be misrepresented; you must not
//        claim that you wrote the original software. If you use this software
//        in a product, an acknowledgment in the product documentation would be
//        appreciated but is not required.
//     2. Altered source versions must be plainly marked as such, and must not be
//        misrepresented as being the original software.
//     3. This notice may not be removed or altered from any source distribution.
// 
// (eg. same as ZLIB license)
// 
//
///////////////////////////////////////////////

#include <stdlib.h>
#include <assert.h>
#include "textfx.h"





///////////////////////////////////////////////
// Compilation options
//


// comment out to use 'clean' (instead of real) truecolor palette
// for calculations 

//#define TFX_USE_REALIBMPAL


// Normally, build 1<<4, ie. 16x16x16 colormap.
// If you require bigger map, increase the value.
// (5 will mean 32x32x32 etc).
// Please note that processing requirements rise
// exponentially when this value is increased.
// Maximum value is 6.

#define TFX_COLORMAP_DEPTH 6

// Don't touch the rest of the defines. 

#define TFX_COLMAPDIM (1<<TFX_COLORMAP_DEPTH)
#define TFX_TRUCOLBITS (8 - TFX_COLORMAP_DEPTH)
#define TFX_COLMAP(r,g,b) *(mBlockMap + ((r)<<(TFX_COLORMAP_DEPTH * 2)) + ((g)<<TFX_COLORMAP_DEPTH) + (b))




///////////////////////////////////////////////
// Global data & variables
//

// The 9999,9999,9999 is the 'light black' color.
// For some reason NT console in fullscreen shows this color
// as color 7 (42,42,42) instead of the correct color (21,21,21)
// Thus, it is better not used here. 
// Value 9999 moves it far enough from the rgb cube.
// ('clean' palette version would be 32,32,32)

#ifdef TFX_USE_REALIBMPAL

static const int TFX_Palette[16*3]={ // IBM classic palette, 16c 
       0, 0, 0,  0, 0,42,  0,42, 0,  0,42,42, 42, 0, 0, 42, 0,42, 42,21, 0, 42,42,42,
9999,9999,9999, 21,21,63, 21,63,21, 21,63,63, 63,21,21, 63,21,63, 63,63,21, 63,63,63};

#else

static const int TFX_Palette[16*3]={ // 'clean' RGB palette 
       0, 0, 0,  0, 0,32,  0,32, 0,  0,32,32, 32, 0, 0, 32, 0,32, 32,32, 0, 32,32,32,
9999,9999,9999,  0, 0,63,  0,63, 0,  0,63,63, 63, 0, 0, 63, 0,63, 63,63, 0, 63,63,63};

#endif




///////////////////////////////////////////////
// Functions
//


TFX_BlockColor::TFX_BlockColor()
{
    mBlockMap=NULL;
}

TFX_BlockColor::~TFX_BlockColor()
{
    delete[] mBlockMap;
}


#define exp2(a) ((a)*(a))

short int TFX_BlockColor::CalcColor(int red, int green, int blue) 
{
    int a,b,c,d,ch = 0,co = 0;
    int lastdist, dist;
  
    lastdist = 0x7fffffff;

    for (c = 0, d = 0; c < 16; c++, d += 3) 
    {
        dist = exp2(TFX_Palette[d + 0] - red) +
               exp2(TFX_Palette[d + 1] - green) +
               exp2(TFX_Palette[d + 2] - blue);
        if (dist < lastdist) 
        {
            lastdist = dist;
            co = c;
            ch = 219; // 100% block in IBMSCII 
        }
    }
    c = co;
    d = c*3;
    for (b=0,a=0;b<16;b++,a+=3) 
    {
        dist = exp2(((TFX_Palette[a + 0] + TFX_Palette[d + 0]) / 2) - red) +
               exp2(((TFX_Palette[a + 1] + TFX_Palette[d + 1]) / 2) - green) +
               exp2(((TFX_Palette[a + 2] + TFX_Palette[d + 2]) / 2) - blue);
    if (dist < lastdist) 
    {
        lastdist = dist;
        if (b > c)
            co = b + (c<<4);
        else
            co = c + (b<<4);
        ch = 177; // 50% block in IBMSCII 
    }
    dist = exp2((TFX_Palette[a + 0] * 3 / 4 + TFX_Palette[d + 0] / 4) - red) +
           exp2((TFX_Palette[a + 1] * 3 / 4 + TFX_Palette[d + 1] / 4) - green) +
           exp2((TFX_Palette[a + 2] * 3 / 4 + TFX_Palette[d + 2] / 4) - blue);
    if (dist < lastdist) 
    {
        lastdist = dist;
        if (b > c)
        {
            co = b + (c<<4);
            ch = 178; // 75% block in IBMSCII 
        }
        else
        {
            co = c + (b<<4);
            ch = 176; // 25% block in IBMSCII 
        }
    }
    dist = exp2((TFX_Palette[a + 0] / 4 + TFX_Palette[d + 0] * 3 / 4) - red) +
           exp2((TFX_Palette[a + 1] / 4 + TFX_Palette[d + 1] * 3 / 4) - green) +
           exp2((TFX_Palette[a + 2] / 4 + TFX_Palette[d + 2] * 3 / 4) - blue);
    if (dist < lastdist) 
    {
      lastdist = dist;
      if (c > b)
      {
          co = c + (b<<4);
          ch = 178; // 75% block in IBMSCII 
      }
      else
      {
          co = b + (c<<4);
          ch = 176; // 25% block in IBMSCII 
      }
    }
  }
  return (short)((co<<8) + ch);
}



void TFX_BlockColor::BuildLUT(int aFlags) 
{
    aFlags=aFlags;
    int r,g,b;
    int f;
    if (mBlockMap != NULL) 
        delete[] mBlockMap;
    f = 64 / TFX_COLMAPDIM;
    mBlockMap = new short[TFX_COLMAPDIM*TFX_COLMAPDIM*TFX_COLMAPDIM];
    for (r = 0; r < TFX_COLMAPDIM; r++) 
    {
        for (g = 0; g < TFX_COLMAPDIM; g++)
        {
            for (b = 0; b < TFX_COLMAPDIM; b++)
            {
                TFX_COLMAP(r,g,b) = CalcColor(r*f,g*f,b*f);
            }
        }
    }
}



void TFX_BlockColor::Dump1x(int *aSource, TFXQuad &aSrcQuad, int aSrcPitch, int aTgtX0, int aTgtY0, short *aTarget)
{
    assert(mBlockMap!=NULL);
    int xsize = aSrcQuad.x1 - aSrcQuad.x0;
    int ysize = aSrcQuad.y1 - aSrcQuad.y0;
    int sourceyofs = aSrcQuad.y0 * aSrcPitch;

    for (int y = 0; y < ysize; y++, sourceyofs+=aSrcPitch)
    {
        for (int x = 0, targetpos = (aTgtY0 + y) * TFX_ConsoleWidth + aTgtX0, sourcepos = sourceyofs + aSrcQuad.x0; x < xsize; x++, targetpos++, sourcepos++)
        {
            int color = aSource[sourcepos];
            aTarget[targetpos] = TFX_COLMAP(((color    ) & 0xff)>>TFX_TRUCOLBITS,
                                            ((color>> 8) & 0xff)>>TFX_TRUCOLBITS,
                                            ((color>>16) & 0xff)>>TFX_TRUCOLBITS);
        }
    }
}



void TFX_BlockColor::Dump2x(int *aSource, TFXQuad &aSrcQuad, int aSrcPitch, int aTgtX0, int aTgtY0, short *aTarget)
{
    assert(mBlockMap!=NULL);
    int xsize = (aSrcQuad.x1 - aSrcQuad.x0) / 2;
    int ysize = (aSrcQuad.y1 - aSrcQuad.y0) / 2;
    int sourceyofs = aSrcQuad.y0 * aSrcPitch;

    for (int y = 0; y < ysize; y++, sourceyofs += aSrcPitch * 2)
    {
        for (int x = 0, targetpos=(aTgtY0 + y) * TFX_ConsoleWidth + aTgtX0, sourcepos = sourceyofs + aSrcQuad.x0; x < xsize; x++, targetpos++, sourcepos += 2)
        {
            int redblue;
            int green;
            redblue  = aSource[sourcepos] & 0xff00ff;
            green    = aSource[sourcepos] & 0x00ff00;
            redblue += aSource[sourcepos + 1] & 0xff00ff;
            green   += aSource[sourcepos + 1] & 0x00ff00;
            redblue += aSource[sourcepos + aSrcPitch] & 0xff00ff;
            green   += aSource[sourcepos + aSrcPitch] & 0x00ff00;
            redblue += aSource[sourcepos + aSrcPitch + 1] & 0xff00ff;
            green   += aSource[sourcepos + aSrcPitch + 1] & 0x00ff00;
            int color = ((redblue>>2) & 0xff00ff) + ((green>>2) & 0x00ff00);

            aTarget[targetpos] = TFX_COLMAP(((color    ) & 0xff)>>TFX_TRUCOLBITS,
                                            ((color>> 8) & 0xff)>>TFX_TRUCOLBITS,
                                            ((color>>16) & 0xff)>>TFX_TRUCOLBITS);
        }
    }
}



void TFX_BlockColor::Dump4x(int *aSource, TFXQuad &aSrcQuad, int aSrcPitch, int aTgtX0, int aTgtY0, short *aTarget)
{
    
    assert(mBlockMap!=NULL);
    int xsize = (aSrcQuad.x1 - aSrcQuad.x0) / 4;
    int ysize = (aSrcQuad.y1 - aSrcQuad.y0) / 4;
    int sourceyofs = aSrcQuad.y0 * aSrcPitch;

    for (int y = 0; y < ysize; y++, sourceyofs += aSrcPitch * 4)
    {
        for (int x = 0, targetpos = (aTgtY0 + y) * TFX_ConsoleWidth + aTgtX0, sourcepos = sourceyofs + aSrcQuad.x0; x < xsize; x++, targetpos++, sourcepos += 4)
        {
            int redblue = 0;
            int green = 0;
            for (int iny = 0, inyp = sourcepos; iny < 4; iny++, inyp += aSrcPitch)
            {
                for (int inx = 0, inpos = inyp; inx < 4; inx++, inpos++)
                {
                    redblue += aSource[inpos] & 0xff00ff;
                    green   += aSource[inpos] & 0x00ff00;
                }
            }

            int color = ((redblue>>4) & 0xff00ff) + ((green>>4) & 0x00ff00);

            aTarget[targetpos] = TFX_COLMAP(((color    ) & 0xff)>>TFX_TRUCOLBITS,
                                            ((color>> 8) & 0xff)>>TFX_TRUCOLBITS,
                                            ((color>>16) & 0xff)>>TFX_TRUCOLBITS);
        }
    }
}
