//
// TinyPTC by Gaffer
// www.gaffer.org/tinyptc
//

#include "tinyptc.h"
#define WIN32_LEAN_AND_MEAN
#include <ddraw.h>

static HMODULE library = 0;
static LPDIRECTDRAW lpDD = 0;
static LPDIRECTDRAWSURFACE lpDDS = 0;
static LPDIRECTDRAWSURFACE lpDDS_back;
static WNDCLASS wc;
static HWND wnd;
static int active;
static int dx;
static int dy;

#ifdef __PTC_WINDOWED__
static LPDIRECTDRAWCLIPPER lpDDC = 0;
static LPDIRECTDRAWSURFACE lpDDS_secondary = 0;
#endif

typedef HRESULT (WINAPI * DIRECTDRAWCREATE) (GUID FAR *lpGUID,LPDIRECTDRAW FAR *lplpDD,IUnknown FAR *pUnkOuter);


#ifdef __PTC_WINDOWED__

static void ptc_paint_primary()
{
    RECT source;
    RECT destination;
    POINT point;

    // check
    if (lpDDS)
    {
        // setup source rectangle
        source.left = 0;
        source.top = 0;
        source.right = dx;
        source.bottom = dy;

        // get origin of client area
        point.x = 0;
        point.y = 0;
        ClientToScreen(wnd,&point);

        // get window client area
        GetClientRect(wnd,&destination);

        // offset destination rectangle
        destination.left += point.x;
        destination.top += point.y;
        destination.right += point.x;
        destination.bottom += point.y;

        // blt secondary to primary surface
        lpDDS->lpVtbl->Blt(lpDDS,&destination,lpDDS_secondary,&source,DDBLT_WAIT,0);
    }
}

#endif


static LRESULT CALLBACK WndProc(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam)
{
    // result data
    int result = 0;

    // handle message
    switch (message)
    {
        #ifdef __PTC_WINDOWED__

        case WM_PAINT:
        {
            // paint primary
            ptc_paint_primary();

            // call default window painting
            return DefWindowProc(hWnd,message,wParam,lParam);
        }
        break;

        #else

        case WM_ACTIVATEAPP:
        {                                       
            // update active flag
            active = (BOOL) wParam;
        }
        break;

        case WM_SETCURSOR:
        {
            // hide cursor
            SetCursor(0);
        }
        break;

        #endif

        case WM_KEYDOWN:
        {
            // close on escape key
            if ((wParam&0xFF)!=27) break;
        }

        case WM_CLOSE:
        {
            // close ptc
            ptc_close();

            // exit process
            ExitProcess(0);
        }
        break;

        default:
        {
            // unhandled messages
            result = DefWindowProc(hWnd,message,wParam,lParam);
        }
    }

    // finished
    return result;
}


int ptc_open(char *title,int width,int height)
{
    #ifdef __PTC_WINDOWED__
    int x;
    int y;
    RECT rect;
    #else
    DDSCAPS capabilities;
    #endif
    DDSURFACEDESC descriptor;
    DIRECTDRAWCREATE DirectDrawCreate;

    // setup data
    dx = width;
    dy = height;

    // load direct draw library
    library = (HMODULE) LoadLibrary("ddraw.dll");
    if (!library) return 0;

    // get directdraw create function address
    DirectDrawCreate = (DIRECTDRAWCREATE) GetProcAddress(library,"DirectDrawCreate");
    if (!DirectDrawCreate) return 0;

    // create directdraw interface
    if (DirectDrawCreate(0,&lpDD,0)!=DD_OK) return 0;

#ifndef __PTC_WINDOWED__

    // register window class
    wc.style = 0;
    wc.lpfnWndProc = WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
#ifdef __PTC_ICON__
    wc.hInstance = GetModuleHandle(0);
    wc.hIcon = LoadIcon(wc.hInstance,__PTC_ICON__);
#else
    wc.hInstance = 0;
    wc.hIcon = 0;
#endif
    wc.hCursor = 0;
    wc.hbrBackground = 0;
    wc.lpszMenuName = 0;
    wc.lpszClassName = title;
    RegisterClass(&wc);

    // create window
#ifdef __PTC_ICON__
    wnd = CreateWindow(title,title,WS_POPUP | WS_SYSMENU,0,0,0,0,0,0,0,0);
#else
    wnd = CreateWindow(title,title,WS_POPUP,0,0,0,0,0,0,0,0);
#endif

    // enter exclusive mode
    if (lpDD->lpVtbl->SetCooperativeLevel(lpDD,wnd,DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN)!=DD_OK) return 0;

    // enter display mode
    if (lpDD->lpVtbl->SetDisplayMode(lpDD,width,height,32)!=DD_OK) return 0;

    // primary with two back buffers
    descriptor.dwSize  = sizeof(descriptor);
    descriptor.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    descriptor.dwBackBufferCount = 2;
    descriptor.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_VIDEOMEMORY | DDSCAPS_COMPLEX | DDSCAPS_FLIP;
    if (lpDD->lpVtbl->CreateSurface(lpDD,&descriptor,&lpDDS,0)!=DD_OK)
    {
        // primary with one back buffer
        descriptor.dwBackBufferCount = 1;
        if (lpDD->lpVtbl->CreateSurface(lpDD,&descriptor,&lpDDS,0)!=DD_OK)
        {
            // primary with no back buffers
            descriptor.dwFlags = DDSD_CAPS;
            descriptor.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_VIDEOMEMORY;
            if (lpDD->lpVtbl->CreateSurface(lpDD,&descriptor,&lpDDS,0)!=DD_OK)
            {
                // failure
                return 0;
            }
        }
    }

    // get back surface
    capabilities.dwCaps = DDSCAPS_BACKBUFFER;
    if (lpDDS->lpVtbl->GetAttachedSurface(lpDDS,&capabilities,&lpDDS_back)!=DD_OK) return 0;

#else

    // register window class
    wc.style = CS_VREDRAW | CS_HREDRAW;
    wc.lpfnWndProc = WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
#ifdef __PTC_ICON__
    wc.hInstance = GetModuleHandle(0);
    wc.hIcon = LoadIcon(wc.hInstance,__PTC_ICON__);
#else
    wc.hInstance = 0;
    wc.hIcon = 0;
#endif
    wc.hCursor = LoadCursor(0,IDC_ARROW);
    wc.hbrBackground = 0;
    wc.lpszMenuName = 0;
    wc.lpszClassName = title;
    RegisterClass(&wc);

    // calculate window size
    rect.left = 0;
    rect.top = 0;
    rect.right = width;
    rect.bottom = height;
    AdjustWindowRect(&rect,WS_OVERLAPPEDWINDOW,0);
    rect.right -= rect.left;
    rect.bottom -= rect.top;

#ifdef __PTC_CENTER_WINDOW__

    // center window
    x = (GetSystemMetrics(SM_CXSCREEN) - rect.right) >> 1;
    y = (GetSystemMetrics(SM_CYSCREEN) - rect.bottom) >> 1;

#else

    // let windows decide
    x = CW_USEDEFAULT;
    y = CW_USEDEFAULT;

#endif

    // create window
    wnd = CreateWindow(title,title,WS_OVERLAPPEDWINDOW,x,y,rect.right,rect.bottom,0,0,0,0);

    // show window
    ShowWindow(wnd,SW_NORMAL);

    // enter cooperative mode
    if (lpDD->lpVtbl->SetCooperativeLevel(lpDD,wnd,DDSCL_NORMAL)!=DD_OK) return 0;

    // primary with no back buffers
    descriptor.dwSize  = sizeof(descriptor);
    descriptor.dwFlags = DDSD_CAPS;
    descriptor.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_VIDEOMEMORY;
    if (lpDD->lpVtbl->CreateSurface(lpDD,&descriptor,&lpDDS,0)!=DD_OK) return 0;

    // create secondary surface
    descriptor.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
    descriptor.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
    descriptor.dwWidth = width;
    descriptor.dwHeight = height;
    if (lpDD->lpVtbl->CreateSurface(lpDD,&descriptor,&lpDDS_secondary,0)!=DD_OK) return 0;
    
    // create clipper
    if (lpDD->lpVtbl->CreateClipper(lpDD,0,&lpDDC,0)!=DD_OK) return 0;

    // set clipper to window
    if (lpDDC->lpVtbl->SetHWnd(lpDDC,0,wnd)!=DD_OK) return 0;

    // attach clipper object to primary surface
    if (lpDDS->lpVtbl->SetClipper(lpDDS,lpDDC)!=DD_OK) return 0;
    
    // set back to secondary
    lpDDS_back = lpDDS_secondary;

#endif

    // setup converter

    // success
    return 1;
}


void convert(void *src,void *dst,int pixels)
{
    // copy conversion
    int i;
    int32 *p;
    int32 *q;
    p = (int32*) src;
    q = (int32*) dst;
    for (i=0; i<pixels; i++) q[i] = p[i];
}


int ptc_update(void *buffer)
{
    int y;
    char8 *src;
    char8 *dst;
    int src_pitch;
    int dst_pitch;
    MSG message;
    DDSURFACEDESC descriptor;

    // process messages
    while (PeekMessage(&message,wnd,0,0,PM_REMOVE))
    {
        // translate and dispatch
        TranslateMessage(&message);
        DispatchMessage(&message);
    }

    #ifndef __PTC_WINDOWED__
    if (active)
    #endif
    {
        // restore surfaces
        lpDDS->lpVtbl->Restore(lpDDS);
        #ifdef __PTC_WINDOWED__
        lpDDS_secondary->lpVtbl->Restore(lpDDS_secondary);
        #endif

        // lock back surface
        descriptor.dwSize = sizeof descriptor;
        if (lpDDS_back->lpVtbl->Lock(lpDDS_back,0,&descriptor,DDLOCK_WAIT,0)!=DD_OK) return 0;
    
        // calculate pitches
        src_pitch = dx * 4;
        dst_pitch = descriptor.lPitch;

        // copy pixels to back surface
        src = (char8*) buffer;
        dst = (char8*) descriptor.lpSurface;
        for (y=0; y<dy; y++)
        {
            // convert line
            convert(src,dst,dx);
            src += src_pitch;
            dst += dst_pitch;
        }

        // unlock back surface
        lpDDS_back->lpVtbl->Unlock(lpDDS_back,descriptor.lpSurface);

        #ifndef __PTC_WINDOWED__
    
            // flip primary surface
            lpDDS->lpVtbl->Flip(lpDDS,0,DDFLIP_WAIT);

        #else

            // paint primary
            ptc_paint_primary();

        #endif
    }

    // sleep
    Sleep(1);
    
    // success
    return 1;
}


void ptc_close()
{
#ifdef __PTC_WINDOWED__

    // check secondary
    if (lpDDS_secondary)
    {
        // release secondary
        lpDDS_secondary->lpVtbl->Release(lpDDS_secondary);
        lpDDS_secondary = 0;
    }

#endif

    // check
    if (lpDDS)
    {
        // release primary
        lpDDS->lpVtbl->Release(lpDDS);
        lpDDS = 0;
    }

    // check
    if (lpDD)
    {
        // leave display mode
        lpDD->lpVtbl->RestoreDisplayMode(lpDD);

        // leave exclusive mode
        lpDD->lpVtbl->SetCooperativeLevel(lpDD,wnd,DDSCL_NORMAL);

        // free direct draw
        lpDD->lpVtbl->Release(lpDD);
        lpDD = 0;
    }

    // check
    if (library)
    {
        // free library
        FreeLibrary(library);
        library = 0;
    }
}
