
#include "music.h"

#include "playvtal.h"

#include <llkey.h>

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>

static PLYV_PRec MusicHandle = NULL;
static XTRN_TStack MusicStack = { 0, 0};

static bool DetectBlaster(int *port, int *irq, int *dma, int *hdma, int *type) {
    const char *p;
    if ( (p = getenv("BLASTER")) == NULL)
        return FALSE;

    while (*p != '\0') {
        if (isalpha(*p)) {
            switch(toupper(*p)) {
                case 'A': if (port != NULL) sscanf(p+1, "%x", port); break;
                case 'I': if (irq != NULL)  sscanf(p+1, "%d", irq); break;
                case 'D': if (dma != NULL)  sscanf(p+1, "%d", dma); break;
                case 'H': if (hdma != NULL)  sscanf(p+1, "%d", hdma); break;
                case 'T': if (type != NULL)  sscanf(p+1, "%d", type); break;
            }
            while (*p != '\0' && *p != ' ')
                p++;
        } else
            p++;
    }
    return TRUE;
}

static bool DetectGUS(int *port, int *irq, int *dma) {
    const char *p;
    if ( (p = getenv("ULTRASND")) == NULL)
        return FALSE;

    if (port != NULL)
        sscanf(p, "%x", port);
    while (*p != '\0' && *p != ',') p++;
    if (*p == '\0') return FALSE;
    p++;    // Skip comma.
    if (dma!= NULL)
        sscanf(p, "%d", dma);
    while (*p != '\0' && *p != ',') p++;
    if (*p == '\0') return FALSE;
    p++;    // Skip comma.
    while (*p != '\0' && *p != ',') p++;
    if (*p == '\0') return FALSE;
    p++;    // Skip comma.
    if (irq != NULL)
        sscanf(p, "%d", irq);
    return TRUE;
}

extern void InitMusicSystem(void) {
    int port = -1, irq = -1, dma = -1, hdma = -1, type = 3;
    static PLYV_TSetup s;

    REQUIRE(XTRN_InitStack(&MusicStack, 8192));

/*
    if (BASE_CheckArg("nosound") > 0) {
        printf("Forced no music.\n");
        return;
    }
*/
    if (DetectGUS(&port, &irq, &dma)) {
        strcpy(s.MusicDevice, "GUS.DEV");
        printf("ULTRASND settings: port %Xh, IRQ %d, DMA %d.\n",
               port, irq, dma);
    } else if (DetectBlaster(&port, &irq, &dma, &hdma, &type)) {
        int i;
        if ( (i = BASE_CheckArg("type")) > 0)
            type = atoi(ArgV[i]);
        if (type >= 6 && hdma >= 0) {
            strcpy(s.MusicDevice, "SB_16.DEV");
            dma = hdma;
        } else if (type < 2) {
            strcpy(s.MusicDevice, "SB_1_5.DEV");
        } else if (type == 2 || type == 4 || type == 5) {
            strcpy(s.MusicDevice, "SB_PRO.DEV");
        } else {
            strcpy(s.MusicDevice, "SB_2_0.DEV");
        }
        printf("BLASTER settings: port %Xh, IRQ %d, DMA %d, card type %d.\n",
               port, irq, dma, type);
        printf("Type #'s are: 1: SB 1.5, 3: SB 2.0, 2,4,5: SB Pro, above: SB 16, AWE, etc.\n"
               " If you find any problems, please supply a correct -type # parameter.\n"
               "\n");
    } else {
        BASE_Abort("ULTRASND and BLASTER variables not found or incorrect.\n"
                   "  You need a soundcard to enjoy this demo");
    }
    s.MusicConfig.port1 = port;
    s.MusicConfig.irq1  = irq;
    s.MusicConfig.dma1  = dma;
    s.MusicConfig.dma2  = dma;
    s.MusicConfig.tickrate = 70*256;
    s.MusicConfig.rate      = (sint16)44000;
    s.MusicConfig.maxchans  = 16;
    s.MusicConfig.stereo    =  TRUE;
    s.MusicConfig.bits16    =  TRUE;
    s.MusicConfig.panning   =  TRUE;
    s.MusicConfig.surround  =  TRUE;
    s.MusicParams.Period    = 65536;
    s.MusicParams.Volume    =   256;
    s.MusicParams.PanWidth  =    60;
    s.MusicParams.PanOffs   =     0;

    printf("VTAL Music device selected: %s\n", //, port 0x%X, irq %d, DMA %d\n",
            s.MusicDevice //,
//            s.MusicConfig.port1,
//            s.MusicConfig.irq1,
//            s.MusicConfig.dma1
          );
    MusicHandle = PLYV_Init(&s);
    if (MusicHandle == NULL) {
        BASE_Abort("*** Error initializing VTAL sound system! ***\n"
                   "  Check your ULTRASND or BLASTER settings.");
    }
/*
    printf("Music system data area = 0x%X, Music driver handle = 0x%X\n",
        MusicHandle,
        (MusicHandle != NULL)?MusicHandle->MusicDev:NULL);
*/
    PLYV_SetMode(MusicHandle, PLYV_POLL);
}

extern void DoneMusicSystem(void) {
    PLYV_SetMode(MusicHandle, PLYV_POLL);
    PLYV_Done(MusicHandle);
    XTRN_DoneStack(&MusicStack);
}

static char SongName[300] = "12345678.123";

extern void InitMusic(const char *fname) {
    static PLAY_TInitData pid;

    if (MusicHandle == NULL) {
        return;
    }

    if (fname == NULL || strncmp(fname, SongName, sizeof(SongName)) != 0) {
        if (fname != NULL)
            strncpy(SongName, fname, sizeof(SongName));
        else
            SongName[0] = '\0';

            // Wait for previous song to end its fade.
        if (MusicHandle != NULL && MusicHandle->SongPlaying && MusicHandle->MusicDev != NULL)
            while (MusicHandle->MusParams.Volume > 0 && LLK_LastScan != kESC)
                PLYV_Poll(MusicHandle, 20);   // Time up to 20 ticks (don't allow music to stop).

        if (fname == NULL) {
            PLYV_MUS_Stop(MusicHandle);
            PLYV_MUS_Unload(MusicHandle);
            return;
        }

//        printf("Loading tune %s\n", fname);
        pid.LoopMod            = TRUE;
        pid.ForceLoopMod       = TRUE;
        pid.PermitFilterChange = TRUE;
        pid.FirstPattern       = 0;
        pid.RepStart           = 0;
        pid.SongLen            = 0;
        pid.RealTimerVal       = 0;

        PLYV_MUS_Load(MusicHandle, fname);
        PLYV_MUS_Start(MusicHandle, &pid);
    }
    PLYV_MUS_ChangeVolume(MusicHandle, 256);
    PLYV_MUS_SetFading(MusicHandle, PLYV_FADEIN, 1);
    PLYV_Poll(MusicHandle, 20);   // Time up to 20 ticks (don't allow music to stop).
}

extern void DoneMusic(void) {
    if (MusicHandle == NULL)
        return;
    PLYV_MUS_SetFading(MusicHandle, PLYV_FADEOUT, 2*70);
}

extern void PollMusic(void) {
    if (MusicHandle == NULL)
        return;
    PLYV_Poll(MusicHandle, 2);
}

extern void CallMusicHandle(void) {
    XTRN_StackExec(&MusicStack, PollMusic);
}

extern bool IsMusicPlaying(void) {
    if (MusicHandle != NULL && MusicHandle->SongPlaying && MusicHandle->MusicDev != NULL)
        if (MusicHandle->MusParams.Volume == 0)
            return FALSE;
    return TRUE;
}

extern bool IsThereMusic(void) {
    if (MusicHandle != NULL && MusicHandle->SongPlaying && MusicHandle->MusicDev != NULL)
        return TRUE;
    return FALSE;
}

extern int GetMusicPattern(void) {
    if (MusicHandle != NULL && MusicHandle->SongPlaying && MusicHandle->MusicDev != NULL)
        return MusicHandle->PlayInfo->NextSeq;
    return -1;
}

extern int GetMusicPos(void) {
    if (MusicHandle != NULL && MusicHandle->SongPlaying && MusicHandle->MusicDev != NULL)
        return MusicHandle->PlayInfo->NextNote;
    return -1;
}

extern void SetMusicPattern(int p) {
    if (MusicHandle != NULL && MusicHandle->SongPlaying && MusicHandle->MusicDev != NULL)
        MusicHandle->PlayInfo->NextSeq = p;
}

extern void SetMusicPos(int p) {
    if (MusicHandle != NULL && MusicHandle->SongPlaying && MusicHandle->MusicDev != NULL)
        MusicHandle->PlayInfo->NextNote = p;
}

