#include "vesavbe.hh"
#include <malloc.h>
#include <stdio.h>

static DPMI_PTR VbeInfoPool = {0,0};
static DPMI_PTR VbeModePool = {0,0};

static struct rminfo {
  long EDI;
  long ESI;
  long EBP;
  long reserved_by_system;
  long EBX;
  long EDX;
  long ECX;
  long EAX;
  short flags;
  short ES,DS,FS,GS,IP,CS,SP,SS;
} RMI;

static union REGS regs;
static struct SREGS sregs;

void VBE_Init (void)
{
  DPMI_AllocDOSMem (512/16, &VbeInfoPool);
  DPMI_AllocDOSMem (256/16, &VbeModePool);
}

void VBE_Done (void)
{
  DPMI_FreeDOSMem (&VbeModePool);
  DPMI_FreeDOSMem (&VbeInfoPool);
}

static void PrepareInfoBlock (VBE_VbeInfoBlock &VBEInfo)
{
  memset (&VBEInfo, 0, sizeof (VBEInfo));
  strncpy (VBEInfo.vbeSignature, "VBE2", 4);
}

static void PrepareRegisters (void)
{
  memset(&RMI,0,sizeof(RMI));
  memset(&sregs,0,sizeof(sregs));
  memset(&regs,0,sizeof(regs));
}

void DPMI_AllocDOSMem (short int paras, DPMI_PTR *p)
{
  /* DPMI call 100h allocates DOS memory */
  PrepareRegisters();
  regs.w.ax=0x0100;
  regs.w.bx=paras;
  int386x( 0x31, &regs, &regs, &sregs);
  p->segment=regs.w.ax;
  p->selector=regs.w.dx;
}

void DPMI_FreeDOSMem (DPMI_PTR *p)
{
  /* DPMI call 101h free DOS memory */
  memset(&sregs,0,sizeof(sregs));
  regs.w.ax=0x0101;
  regs.w.dx=p->selector;
  int386x( 0x31, &regs, &regs, &sregs);
}

void * DPMI_MAP_PHYSICAL (void *p, long size)
{
  /* DPMI call 101h free DOS memory */
  PrepareRegisters();
  regs.w.ax=0x0800;
  regs.w.bx=((long)p)>>16;
  regs.w.cx=((long)p)&0xffff;
  regs.w.si=size>>16;
  regs.w.di=size&0xffff;
  int386x( 0x31, &regs, &regs, &sregs);
  return ((void *) ((regs.w.bx << 16) + regs.w.cx));
}

void VBE_Mode_Information (short Mode, VBE_ModeInfoBlock __far * a)
{
  PrepareRegisters();
  RMI.EAX=0x00004f01;               // Get SVGA-Mode Information
  RMI.ECX=Mode;
  RMI.ES=VbeModePool.segment;           // Segment of realmode data
  RMI.EDI=0;                        // offset of realmode data

  regs.w.ax = 0x0300;               // Simulate Real-Mode interrupt
  regs.h.bl = 0x10;                 // Call int 10 from realmode
  regs.h.bh = 0;
  regs.w.cx = 0;
  sregs.es = FP_SEG(&RMI);          // Ptr to data
  regs.x.edi = FP_OFF(&RMI);
  int386x(0x31, &regs, &regs, &sregs );
  _fmemcpy (a, MK_FP (VbeModePool.selector, 0), sizeof (VBE_ModeInfoBlock));
}


int VBE_IsModeLinear (short Mode)
{
  VBE_ModeInfoBlock a;
  VBE_Mode_Information (Mode, &a);
  return ((a.ModeAttributes & 128)==128);
}


void VBE_SetDisplayStart (short x, short y)
{
  PrepareRegisters();
  RMI.EAX=0x00004f07;
  RMI.EBX=0;
  RMI.ECX=x;
  RMI.EDX=y;
  regs.w.ax = 0x0300;               // Simulate Real-Mode interrupt
  regs.h.bl = 0x10;                 // Call int 10 from realmode
  regs.h.bh = 0;
  regs.w.cx = 0;
  sregs.es = FP_SEG(&RMI);          // Ptr to data
  regs.x.edi = FP_OFF(&RMI);
  int386x(0x31, &regs, &regs, &sregs );
}

void VBE_SetMode (short Mode, int linear, int clear)
{
  int rawmode = Mode & 0x01ff;
  if ( linear) rawmode |= 1<<14;
  if (!clear)  rawmode |= 1<<15;
  PrepareRegisters();
  RMI.EAX=0x00004f02;               // Get SVGA-Mode Information
  RMI.EBX=rawmode;
  regs.w.ax = 0x0300;               // Simulate Real-Mode interrupt
  regs.h.bl = 0x10;                 // Call int 10 from realmode
  regs.h.bh = 0;
  regs.w.cx = 0;
  sregs.es = FP_SEG(&RMI);          // Ptr to data
  regs.x.edi = FP_OFF(&RMI);
  int386x(0x31, &regs, &regs, &sregs );
}

void VBE_Controller_Information  (VBE_VbeInfoBlock __far * a)
{
  PrepareRegisters();
  _fmemcpy (MK_FP (VbeInfoPool.selector, 0), a, 512);
  RMI.EAX=0x00004f00;               // Get SVGA-Information
  RMI.ES=VbeInfoPool.segment;              // Segment of realmode data
  RMI.EDI=0;                        // offset of realmode data

  regs.w.ax = 0x0300;               // Simulate Real-Mode interrupt
  regs.h.bl = 0x10;                 // Call int 10 from realmode
  regs.h.bh = 0;
  regs.w.cx = 0;
  sregs.es = FP_SEG(&RMI);          // Ptr to data
  regs.x.edi = FP_OFF(&RMI);
  int386x(0x31, &regs, &regs, &sregs );
  _fmemcpy (a, MK_FP (VbeInfoPool.selector, 0), 512);

  // Jetzt neu: Direktes Konvertieren der Realmode Pointer zu echten pointern
  a->OemStringPtr=(char*)((((unsigned long)a->OemStringPtr>>16)<<4)+(unsigned short)a->OemStringPtr);
  a->VideoModePtr=(unsigned short*)((((unsigned long)a->VideoModePtr>>16)<<4)+(unsigned short)a->VideoModePtr);
  a->OemVendorNamePtr=(char*)((((unsigned long)a->OemVendorNamePtr>>16)<<4)+(unsigned short)a->OemVendorNamePtr);
  a->OemProductNamePtr=(char*)((((unsigned long)a->OemProductNamePtr>>16)<<4)+(unsigned short)a->OemProductNamePtr);
  a->OemProductRevPtr=(char*)((((unsigned long)a->OemProductRevPtr>>16)<<4)+(unsigned short)a->OemProductRevPtr);
}

int VBE_Test (void)
{
  VBE_VbeInfoBlock VBEInfo;
  PrepareInfoBlock (VBEInfo);
  VBE_Controller_Information (&VBEInfo);
  return (VBEInfo.vbeVersion>=0x200);
}

unsigned int VBE_VideoMemory (void)
{
  VBE_VbeInfoBlock VBEInfo;
  PrepareInfoBlock (VBEInfo);
  VBE_Controller_Information (&VBEInfo);
  return (VBEInfo.TotalMemory*1024*64);
}

int VBE_FindMode (int xres, int yres, char bpp)
{
  VBE_VbeInfoBlock VBEInfo;
  PrepareInfoBlock (VBEInfo);
  VBE_Controller_Information (&VBEInfo);
  // First try to find the mode in the ControllerInfoBlock:
  for (int i=0; VBEInfo.VideoModePtr[i]!=0xffff; i++ ) {
    VBE_ModeInfoBlock Info;
    VBE_Mode_Information (VBEInfo.VideoModePtr[i], &Info);
    if ((xres == Info.XResolution) &&
        (yres == Info.YResolution) &&
        (bpp == Info.BitsPerPixel)) return VBEInfo.VideoModePtr[i];
  }
  return -1;
}

char * VBE_GetVideoPtr (int mode)
{
  VBE_VbeInfoBlock VBEInfo;
  PrepareInfoBlock (VBEInfo);
  VBE_Controller_Information (&VBEInfo);
  VBE_ModeInfoBlock ModeInfo;
  VBE_Mode_Information (mode, &ModeInfo);
  return (char *) DPMI_MAP_PHYSICAL (ModeInfo.PhysBasePtr, (VBEInfo.TotalMemory*64*1024));
}

static vbelastbank = -1;
void VBE_SetBank (short bnk)
{
  if ( bnk==vbelastbank ) return;
  memset(&RMI,0,sizeof(RMI));       // Initialize the VESA-Data
  memset(&sregs,0,sizeof(sregs));
  memset(&regs,0,sizeof(regs));
  RMI.EAX=0x00004f05;
  RMI.EBX=0;
  RMI.EDX=bnk;
  regs.w.ax = 0x0300;               // Simulate Real-Mode interrupt
  regs.h.bl = 0x10;                 // Call int 10 from realmode
  regs.h.bh = 0;
  regs.w.cx = 0;
  sregs.es = FP_SEG(&RMI);          // Ptr to data
  regs.x.edi = FP_OFF(&RMI);
  int386x(0x31, &regs, &regs, &sregs );
  vbelastbank=bnk;

}

short VBE_MaxBytesPerScanline (void)
{
  memset(&RMI,0,sizeof(RMI));       // Initialize the VESA-Data
  memset(&sregs,0,sizeof(sregs));
  memset(&regs,0,sizeof(regs));
  RMI.EAX=0x00004f06;
  RMI.EBX=3;
  regs.w.ax = 0x0300;               // Simulate Real-Mode interrupt
  regs.h.bl = 0x10;                 // Call int 10 from realmode
  regs.h.bh = 0;
  regs.w.cx = 0;
  sregs.es = FP_SEG(&RMI);          // Ptr to data
  regs.x.edi = FP_OFF(&RMI);
  int386x(0x31, &regs, &regs, &sregs );
  return regs.w.bx;
}

void VBE_SetPixelsPerScanline (short Pixels)
{
  memset(&RMI,0,sizeof(RMI));       // Initialize the VESA-Data
  memset(&sregs,0,sizeof(sregs));
  memset(&regs,0,sizeof(regs));
  RMI.EAX=0x00004f06;
  RMI.EBX=0;
  RMI.ECX=Pixels;
  regs.w.ax = 0x0300;               // Simulate Real-Mode interrupt
  regs.h.bl = 0x10;                 // Call int 10 from realmode
  regs.h.bh = 0;
  regs.w.cx = 0;
  sregs.es = FP_SEG(&RMI);          // Ptr to data
  regs.x.edi = FP_OFF(&RMI);
  int386x(0x31, &regs, &regs, &sregs );
}

void VBE_SetDACWidth (char bits)
{
  memset(&RMI,0,sizeof(RMI));       // Initialize the VESA-Data
  memset(&sregs,0,sizeof(sregs));
  memset(&regs,0,sizeof(regs));
  RMI.EAX=0x00004f08;
  RMI.EBX=bits<<8;
  regs.w.ax = 0x0300;               // Simulate Real-Mode interrupt
  regs.h.bl = 0x10;                 // Call int 10 from realmode
  regs.h.bh = 0;
  regs.w.cx = 0;
  sregs.es = FP_SEG(&RMI);          // Ptr to data
  regs.x.edi = FP_OFF(&RMI);
  int386x(0x31, &regs, &regs, &sregs );
}

int VBE_8BitDAC (void)
{
  VBE_VbeInfoBlock VBEInfo;
  PrepareInfoBlock (VBEInfo);
  VBE_Controller_Information (&VBEInfo);
  return (VBEInfo.Capabilities & 1 );
}


