//
//                       __      __      _  __ 
//                   .: (__ (__ (--) >-< ( (-_ :.
//
//
// (c) Copyright 1997 Mikko E. Mononen <memon@inside.org>
//

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "globals.hh"
#include "file_tga.hh"
#include "clax.h"
#include "mem.hh"
#include "materials.hh"

typedef struct {
  int     numTextures;
  char**  name;
  char**  texture;
} TextureList;



inline
char*
align64k( char* buffer )
{
  return (char *)(((int)(buffer + 0xffff)) & ~0xffff);
}

inline
char*
align256( char* buffer )
{
  return (char *)(((int)(buffer + 0xff)) & ~0xff);
}

char
bestMatch( char r, char g, char b, char* palette )
{
  int   dist;
  int   error = 1000000;
  char  best;

  if ( r < 0 ) r = 0;
  if ( g < 0 ) g = 0;
  if ( b < 0 ) b = 0;
  if ( r > 63 ) r = 63;
  if ( g > 63 ) g = 63;
  if ( b > 63 ) b = 63;
     
  for ( int k = 0; k < 256; k++ ) {
    dist = 77 * (r - palette[k * 3]) * (r - palette[k * 3]) +
           150 * (g - palette[k * 3 + 1]) * (g - palette[k * 3 + 1]) +
           29 * (b - palette[k * 3 + 2]) * (b - palette[k * 3 + 2]);
    if ( dist < error ) {
      error = dist;
      best = k;
    }
  }

  return best;
}


void
calcColorRamp( c_MATERIAL* mat )
{
//  printf( " - calculating color ramp..." );  fflush( stdout );

  for ( int i = 0; i < 64; i++ ) {
    float idiff = cos( (double)(63 - i) / 64 * (M_PI / 2.0) );
    float ispec = pow( idiff, mat->shininess * 200 ) * mat->shin_strength;

    int  r = (int)((0.25  * mat->ambient.r +
              idiff * mat->diffuse.r +
              ispec * mat->specular.r) * 63);
    int  g = (int)((0.25  * mat->ambient.g +
              idiff * mat->diffuse.g +
              ispec * mat->specular.g) * 63);
    int  b = (int)((0.25  * mat->ambient.b +
              idiff * mat->diffuse.b +
              ispec * mat->specular.b) * 63);

    mat->color_ramp[i] = bestMatch( r, g, b, IE_globals.palette );
  }

//  printf( "done.\n" );  fflush( stdout );
}


void
calcShadeTable()
{
//  printf( " calculating shade table..." );  fflush( stdout );

  for ( int i = 0; i < 256; i++ ) {    // each color
    for ( int j = 0; j < 64; j++ ) {   // shades

      float idiff = cos( (double)(63 - j) / 64 * (M_PI / 2.0) );
      float ispec = pow( idiff, 16 );

      int  r = (int)((float)IE_globals.palette[i * 3] * idiff);// + ispec * 32);
      int  g = (int)((float)IE_globals.palette[i * 3 + 1] * idiff);// + ispec * 32);
      int  b = (int)((float)IE_globals.palette[i * 3 + 2] * idiff);// + ispec * 32);

      IE_globals.shadeTable[j * 256 + i] = bestMatch( r, g, b, IE_globals.palette );
    }
  }

//  printf( "done.\n" );  fflush( stdout );
}

int
saveShadeTable( char* filename )
{
  FILE*  fp;

  if ( (fp = fopen( filename, "wb" )) == 0 )
    return clax_err_badfile;

  fwrite( IE_globals.shadeTable, 256 * 64, 1, fp );
  fclose( fp );
  
  return clax_err_ok;
}

int
loadShadeTable( char* filename )
{
  FILE*  fp;

  if ( (fp = fopen( filename, "rb" )) == 0 )
    return clax_err_badfile;

  fread( IE_globals.shadeTable, 256 * 64, 1, fp );
  fclose( fp );

  return clax_err_ok;
}


void
calcFogTable()
{
//  printf( " calculating fog table..." ); fflush( stdout );

  float  fr = IE_globals.fogColor.r * 63;
  float  fg = IE_globals.fogColor.g * 63;
  float  fb = IE_globals.fogColor.b * 63;

  for ( int i = 0; i < 256; i++ ) {    // each color
    for ( int j = 0; j < 64; j++ ) {   // shades

      float  a = ((float)j / 63.0);

      int  r = (int)( (float)IE_globals.palette[i * 3] * (1.0 - a)     + fr * a );
      int  g = (int)( (float)IE_globals.palette[i * 3 + 1] * (1.0 - a) + fg * a );
      int  b = (int)( (float)IE_globals.palette[i * 3 + 2] * (1.0 - a) + fb * a );

      IE_globals.fogTable[j * 256 + i] = bestMatch( r, g, b, IE_globals.palette );
    }
  }

//  printf( "done.\n" );  fflush( stdout );
}

int
saveFogTable( char* filename )
{
  FILE*  fp;

  if ( (fp = fopen( filename, "wb" )) == 0 )
    return clax_err_badfile;

  fwrite( IE_globals.fogTable, 256 * 64, 1, fp );
  fclose( fp );
  
  return clax_err_ok;
}

int
loadFogTable( char* filename )
{
  FILE*  fp;

  if ( (fp = fopen( filename, "rb" )) == 0 )
    return clax_err_badfile;

  fread( IE_globals.fogTable, 256 * 64, 1, fp );
  fclose( fp );

  return clax_err_ok;
}


int
putTexture( char* filename, TextureList* list )
{
  int  i, found;

  found = 0;

  // Is there already such texture?
  for ( i = 0; i < list->numTextures; i++ )
    if ( stricmp( list->name[i], filename ) == 0 )
      found = 1;

  if ( found == 0 ) {      
    // Add new texture to list
    char**  tmp;
    tmp = new char* [list->numTextures + 1];
    if ( !tmp )
      return clax_err_nomem;
    memcpy( tmp, list->name, list->numTextures * sizeof( char* ) );

    if ( list->name )
      delete [] list->name;
    list->name = tmp;
    list->name[list->numTextures] = new char [strlen( filename ) + 1];
    if ( !list->name[list->numTextures] )
      return clax_err_nomem;

    strcpy( list->name[list->numTextures], filename );
    list->numTextures++;
  }
  
  return clax_err_ok;
}

char*
getTexture( char* filename, TextureList* list )
{
  int  i, found;

  for ( i = 0; i < list->numTextures; i++ )
    if ( stricmp( list->name[i], filename ) == 0 )
      return list->texture[i];

  return 0;
}

int
initTextures( TextureList* list )
{
  int    i;
  char*  texturePtr;

//  printf( " Loading %d textures\n", list->numTextures );
//  fflush( stdout );

  // allocate space for textures and 64k align it
  texturePtr = align64k( initTexturePool( (list->numTextures + 1) * (256 * 256) ) );
  if ( texturePtr == 0 )
    return clax_err_nomem;
    
  list->texture = new char* [list->numTextures];
  if ( !list->texture )
    return clax_err_nomem;

  for ( i = 0; i < list->numTextures; i++ ) {
//    printf( " - %s ... ", list->name[i] );
    if ( loadTGA( list->name[i], texturePtr, 0 ) == clax_err_ok ) {
      list->texture[i] = texturePtr;
      texturePtr += (256 * 256);
//      printf( "ok!\n" );
    } else {
      list->texture[i] = 0;
//      printf( " - %s ... notfound", list->name[i] ); fflush( stdout );
    }
  }

  return clax_err_ok;
}


// Prepare materials for given palette.

int
initMaterials( c_SCENE* scene, int flags )
{

  int           numMaterials = 0;
  int           numRamps = 0;
  char*         tablePtr;
  char*         rampPtr;
  c_MATERIAL*   mat;
  c_MATERIAL**  matList;
  c_OBJECT*     obj;
  w_NODE*       node;
  TextureList   textureList;

  textureList.numTextures = 0;

  // Find out how many texture maps do we have.
  for ( node = scene->world; node; node = node->next ) {
    if ( node->type == clax_obj_material ) {
      mat = (c_MATERIAL *)node->object;
      if ( mat->texture.file )
        putTexture( mat->texture.file, &textureList );
      else if ( mat->reflection.file )
        putTexture( mat->reflection.file, &textureList );
      numRamps++;
    }
  }
  
  if ( initTextures( &textureList ) != clax_err_ok )
    return clax_err_nomem;

  // Load texture maps
  for ( node = scene->world; node; node = node->next ) {
    if ( node->type == clax_obj_material ) {
      mat = (c_MATERIAL *)node->object;

//      printf( " material: %s\n", mat->name );
      
      if ( mat->texture.file ) {
        // This material has texture map.
//        printf(" - texture: %s\n", mat->texture.file );
//        fflush(stdout);
        mat->texture.data = getTexture( mat->texture.file, &textureList );
      }
      else if ( mat->reflection.file ) {
        // This material has texture map.
//        printf(" - reflection map: %s\n", mat->reflection.file );
//        fflush(stdout);
        mat->reflection.data = getTexture( mat->reflection.file, &textureList );
      }

      calcColorRamp( mat );
    }
  }

//  printf("\n" );

  if ( flags & 1 ) {
    calcShadeTable();
    calcFogTable();
  }

  return clax_err_ok;
}