
{$define fast}

(*
** Fast Fourier Transform unit
** Written by Bas van Gaalen in September '96, using
**   'Inleiding Digitale Signaalverwerking' of Paul A. Lynn and W. Fuerst.
*)

unit _fft;

interface

uses
  types;

const
  maxfpr=100;  { Maximum number of frequency-values for fingerprint-checking }
  psize=80;                                              { Diagram/Plot size }

type
  vartype=longint;                                       { Type of variables }
  complex=record                                            { Complex number }
    re,im:vartype;
  end;
  datastruc=array[0..fftsize] of complex;                { The data-structure }
  magstruc=array[1..maxfpr] of byte;               { The magnitude-structure }

procedure fft(var d:datastruc; maxvals:word);
procedure convert(d:datastruc; var m:magstruc; mfpr:word);

implementation

const
  sine:array[1..15] of byte=(0,128,91,49,25,13,6,3,2,1,0,0,0,0,0);
  cdiv=128;

var
  divfact:byte;

(* Fast Fourier Transformation ---------------------------------------------*)

(*
** fft         : Fast Fourier Transformation routine
** d           : datastructure to analyse
** maxvals     : number of data points to analyse
*)

procedure fft{(var d:datastruc; maxvals:word)};

(* Complex multiply - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(*
** cmul        : perform a complex multiplication
** a,b         : two complex numbers to multiply
** c           : product of the two numbers
*)

procedure cmul(a,b:complex; var c:complex);
begin
  c.re:=(b.re*a.re-b.im*a.im) div cdiv;
  c.im:=(b.re*a.im+b.im*a.re) div cdiv;
end;

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*)

const
  pi=3.1415926535897932385;

var
  a,c,w,z,store:complex;
  i,j,k,l,idx,po2,po2m1,hmax:integer;
  power:byte;

begin
  { Calculate power: 2^power=maxvals }
  power:=0; i:=1;
  while i<maxvals do begin
    inc(power);
    i:=i shl 1;
  end;

  { Shuffle input data }
  j:=1;
  hmax:=maxvals shr 1;
  for i:=1 to (maxvals-1) shr divfact do begin
    if i<j then begin
      store:=d[j-1];
      d[j-1]:=d[i-1];
      d[i-1]:=store;
    end;
    k:=hmax;
    while k<j do begin
      dec(j,k);
      k:=k shr 1;
    end;
    inc(j,k);
  end;

  { Fast Fourier Transform }
  po2:=1;
  for l:=1 to power do begin
    po2m1:=po2;
    po2:=po2 shl 1;
    c.re:=cdiv; c.im:=0;
    w.re:=cdiv; w.im:=sine[l];
    for j:=1 to po2m1 do begin
      i:=j;
      repeat
        idx:=po2m1+i;
        a:=d[idx-1];
        cmul(c,a,z);
        d[idx-1].re:=d[i-1].re-z.re;
        d[idx-1].im:=d[i-1].im-z.im;
        d[i-1].re:=d[i-1].re+z.re;
        d[i-1].im:=d[i-1].im+z.im;
        inc(i,po2);
      until i>maxvals shr divfact;
      cmul(c,w,c);
    end;
  end;
end;

(* Conversion and Normalization --------------------------------------------*)

(*
** convert     : converts the complex Fourier signal to magnitude-spectrum
** m           : resulting (normalized and real) magnitude-spectrum
** mfpr        : number of points to save in the spectrum
*)

procedure convert{(d:datastruc; var m:magstruc; mfpr:word)};
var
  int_m:array[1..maxfpr] of word;               { Internal Magnitude (local) }
  i,max:word;
begin
  { Convert complex data to magnitude-spectrum and find out max }
  max:=0;
  for i:=1 to mfpr do begin
    with d[i] do
      int_m[i]:=round(sqrt(re*re+im*im));
    if abs(int_m[i])>max then max:=int_m[i];
  end;

  { Normalize }
  if max=0 then exit;
  for i:=1 to mfpr do
    m[i]:=round((psize/max)*int_m[i]);
end;

begin
{$ifdef fast}
  divfact:=1;
{$else}
  divfact:=0;
{$endif}
end.
