/*********************************************************************************************************
*
*	JXPdemo.cpp
*
*	Demo programme for the JXP library
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <process.h>
#include <io.h>

#include "JXP.h"

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

#define LOG_CLASS_NAME			"JXP Demo Log Class"
#define IMAGE_CLASS_NAME		"JXP Demo Image Class"
#define WINDOW_NAME				"JXP 2.01 Demo"
#define WINDOW_NAME_TRUECOLOUR	"JXP 2.01 Demo - True Colour"
#define WINDOW_NAME_PALETTISED	"JXP 2.01 Demo - Palettised"

#define UM_CREATE_IMAGE_WINDOW	(WM_USER+0x100)

#define ROUND_UP_TO_4(A)		(((A)+3)&0xfffffffc)

#define MAX_ARGUMENTS			128
#define MAX_IMAGES				64

/*********************************************************************************************************/
// Local typedefs
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static function prototypes
/*********************************************************************************************************/

static LRESULT CALLBACK LogWindowProc(HWND hWnd,UINT uMsg,WPARAM wParam,LPARAM lParam);
static LRESULT CALLBACK ImageWindowProc(HWND hWnd,UINT uMsg,WPARAM wParam,LPARAM lParam);

static void JXP_API_CALL AddLogMessage(char *Message);

static unsigned int __stdcall DecodeImagesThread(void *data);

static void SetImageWindowSize(int Width, int Height);

static void PaintImageWindow(HDC hDC);

static void ParseCommandLine(char *CommandLine);

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static variables
/*********************************************************************************************************/

static HWND hWndLog=NULL;
static HWND hWndEdit=NULL;
static HWND hWndImage=NULL;

static HINSTANCE g_hInstance;

static int LogLength=0;

static HANDLE DecodeImagesThreadHandle;
static unsigned int DecodeImagesThreadID;

static int NumImages;
static JXP_Image_24b **Images24b=NULL;
static JXP_Image_8b **Images8b=NULL;

static int CurrentImageIndex;
static JXP_Image_24b *CurrentImage24b=NULL;
static JXP_Image_8b *CurrentImage8b=NULL;

static JXP_Image_GreyScale *Mask=NULL;

static int NumArguments;
static char *Arguments[MAX_ARGUMENTS];

/*********************************************************************************************************/
// Functions
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	WinMain()
*
*	desc:		Programme entry point
*
*	notes:
*
**********************************************************************************************************/

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nShowCmd)
{
	WNDCLASS wc;
	MSG msg;

	g_hInstance=hInstance;

	// Register window class for log window
	wc.style=CS_HREDRAW|CS_VREDRAW;
	wc.lpfnWndProc=LogWindowProc;
	wc.cbClsExtra=0;
	wc.cbWndExtra=sizeof(DWORD);
	wc.hInstance=hInstance;
	wc.hIcon=NULL;
	wc.hCursor=LoadCursor(NULL,IDC_ARROW);
	wc.hbrBackground=(HBRUSH)(COLOR_WINDOW);
	wc.lpszMenuName=NULL;
	wc.lpszClassName=LOG_CLASS_NAME;
	if (!RegisterClass(&wc)) return 0;

	// Register window class for image window
	wc.style=CS_HREDRAW|CS_VREDRAW;
	wc.lpfnWndProc=ImageWindowProc;
	wc.cbClsExtra=0;
	wc.cbWndExtra=sizeof(DWORD);
	wc.hInstance=hInstance;
	wc.hIcon=NULL;
	wc.hCursor=LoadCursor(NULL,IDC_ARROW);
	wc.hbrBackground=(HBRUSH)(COLOR_WINDOW);
	wc.lpszMenuName=NULL;
	wc.lpszClassName=IMAGE_CLASS_NAME;
	if (!RegisterClass(&wc)) return 0;

	// Open windows for logging
	hWndLog=CreateWindowEx(WS_EX_CLIENTEDGE,LOG_CLASS_NAME,WINDOW_NAME,WS_OVERLAPPEDWINDOW,CW_USEDEFAULT,CW_USEDEFAULT,
		700,500,NULL,NULL,hInstance,NULL);
	if (!hWndLog) return 0;

	hWndEdit=CreateWindow("EDIT","",WS_CHILD|ES_MULTILINE|ES_READONLY|WS_HSCROLL|WS_VSCROLL,
		0,0,0,0,hWndLog,NULL,hInstance,NULL);
	if (!hWndEdit) return 0;

	ShowWindow(hWndEdit,SW_SHOW);
	ShowWindow(hWndLog,SW_SHOW);

	AddLogMessage("JXP 2.01\n");
	AddLogMessage("Copyright (C) 1997-2000, Saxon Druce (Penfold/JiNX)\n\n");

	// Create the list of arguments
	ParseCommandLine(lpCmdLine);

	if (NumArguments==0)
	{
		// No arguments - print message
		AddLogMessage("Argument not found. Syntax:\n\n");
		AddLogMessage("JXPdemo [options] filename[.jpg] [<filenames...>]\n");
		AddLogMessage("   /no \t\t\t no library output\n");
		AddLogMessage("   /nd \t\t\t no dithering\n");
		AddLogMessage("   /c <num> \t\t reduce to <num> colours\n");
		AddLogMessage("   /kw \t\t\t keep white\n");
		AddLogMessage("   /kb \t\t\t keep black\n");
		AddLogMessage("   /fip \t\t\t first image palette\n");
		AddLogMessage("   /m <filename[.msk]> \t mask file\n");
		AddLogMessage("   /sf <num> \t\t scale images by <num> percent\n");
		AddLogMessage("   /nn \t\t\t nearest neighbour resampling\n");
	}
	else
	{
		// Start a thread to decode the images in
		DecodeImagesThreadHandle=(void *)_beginthreadex(NULL,0,DecodeImagesThread,NULL,0,&DecodeImagesThreadID);
	}

	// Handle window messages
	for (;;)
	{
		if (GetMessage(&msg,NULL,0,0)<=0)
		{
			// received WM_QUIT -> shutdown
			
			// Close windows
			DestroyWindow(hWndLog);
			
			if (hWndImage)
				DestroyWindow(hWndImage);

			// Free arguments
			int i;
			for (i=0; i<NumArguments; i++)
				free(Arguments[i]);

			// Free images
			if (Images24b)
			{
				for (i=0; i<NumImages; i++)
				{
					if (Images24b[i])
						delete Images24b[i];
				}

				free(Images24b);
			}

			if (Images8b)
			{
				for (i=0; i<NumImages; i++)
				{
					if (Images8b[i])
						delete Images8b[i];
				}

				free(Images8b);
			}

			if (Mask)
				delete Mask;

			return msg.wParam;
		}

		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	return 0;
}

/*********************************************************************************************************
*
*	function:	LogWindowProc()
*
*	desc:		Window procedure for the logging window
*
*	notes:
*
**********************************************************************************************************/

static LRESULT CALLBACK LogWindowProc(HWND hWnd,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
	RECT rect;

	switch (uMsg)
	{
		case WM_CLOSE:
			PostQuitMessage(0);
			return 0;

		case WM_KEYDOWN:
			if (wParam==VK_ESCAPE)
				PostQuitMessage(0);
			return 0;

		case WM_SIZE:
			GetClientRect(hWnd,&rect);
			MoveWindow(hWndEdit,0,0,rect.right,rect.bottom,TRUE);
			return 0;

		case UM_CREATE_IMAGE_WINDOW:
			hWndImage=CreateWindowEx(WS_EX_CLIENTEDGE,IMAGE_CLASS_NAME,WINDOW_NAME,WS_OVERLAPPEDWINDOW,CW_USEDEFAULT,CW_USEDEFAULT,
				700,500,NULL,NULL,g_hInstance,NULL);

			if (CurrentImage24b)
			{
				SetImageWindowSize(CurrentImage24b->GetWidth(),CurrentImage24b->GetHeight());
				SetWindowText(hWndImage,WINDOW_NAME_TRUECOLOUR);
			}
			else
			{
				SetImageWindowSize(CurrentImage8b->GetWidth(),CurrentImage8b->GetHeight());
				SetWindowText(hWndImage,WINDOW_NAME_PALETTISED);
			}

			ShowWindow(hWndImage,SW_SHOW);
			return 0;

		default:
			return DefWindowProc(hWnd,uMsg,wParam,lParam);
	}
}

/*********************************************************************************************************
*
*	function:	ImageWindowProc()
*
*	desc:		Window procedure for the image window
*
*	notes:
*
**********************************************************************************************************/

static LRESULT CALLBACK ImageWindowProc(HWND hWnd,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
	PAINTSTRUCT Paint;
	HDC hDC;

	switch (uMsg)
	{
		case WM_CLOSE:
			PostQuitMessage(0);
			return 0;

		case WM_PAINT:
			hDC=BeginPaint(hWnd,&Paint);
			PaintImageWindow(hDC);
			EndPaint(hWnd,&Paint);
			return 0;

		case WM_KEYDOWN:

			switch (wParam)
			{
				case VK_ESCAPE:
					PostQuitMessage(0);
					return 0;
					break;

				case VK_LEFT:
					if (CurrentImageIndex==0)
						CurrentImageIndex=NumImages-1;
					else
						CurrentImageIndex--;
					break;
				
				case VK_RIGHT:
					if (CurrentImageIndex==NumImages-1)
						CurrentImageIndex=0;
					else
						CurrentImageIndex++;
					break;

				case VK_UP:
				case VK_DOWN:
					if (CurrentImage24b)
					{
						CurrentImage24b=NULL;
						CurrentImage8b=Images8b[CurrentImageIndex];
					}
					else
					{
						CurrentImage24b=Images24b[CurrentImageIndex];
						CurrentImage8b=NULL;
					}
					break;
				
				default:
					return 0;
			}

			if (CurrentImage24b)
			{
				CurrentImage24b=Images24b[CurrentImageIndex];
				SetImageWindowSize(CurrentImage24b->GetWidth(),CurrentImage24b->GetHeight());
				SetWindowText(hWndImage,WINDOW_NAME_TRUECOLOUR);
			}
			else
			{
				CurrentImage8b=Images8b[CurrentImageIndex];
				SetImageWindowSize(CurrentImage8b->GetWidth(),CurrentImage8b->GetHeight());
				SetWindowText(hWndImage,WINDOW_NAME_PALETTISED);
			}

			InvalidateRect(hWnd,NULL,FALSE);
			UpdateWindow(hWnd);
			return 0;

		default:
			return DefWindowProc(hWnd,uMsg,wParam,lParam);
	}
}

/*********************************************************************************************************
*
*	function:	AddLogMessage()
*
*	desc:		Adds a message to the log window
*
*	notes:
*
**********************************************************************************************************/

static void JXP_API_CALL AddLogMessage(char *Message)
{
	char Buf[1024];
	int Pos;

	while (LogLength>20000)
	{
		// Window is getting full - remove some content from the beginning of the log
		SendMessage(hWndEdit,EM_SETSEL,0,1000);
		SendMessage(hWndEdit,EM_REPLACESEL,FALSE,(DWORD)"Log removed...\r\n");

		LogLength-=1000-strlen("Log removed...\r\n");
	}

	// Now send the message to the window - need to convert any line breaks into \r\n for
	// them to display correctly

	Pos=0;
	while (*Message)
	{
		if (*Message=='\n')
		{
			Buf[Pos++]='\r';
			Buf[Pos++]='\n';
		}
		else if (*Message!='\r')
		{
			Buf[Pos++]=*Message;
		}
		Message++;

		if (Pos>=1022)
		{
			// Message is too long to fit in Buf - send what we have so far
			Buf[Pos]=0;
			
			SendMessage(hWndEdit,EM_SETSEL,LogLength,LogLength);
			SendMessage(hWndEdit,EM_REPLACESEL,FALSE,(DWORD)Buf);
			
			LogLength+=strlen(Buf);
			Pos=0;
		}
	}

	// Send anything remaining in buf
	Buf[Pos]=0;

	SendMessage(hWndEdit,EM_SETSEL,LogLength,LogLength);
	SendMessage(hWndEdit,EM_REPLACESEL,FALSE,(DWORD)Buf);

	LogLength+=strlen(Buf);
}

/*********************************************************************************************************
*
*	function:	DecodeImagesThread()
*
*	desc:		Procedure for the thread which decodes the images - so that the main thread can
*				continue to handle window messages, and update the output
*
*	notes:
*
**********************************************************************************************************/

static unsigned int __stdcall DecodeImagesThread(void *data)
{
	char Filenames[MAX_IMAGES][_MAX_PATH];
	int NumFilenames=0;
	JXP_DitherType DitherType=JXP_DITHER_FS;
	int NumColours=256;
	int PaletteGenerationFlags=0;
	int FirstImagePalette=0;
	int UseMaskFile=0;
	int ScaleFactor=100;
	char MaskFilename[_MAX_PATH];
	JXP_ResampleType ResampleType=JXP_RESAMPLE_BILINEAR;
	int i;

	// Send output to log window
	JXP_Utils.SetOutputFunction(AddLogMessage);

	// Go through the list of arguments, setting options and creating a list of JPGs to load
	for (i=0; i<NumArguments; i++)
	{
		char Arg[256];
		
		strcpy(Arg,Arguments[i]);
		strlwr(Arg);

		if (Arg[0]=='/')
		{
			// This argument is a parameter

			if (!strcmp(Arg,"/no"))
			{
				AddLogMessage("Turning off library output\n");
				JXP_Utils.SetOutputFunction(NULL);
			}
			else if (!strcmp(Arg,"/nd"))
			{
				AddLogMessage("Turning off dithering\n");
				DitherType=JXP_DITHER_NONE;
			}
			else if (!strcmp(Arg,"/c"))
			{
				if (i==NumArguments-1)
				{
					AddLogMessage("Value following /c parameter missing\n");
					return 1;
				}

				NumColours=atoi(Arguments[i+1]);
				i++;
			}
			else if (!strcmp(Arg,"/kw"))
			{
				AddLogMessage("Keeping white\n");
				PaletteGenerationFlags|=JXP_GP_KEEP_WHITE;
			}
			else if (!strcmp(Arg,"/kb"))
			{
				AddLogMessage("Keeping black\n");
				PaletteGenerationFlags|=JXP_GP_KEEP_BLACK;
			}
			else if (!strcmp(Arg,"/fip"))
			{
				AddLogMessage("Using first image only for palette generation\n");
				FirstImagePalette=1;
			}
			else if (!strcmp(Arg,"/m"))
			{
				if (i==NumArguments-1)
				{
					AddLogMessage("Filename following /m parameter missing\n");
					return 1;
				}

				char Drive[_MAX_DRIVE];
				char Dir[_MAX_DIR];
				char Fname[_MAX_FNAME];
				char Ext[_MAX_EXT];

				_splitpath(Arguments[i+1],Drive,Dir,Fname,Ext);
				if (!*Ext)
					strcpy(Ext,".msk");
				_makepath(MaskFilename,Drive,Dir,Fname,Ext);

				UseMaskFile=1;

				i++;
			}
			else if (!strcmp(Arg,"/sf"))
			{
				if (i==NumArguments-1)
				{
					AddLogMessage("Value following /sf parameter missing\n");
					return 1;
				}

				ScaleFactor=atoi(Arguments[i+1]);
				i++;

				if (ScaleFactor<=0)
				{
					AddLogMessage("Scale factor must be greater than zero\n");
					return 1;
				}
			}
			else if (!strcmp(Arg,"/nn"))
			{
				AddLogMessage("Using nearest neighbour resampling\n");
				ResampleType=JXP_RESAMPLE_NN;
			}
			else
			{
				// Undefined parameter
				AddLogMessage("Invalid argument - '");
				AddLogMessage(Arg);
				AddLogMessage("'\n");
				return 1;
			}
		}
		else
		{
			// This argument is a filename - potentially a wildcard

			if (!strchr(Arg,'*') && !strchr(Arg,'?'))
			{
				char Path[_MAX_PATH];
				char Drive[_MAX_DRIVE];
				char Dir[_MAX_DIR];
				char Fname[_MAX_FNAME];
				char Ext[_MAX_EXT];

				// Not a wildcard - add .jpg if the filename doesn't have an extension
				_splitpath(Arg,Drive,Dir,Fname,Ext);

				if (!*Ext)
					strcpy(Ext,".jpg");

				_makepath(Path,Drive,Dir,Fname,Ext);

				if (NumFilenames==MAX_IMAGES)
				{
					AddLogMessage("Too many filenames (including wildcard expansions)\n");
					return 1;
				}

				strcpy(Filenames[NumFilenames],Path);
				NumFilenames++;
			}
			else
			{
				struct _finddata_t FileInfo;
				char Path[_MAX_PATH];
				char Drive[_MAX_DRIVE];
				char Dir[_MAX_DIR];
				long Handle;

				// Is a wildcard - search for matching files

				_splitpath(Arg,Drive,Dir,NULL,NULL);

				Handle=_findfirst(Arg,&FileInfo);

				if (Handle!=-1)
				{
					do
					{
						if (NumFilenames==MAX_IMAGES)
						{
							AddLogMessage("Too many filenames (including wildcard expansions)\n");
							return 1;
						}

						_makepath(Path,Drive,Dir,FileInfo.name,NULL);

						strcpy(Filenames[NumFilenames],Path);
						NumFilenames++;

					} while (_findnext(Handle,&FileInfo)!=-1);

					_findclose(Handle);
				}
			}
		}
	}

	AddLogMessage("\n");

	if (NumFilenames==0)
	{
		AddLogMessage("No files to decode\n");
		return 1;
	}

	// Allocate image pointer array
	NumImages=NumFilenames;
	Images24b=(JXP_Image_24b **)malloc(sizeof(JXP_Image_24b *)*NumImages);
	if (!Images24b)
	{
		AddLogMessage("Insufficient memory\n");
		return 1;
	}
	memset(Images24b,0,sizeof(JXP_Image_24b *)*NumImages);

	// Set byte order to BGR
	JXP_Utils.SetByteOrder(JXP_BYTEORDER_BGR);

	// Decode all of the JPEGs
	for (i=0; i<NumImages; i++)
	{
		AddLogMessage("Decoding image '");
		AddLogMessage(Filenames[i]);
		AddLogMessage("'\n\n");

		if (JXP_DecodeJPEGFile(Filenames[i],&Images24b[i])!=JXP_SUCCESS)
		{
			JXP_Utils.Output("JXP Error in JXP_DecodeJPEGFile(): %s \n",JXP_Utils.ErrorDescription(JXP_Utils.GetLastError()));
			return 1;
		}
	}

	if (ScaleFactor!=100)
	{
		JXP_Utils.Output("Rescaling images by %d%%\n\n",ScaleFactor);

		for (i=0; i<NumImages; i++)
		{
			int NewWidth,NewHeight;
			
			// Compute new dimensions
			NewWidth=ScaleFactor*Images24b[i]->GetWidth()/100;
			if (NewWidth<2)
				NewWidth=2;

			NewHeight=ScaleFactor*Images24b[i]->GetHeight()/100;
			if (NewHeight<2)
				NewHeight=2;

			// Rescale image
			JXP_Image_24b *NewImage;
			if (JXP_ResampleImage(Images24b[i],NewWidth,NewHeight,&NewImage,ResampleType)!=JXP_SUCCESS)
			{
				JXP_Utils.Output("JXP Error in JXP_ResampleImage(): %s \n",JXP_Utils.ErrorDescription(JXP_Utils.GetLastError()));
				return 1;
			}

			// Replace old image with resampled one
			delete Images24b[i];
			Images24b[i]=NewImage;
		}
	}

	// Create a palette from the images
	JXP_Palette *Palette;
	if (!FirstImagePalette)
	{
		// Use all of the images to create the palette
		if (JXP_GeneratePaletteMultiple(Images24b,NumImages,&Palette,NumColours,PaletteGenerationFlags)!=JXP_SUCCESS)
		{
			JXP_Utils.Output("JXP Error in JXP_GeneratePaletteMultiple(): %s \n",JXP_Utils.ErrorDescription(JXP_Utils.GetLastError()));
			return 1;
		}
	}
	else
	{
		// Create the palette from the first image only
		if (JXP_GeneratePalette(Images24b[0],&Palette,NumColours,PaletteGenerationFlags)!=JXP_SUCCESS)
		{
			JXP_Utils.Output("JXP Error in JXP_GeneratePalette(): %s \n",JXP_Utils.ErrorDescription(JXP_Utils.GetLastError()));
			return 1;
		}
	}

	// Now fit the 24bit images to the palette
	Images8b=(JXP_Image_8b **)malloc(sizeof(JXP_Image_8b *)*NumImages);
	if (!Images8b)
	{
		AddLogMessage("Insufficient memory\n");
		return 1;
	}
	memset(Images8b,0,sizeof(JXP_Image_8b *)*NumImages);

	for (i=0; i<NumImages; i++)
	{
		if (JXP_FitToPalette(Images24b[i],Palette,&Images8b[i],DitherType)!=JXP_SUCCESS)
		{
			JXP_Utils.Output("JXP Error in JXP_FitToPalette(): %s \n",JXP_Utils.ErrorDescription(JXP_Utils.GetLastError()));
			return 1;
		}
	}

	// Now done with the palette
	delete Palette;

	if (UseMaskFile)
	{
		AddLogMessage("\nReading mask file\n");

		// Read in the mask file
		if (JXP_ReadMaskFile(MaskFilename,&Mask)!=JXP_SUCCESS)
		{
			JXP_Utils.Output("JXP Error in JXP_ReadMaskFile(): %s \n",JXP_Utils.ErrorDescription(JXP_Utils.GetLastError()));
			return 1;
		}
	}

	// Start with viewing the first image, in 24bit mode
	CurrentImageIndex=0;
	CurrentImage24b=Images24b[0];

	// Post message to the log window to tell it to create the image window
	PostMessage(hWndLog,UM_CREATE_IMAGE_WINDOW,0,0);

	CloseHandle(DecodeImagesThreadHandle);

	return 0;
}

/*********************************************************************************************************
*
*	function:	SetImageWindowSize()
*
*	desc:		Sets the size of the image window to have a client area of the specified dimensions
*
*	notes:
*
**********************************************************************************************************/

static void SetImageWindowSize(int Width, int Height)
{
	int WindowWidth,WindowHeight;
	RECT Rect;

	// Get current window position
	GetWindowRect(hWndImage,&Rect);

	// Compute the actual window dimensions including the borders
	WindowWidth=Width+2*GetSystemMetrics(SM_CXFIXEDFRAME)+5;
	WindowHeight=Height+2*GetSystemMetrics(SM_CYFIXEDFRAME)+GetSystemMetrics(SM_CYCAPTION)+5;

	// Set new window size
	MoveWindow(hWndImage,Rect.left,Rect.top,WindowWidth,WindowHeight,TRUE);
}

/*********************************************************************************************************
*
*	function:	PaintImageWindow()
*
*	desc:		Paints the image window with the current image
*
*	notes:
*
**********************************************************************************************************/

static void PaintImageWindow(HDC hDC)
{
	HBITMAP hBitmap;
	BITMAPINFO binfo;
	int Width,Height;
	unsigned char *GDIData;
	int i,j;

	// Get current image dimensions
	if (CurrentImage24b)
	{
		Width=CurrentImage24b->GetWidth();
		Height=CurrentImage24b->GetHeight();
	}
	else
	{
		Width=CurrentImage8b->GetWidth();
		Height=CurrentImage8b->GetHeight();
	}

	// Create a bitmap, to get a pointer to its data
	memset(&binfo,0,sizeof(BITMAPINFO));

	binfo.bmiHeader.biSize=sizeof(BITMAPINFOHEADER);
	binfo.bmiHeader.biWidth=Width;
	binfo.bmiHeader.biHeight=-Height;
	binfo.bmiHeader.biPlanes=1;
	binfo.bmiHeader.biBitCount=24;
	binfo.bmiHeader.biCompression=BI_RGB;
	binfo.bmiHeader.biSizeImage=0;
	binfo.bmiHeader.biXPelsPerMeter=10000;
	binfo.bmiHeader.biYPelsPerMeter=10000;
	binfo.bmiHeader.biClrUsed=0;
	binfo.bmiHeader.biClrImportant=0;

	hBitmap=CreateDIBSection(hDC,&binfo,DIB_PAL_COLORS,(void **)&GDIData,NULL,0);

	if (!Mask)
	{
		// Copy image into bitmap
		if (CurrentImage24b)
		{
			for (i=0; i<Height; i++)
				memcpy(GDIData+ROUND_UP_TO_4(3*Width)*i,CurrentImage24b->GetData()+3*Width*i,3*Width);
		}
		else
		{
			unsigned char *Data=CurrentImage8b->GetData();
			unsigned char *PaletteData=CurrentImage8b->GetPalette()->GetData();

			for (i=0; i<Height; i++)
			{
				for (j=0; j<Width; j++)
				{
					GDIData[ROUND_UP_TO_4(3*Width)*i+3*j]  =PaletteData[3*Data[Width*i+j]+2];
					GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+1]=PaletteData[3*Data[Width*i+j]+1];
					GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+2]=PaletteData[3*Data[Width*i+j]];
				}
			}
		}
	}
	else
	{
		// Copy the image into the bitmap, using the mask to mask pixels
		if (CurrentImage24b)
		{
			unsigned char *Data=CurrentImage24b->GetData();
			unsigned char *MaskData=Mask->GetData();
			int MaskWidth=Mask->GetWidth();
			int MaskHeight=Mask->GetHeight();

			for (i=0; i<Height; i++)
			{
				for (j=0; j<Width; j++)
				{
					if (i<MaskHeight && j<MaskWidth && MaskData[MaskWidth*i+j])
					{
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j]  =Data[3*(Width*i+j)];
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+1]=Data[3*(Width*i+j)+1];
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+2]=Data[3*(Width*i+j)+2];
					}
					else
					{
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j]  =0;
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+1]=0;
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+2]=0;
					}
				}
			}
		}
		else
		{
			unsigned char *Data=CurrentImage8b->GetData();
			unsigned char *PaletteData=CurrentImage8b->GetPalette()->GetData();
			unsigned char *MaskData=Mask->GetData();
			int MaskWidth=Mask->GetWidth();
			int MaskHeight=Mask->GetHeight();

			for (i=0; i<Height; i++)
			{
				for (j=0; j<Width; j++)
				{
					if (i<MaskHeight && j<MaskWidth && MaskData[MaskWidth*i+j])
					{
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j]  =PaletteData[3*Data[Width*i+j]+2];
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+1]=PaletteData[3*Data[Width*i+j]+1];
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+2]=PaletteData[3*Data[Width*i+j]];
					}
					else
					{
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j]  =0;
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+1]=0;
						GDIData[ROUND_UP_TO_4(3*Width)*i+3*j+2]=0;
					}
				}
			}
		}
	}

	StretchDIBits(hDC,0,0,Width,Height,0,0,Width,Height,GDIData,&binfo,DIB_RGB_COLORS,SRCCOPY);

	DeleteObject(hBitmap);
}

/*********************************************************************************************************
*
*	function:	ParseCommandLine()
*
*	desc:		Parses the command line string into an array of arguments
*
*	notes:
*
**********************************************************************************************************/

static void ParseCommandLine(char *CommandLine)
{
	int Search;
	int Pos;
	char Buf[256];

	NumArguments=0;

	for (;;)
	{
		while (*CommandLine==' ')
			CommandLine++;

		if (!*CommandLine)
			break;

		if (*CommandLine=='"')
		{
			CommandLine++;
			Search='"';
		}
		else
		{
			Search=' ';
		}

		Pos=0;

		while (*CommandLine && *CommandLine!=Search)
		{
			Buf[Pos++]=*CommandLine;
			CommandLine++;
		}

		if (*CommandLine)
			CommandLine++;

		Buf[Pos]=0;

		Arguments[NumArguments]=strdup(Buf);
		NumArguments++;

		if (NumArguments==MAX_ARGUMENTS)
			break;
	}
}

/*********************************************************************************************************/
