/*********************************************************************************************************
*
*	JXP_Targa.cpp
*
*	Some basic functions to handle Targa images
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

// Revision history:
// ----------------

// v2.01 - 22/1/2000
// No changes to this file in this version.
//
// v2.0 - 4/1/2000
// First version to include Targa loading convenience functions.

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "JXP_Targa.h"

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

/*********************************************************************************************************/
// Local typedefs
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static function prototypes
/*********************************************************************************************************/

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Functions
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Targa_Interface::GetTargaFormat()
*
*	desc:		Gets the type of the file in 'Source'
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Targa_Interface::GetTargaFormat(JXP_Data *Source, JXP_TargaType *Format)
{
	int C;

	// The Targa format is in the 3rd byte in the file
	C=Source->GetData()[2];

	// Work out the Targa format
	switch (C)
	{
		case 1:
			*Format=JXP_TGA_INDEXED;
			break;
		case 2:
			*Format=JXP_TGA_RGB;
			break;
		default:
			*Format=JXP_TGA_UNSUPPORTED;
	}

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_Targa_Interface::GetTargaFileFormat()
*
*	desc:		Gets the type of the file 'Filename'
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Targa_Interface::GetTargaFileFormat(char *Filename, JXP_TargaType *Format)
{
	FILE *Fptr;
	int C;

	// Open file
	Fptr=fopen(Filename,"rb");
	if (!Fptr)
	{
		JXP_Utils.SetLastError(JXP_ERROR_FILE_NOT_FOUND);
		return JXP_FAILURE;
	}

	// The Targa format is in the 3rd byte in the file
	fgetc(Fptr);
	fgetc(Fptr);
	C=fgetc(Fptr);

	// Work out the Targa format
	switch (C)
	{
		case 1:
			*Format=JXP_TGA_INDEXED;
			break;
		case 2:
			*Format=JXP_TGA_RGB;
			break;
		default:
			*Format=JXP_TGA_UNSUPPORTED;
	}

	// Close the file
	fclose(Fptr);

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_Targa_Interface::Read24bTarga()
*
*	desc:		Reads in a 24bit RGB TGA
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Targa_Interface::Read24bTarga(JXP_Data *Source, JXP_Image_24b **Destination)
{
	unsigned char *Data,*DestinationData;
	int Width,Height;
	JXP_Image_24b *Image;
	int i;

	// Get source data
	Data=Source->GetData();

	// Check id field length
	if (Data[0]!=0)
	{
		JXP_Utils.Output("Cannot read TGA with non-zero identification field length\n");
		JXP_Utils.SetLastError(JXP_ERROR_NOT_SUPPORTED);
		return JXP_FAILURE;
	}

	// Check colour map
	if (Data[1]!=0)
	{
		JXP_Utils.Output("Cannot read 24b TGA with a colour map\n");
		JXP_Utils.SetLastError(JXP_ERROR_NOT_SUPPORTED);
		return JXP_FAILURE;
	}

	// Check Targa type
	if (Data[2]!=2)
	{
		JXP_Utils.Output("Targa is not a 24bit image\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Check image origin
	if (Data[8]!=0 || Data[9]!=0 || Data[10]!=0 || Data[11]!=0)
	{
		JXP_Utils.Output("Image origin not (0,0)\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Get image dimensions
	Width=Data[12]+(Data[13]<<8);
	Height=Data[14]+(Data[15]<<8);

	// Check bitdepth
	if (Data[16]!=24)
	{
		JXP_Utils.Output("Bit depth not 24bit, found\n",Data[16]);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Test for whether TGA is upside down
	int UpsideDown=!(Data[17]&32);

	// Skip data pointer to start of image
	Data+=18;

	// Create new image
	Image=new JXP_Image_24b(Width,Height);

	// Get destination data
	DestinationData=Image->GetData();

	// Now read in the TGA
	if (!UpsideDown)
	{
		memcpy(DestinationData,Data,3*Width*Height);
	}
	else
	{
		for (i=0; i<Height; i++)
		{
			memcpy(DestinationData+3*Width*i,Data+3*Width*(Height-i-1),3*Width);
		}
	}

	// TGA is in BGR order - swap if byte order is set to RGB
	int Temp;
	if (JXP_Utils.GetByteOrder()==JXP_BYTEORDER_RGB)
	{
		for (i=0; i<Width*Height; i++)
		{
			Temp=DestinationData[3*i];
			DestinationData[3*i]=DestinationData[3*i+2];
			DestinationData[3*i+2]=Temp;
		}
	}

	// Set return image
	*Destination=Image;

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_Targa_Interface::Read8bTarga()
*
*	desc:		Reads in a 8bit RGB TGA
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Targa_Interface::Read8bTarga(JXP_Data *Source, JXP_Image_8b **Destination)
{
	unsigned char *Data,*DestinationData;
	int ColourMapStart,ColourMapEnd;
	int Width,Height;
	JXP_Image_8b *Image;
	int i;

	// Get source data
	Data=Source->GetData();

	// Check id field length
	if (Data[0]!=0)
	{
		JXP_Utils.Output("Cannot read TGA with non-zero identification field length\n");
		JXP_Utils.SetLastError(JXP_ERROR_NOT_SUPPORTED);
		return JXP_FAILURE;
	}

	// Check colour map
	if (Data[1]!=1)
	{
		JXP_Utils.Output("TGA has invalid colour map type\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Check Targa type
	if (Data[2]!=1)
	{
		JXP_Utils.Output("Targa is not an 8bit image\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Get colour map start and end
	ColourMapStart=Data[3]+(Data[4]<<8);
	ColourMapEnd=ColourMapStart+Data[5]+(Data[6]<<8);
	if (ColourMapStart<0 || ColourMapStart>255)
	{
		JXP_Utils.Output("First colour map entry is invalid\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	if (ColourMapEnd<0 || ColourMapEnd>256)
	{
		JXP_Utils.Output("Last colour map entry is invalid\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Check colour map is 24bit
	if (Data[7]!=24)
	{
		JXP_Utils.Output("Colour map must be 24bit\n");
		JXP_Utils.SetLastError(JXP_ERROR_NOT_SUPPORTED);
		return JXP_FAILURE;
	}

	// Check image origin
	if (Data[8]!=0 || Data[9]!=0 || Data[10]!=0 || Data[11]!=0)
	{
		JXP_Utils.Output("Image origin not (0,0)\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Get image dimensions
	Width=Data[12]+(Data[13]<<8);
	Height=Data[14]+(Data[15]<<8);

	// Check bitdepth
	if (Data[16]!=8)
	{
		JXP_Utils.Output("Bit depth not 8bit, found\n",Data[16]);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Test for whether TGA is upside down
	int UpsideDown=!(Data[17]&32);

	// Skip data pointer to start of palette
	Data+=18;

	// Create new image
	Image=new JXP_Image_8b(Width,Height);

	// Now read in the palette
	for (i=ColourMapStart; i<ColourMapEnd; i++)
	{
		// Palette is in BGR - convert to RGB
		Image->GetPalette()->GetData()[3*i]  =Data[3*(i-ColourMapStart)+2];
		Image->GetPalette()->GetData()[3*i+1]=Data[3*(i-ColourMapStart)+1];
		Image->GetPalette()->GetData()[3*i+2]=Data[3*(i-ColourMapStart)];
	}

	// Skip data pointer to start of image
	Data+=3*(ColourMapEnd-ColourMapStart);

	// Get destination data
	DestinationData=Image->GetData();

	// Now read in the TGA
	if (!UpsideDown)
	{
		memcpy(DestinationData,Data,Width*Height);
	}
	else
	{
		for (i=0; i<Height; i++)
		{
			memcpy(DestinationData+Width*i,Data+Width*(Height-i-1),Width);
		}
	}

	// Set return image
	*Destination=Image;

	return JXP_SUCCESS;
}

/*********************************************************************************************************/
