/*********************************************************************************************************
*
*	JXP_Resample.cpp
*
*	Some functions to perform resampling
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

// Revision history:
// ----------------

// v2.01 - 22/1/2000
// Fixed a bug with reading outside the source image in the bilinear filter.
//
// v2.0 - 4/1/2000
// First version to include resampling functions.

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "JXP_Resample.h"

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

/*********************************************************************************************************/
// Local typedefs
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static function prototypes
/*********************************************************************************************************/

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Functions
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Resample_Interface::ResampleImage()
*
*	desc:		Resamples an image
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Resample_Interface::ResampleImage(JXP_Image_24b *Source, int Width, int Height, JXP_Image_24b **Destination, JXP_ResampleType ResampleType)
{
	// Check requested width and height
	if (Width<2 || Height<2)
	{
		JXP_Utils.Output("Error - width and height must be greater than one\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_PARAMETER);
		return JXP_FAILURE;
	}

	// Create new image of desired dimensions
	JXP_Image_24b *Image;
	Image=new JXP_Image_24b(Width,Height);

	switch (ResampleType)
	{
		case JXP_RESAMPLE_NN:
			{
				// Nearest Neighbour
				int SourceWidth,SourceHeight;
				unsigned char *SourceData,*DestinationData;

				// Get source info
				SourceWidth=Source->GetWidth();
				SourceHeight=Source->GetHeight();
				SourceData=Source->GetData();

				// Get destination info
				DestinationData=Image->GetData();

				// Do resample
				int x,y;
				for (y=0; y<Height; y++)
				{
					for (x=0; x<Width; x++)
					{
						int SourceX,SourceY;

						SourceX=x*SourceWidth/Width;
						SourceY=y*SourceHeight/Height;

						DestinationData[3*(y*Width+x)]=SourceData[3*(SourceY*SourceWidth+SourceX)];
						DestinationData[3*(y*Width+x)+1]=SourceData[3*(SourceY*SourceWidth+SourceX)+1];
						DestinationData[3*(y*Width+x)+2]=SourceData[3*(SourceY*SourceWidth+SourceX)+2];
					}
				}
			}
			break;

		case JXP_RESAMPLE_BILINEAR:
			{
				// Bilinear Filter
				int SourceWidth,SourceHeight;
				unsigned char *SourceData,*DestinationData;

				// Get source info
				SourceWidth=Source->GetWidth();
				SourceHeight=Source->GetHeight();
				SourceData=Source->GetData();

				// Get destination info
				DestinationData=Image->GetData();

				// Do resample
				int x,y;
				for (y=0; y<Height; y++)
				{
					for (x=0; x<Width; x++)
					{
						float SourceX,SourceY;

						SourceX=((float)x)*(SourceWidth-1.0f)/(Width-1.0f);
						SourceY=((float)y)*(SourceHeight-1.0f)/(Height-1.0f);

						int Left=(int)SourceX;
						int Right=Left+1;
						int Top=(int)SourceY;
						int Bottom=Top+1;

						if (Right>=SourceWidth)
							Right=SourceWidth-1;
						if (Bottom>=SourceHeight)
							Bottom=SourceHeight-1;

						float t,omt;

						int i;

						for (i=0; i<3; i++)
						{
							t=SourceX-Left;
							omt=1.0f-t;

							float MidTop=omt*SourceData[3*(Top*SourceWidth+Left)+i] + 
								t*SourceData[3*(Top*SourceWidth+Right)+i];

							float MidBottom=omt*SourceData[3*(Bottom*SourceWidth+Left)+i] + 
								t*SourceData[3*(Bottom*SourceWidth+Right)+i];

							t=SourceY-Top;
							omt=1.0f-t;

							float Mid=omt*MidTop + t*MidBottom;

							int MidInt=(int)Mid;

							if (MidInt<0)
								DestinationData[3*(y*Width+x)+i]=0;
							else if (MidInt>255)
								DestinationData[3*(y*Width+x)+i]=255;
							else
								DestinationData[3*(y*Width+x)+i]=MidInt;
						}
					}
				}
			}
			break;

		default:
			JXP_Utils.Output("Error - invalid resample type\n");
			JXP_Utils.SetLastError(JXP_ERROR_INVALID_PARAMETER);
			return JXP_FAILURE;
	}

	// Set return image
	*Destination=Image;

	return JXP_SUCCESS;
}

/*********************************************************************************************************/
