/*********************************************************************************************************
*
*	JXP_Mask.cpp
*
*	Functions for handling of .MSK Mask files
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

// A mask file is a run length encoded black and white image,
// which can be useful for certain applications.

// Revision history:
// ----------------

// v2.01 - 22/1/2000
// No changes to this file in this version.
//
// v2.0 - 4/1/2000
// Major rewrite, including conversion to C++. Added library function to
// generate mask files, not just read them.
//
// v1.21 - 3/1/1998
// No changes to this file in this version.
//
// v1.2 - 2/1/1998
// Initial inclusion of mask file support.

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "JXP_Mask.h"

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

/*********************************************************************************************************/
// Local typedefs
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static function prototypes
/*********************************************************************************************************/

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Functions
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Mask_Interface::GenerateMask()
*
*	desc:		Generates a mask file from the source image
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Mask_Interface::GenerateMask(JXP_Image_24b *Source, JXP_Data **Destination)
{
	unsigned char *SourceData,*DestinationData;
	int Width,Height;

	// Get info on source image
	Width=Source->GetWidth();
	Height=Source->GetHeight();
	SourceData=Source->GetData();

	// Allocate temporary block for destination data - mask file data can never generate more than
	// 1 byte per pixel
	DestinationData=(unsigned char *)malloc(Width*Height);
	if (!DestinationData)
	{
		JXP_Utils.Output("Error - Insufficient memory\n");
		JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
		return JXP_FAILURE;
	}

	int CurrentColour,Count,Pos;
	int i;

	// Now generate the mask file

	// CurrentColour=0 for black, 1 for white
	CurrentColour=0;
	Count=0;
	Pos=0;

	for (i=0; i<Width*Height; i++)
	{
		if (SourceData[3*i]==0 && SourceData[3*i+1]==0 && SourceData[3*i+2]==0)
		{
			// i'th pixel is black
			if (CurrentColour==1)
			{
				// Last colour was white
				DestinationData[Pos++]=Count;
				CurrentColour=0;
				Count=1;
			}
			else
			{
				Count++;
				if (Count==255)
				{
					DestinationData[Pos++]=255;
					CurrentColour=1;
					Count=0;
				}
			}
		}
		else if (SourceData[3*i]==255 && SourceData[3*i+1]==255 && SourceData[3*i+2]==255)
		{
			// i'th pixel is white
			if (CurrentColour==0)
			{
				// Last colour was black
				DestinationData[Pos++]=Count;
				CurrentColour=1;
				Count=1;
			}
			else
			{
				Count++;
				if (Count==255)
				{
					DestinationData[Pos++]=255;
					CurrentColour=0;
					Count=0;
				}
			}
		}
		else
		{
			JXP_Utils.Output("Pixel %d has invalid colour (%d,%d,%d)\n",i,SourceData[3*i],SourceData[3*i+1],SourceData[3*i+2]);
			JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
			free(DestinationData);
			return JXP_FAILURE;
		}
	}

	// Add final count if it is not zero
	if (Count)
		DestinationData[Pos++]=Count;

	// Pos is now the length of the mask compressed image data. Mask files
	// have a 10 byte header, so allocate the destination data block

	JXP_Data *Mask;
	Mask=new JXP_Data(Pos+10);

	// Fill in header for MSK 1.00 file
	Mask->GetData()[0]='M';
	Mask->GetData()[1]='S';
	Mask->GetData()[2]='K';
	Mask->GetData()[3]=1;
	Mask->GetData()[4]=0;
	Mask->GetData()[5]=0;

	// Fill in image dimensions
	Mask->GetData()[6]=Width>>8;
	Mask->GetData()[7]=Width&0xFF;
	Mask->GetData()[8]=Height>>8;
	Mask->GetData()[9]=Height&0xFF;

	// Copy image data
	memcpy(Mask->GetData()+10,DestinationData,Pos);

	// Free temporary data
	free(DestinationData);

	// Return mask
	*Destination=Mask;

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_Mask_Interface::ReadMask()
*
*	desc:		Reads a mask file
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Mask_Interface::ReadMask(JXP_Data *Source, JXP_Image_GreyScale **Destination)
{
	unsigned char *SourceData;
	int SourceLength;

	// Get source data
	SourceData=Source->GetData();
	SourceLength=Source->GetLength();

	if (SourceData[0]!='M' ||
		SourceData[1]!='S' ||
		SourceData[2]!='K' ||
		SourceData[3]!=1 ||
		SourceData[4]!=0 ||
		SourceData[5]!=0)
	{
		JXP_Utils.Output("Error - not a valid v1.00 MSK file\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	JXP_Utils.Output("Verified MSK format, version 1.00\n");

	// Get image dimensions
	int Width,Height;
	Width=(SourceData[6]<<8)+SourceData[7];
	Height=(SourceData[8]<<8)+SourceData[9];

	// Create destination image, and get data pointer
	JXP_Image_GreyScale *Image;
	Image=new JXP_Image_GreyScale(Width,Height);

	unsigned char *DestinationData;
	DestinationData=Image->GetData();

	// Now read the file
	int CurrentColour;
	int Pos;

	CurrentColour=0; // start with black
	Pos=0;
	int i;

	// Note image starts at 10 bytes into the source data
	for (i=10; i<SourceLength; i++)
	{
		// Output SourceData[i] bytes of CurrentColour to DestinationData[Pos];

		if (Pos+SourceData[i]>Width*Height)
		{
			JXP_Utils.Output("Error - data stream goes past end of image\n");
			JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
			return JXP_FAILURE;
		}

		memset(DestinationData+Pos,CurrentColour,SourceData[i]);
		Pos+=SourceData[i];

		// Swap CurrentColour
		CurrentColour=(CurrentColour?0:255);
	}

	if (Pos!=Width*Height)
	{
		JXP_Utils.Output("Error - data stream did not fill destination image\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Return image
	*Destination=Image;

	return JXP_SUCCESS;
}

/*********************************************************************************************************/
