/*********************************************************************************************************
*
*	JXP_JPEG.h
*
*	JPEG functionality of the JXP library
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

// Revision history:
// ----------------

// v2.01 - 22/1/2000
// No changes to this file in this version.
//
// v2.0 - 4/1/2000
// Major rewrite, including conversion to C++.
//
// v1.21 - 3/1/1998
// No changes to this file in this version.
//
// v1.2 - 2/1/1998
// No changes to this file in this version.
//
// v1.1 - 1/1/1998
// Fixed a bug with non-initialisation of restart_interval
// causing decoding more than one JPEG to die.
//
// v1.0 - 23/12/1997
// Initial release.

#ifndef __JXP_JPEG_H
#define __JXP_JPEG_H

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include "JXP.h"

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

/*********************************************************************************************************/
// Global typedefs
/*********************************************************************************************************/

// JXP_JPEG_HuffmanTable - defines a huffman table
typedef struct
{
	int Defined;						// set to 1 if a table is defined
	int Bits[16];
	unsigned char *Huffval;
	int *Huffsize;
	int *Huffcode;

	int Valptr[16];
	int Mincode[16];
	int Maxcode[16];

} JXP_JPEG_HuffmanTable;

// JXP_JPEG_Component - defines one of the JPEG components (Y, Cb or Cr)
typedef struct
{
	int ID;								// component ID
	int H,V;							// horizontal and vertical sampling factors
	int QT;								// quatisation table (id)
  
	unsigned char *QTable;				// pointer to quantisation table
	JXP_JPEG_HuffmanTable *DC_HTable;	// pointers to DC and AC Huffman tables
	JXP_JPEG_HuffmanTable *AC_HTable;

	int Predictor;						// the DC predictor value

	unsigned char *Data;				// decoded data in this plane

} JXP_JPEG_Component;

// JXP_JPEG_Interface - interface class which performs JPEG functions
class JXP_JPEG_Interface
{
public:

	// DecodeJPEG() - the main decoding function
	int JXP_API_CALL DecodeJPEG(JXP_Data *Source, JXP_Image_24b **Destination);

private:

	// The source data, and current position in it
	JXP_Data *Source;
	int CurrentPosition;

	// The restart interval
	int RestartInterval;

	// The 4 possible Huffman tables for DC and AC coefficients
	JXP_JPEG_HuffmanTable DC_HTable[4];
	JXP_JPEG_HuffmanTable AC_HTable[4];

	// The 4 possible quantisation tables
	unsigned char *QTable[4];

	// The 3 components of a JPEG - the luminance channel (Y) and 
	// the chrominance channels (Cb and Cr)
	JXP_JPEG_Component Y,Cb,Cr;

	// Image dimensions - Samples (Width) * Lines (Height)
	int Lines,Samples;

	// The vertical and horizontal dimensions, in numbers of MCU
	int H_MCU,V_MCU;

	// Values used by the GetBit() method
	int BitB,BitCount;

	// GetByte() - gets the next byte in the source data
	unsigned char JXP_API_CALL GetByte(void);

	// UngetByte() - undoes the previous call to GetByte()
	void JXP_API_CALL UngetByte(void);

	// GetShort() - gets the next 2 bytes in the data stream as a big-endian 16bit int
	int JXP_API_CALL GetShort(void);

	// GetMarker() - reads a marker from the JPG
	int JXP_API_CALL GetMarker(int *Marker);

	// DecodeImage() - decodes the image into the Y, Cb and Cr channels
	int JXP_API_CALL DecodeImage(void);

	// InterpretJFIFSegment() - interprets the JFIF segment in the file
	int JXP_API_CALL InterpretJFIFSegment(void);

	// SkipUnknownSegment() - skips an unrecognised segment
	void JXP_API_CALL SkipUnknownSegment(int Marker);

	// ReadQuantisationTables() - reads the quantisation tables
	int JXP_API_CALL ReadQuantisationTables(void);

	// Decodes the baseline DCT frame
	int JXP_API_CALL DecodeFrame(void);

	// ReadHuffmanTables() - reads the Huffman tables
	int JXP_API_CALL ReadHuffmanTables(void);

	// DefineRestartInterval() - defines the restart interval
	void JXP_API_CALL DefineRestartInterval(void);

	// GetBit() - gets the next bit from the data stream
	int JXP_API_CALL GetBit(int *Bit);

	// ResetGetBit() - resets the get bit function
	void JXP_API_CALL ResetGetBit(void);

	// DecodeScan() - decodes the scan
	int JXP_API_CALL DecodeScan(void);

	// DecodeMCU() - decodes an MCU
	int JXP_API_CALL DecodeMCU(int MCU);

	// DecodeDCTCoefficients() - decodes the DCT coefficients
	int JXP_API_CALL DecodeDCTCoefficients(JXP_JPEG_Component *Component, int *Coefficients);

	// InverseDCT() - performs the inverse DCT (Discrete Cosine Transform)
	void JXP_API_CALL InverseDCT(int *ZigZag, unsigned char *QTable, int *Out);

	// InverseDCT1D() - performs a 1D fast inverse DCT
	void JXP_API_CALL InverseDCT1D(float *In, float *Out);

	// DecodeHuffman() - decodes the next Huffman-encoded value in the bitstream
	int JXP_API_CALL DecodeHuffman(JXP_JPEG_HuffmanTable *HTable, int *Value);

	// Receive(), Extend() - used during Huffman decoding
	int JXP_API_CALL Receive(int Num, int *Value);
	void JXP_API_CALL Extend(int *V, int T);
};

/*********************************************************************************************************/
// Global function prototypes
/*********************************************************************************************************/

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

/*********************************************************************************************************/

#endif // __JXP_JPEG_H
