/*********************************************************************************************************
*
*	JXP_JPEG.cpp
*
*	JPEG functionality of the JXP library
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

// Revision history:
// ----------------

// v2.01 - 22/1/2000
// No changes to this file in this version.
//
// v2.0 - 4/1/2000
// Major rewrite, including conversion to C++.
//
// v1.21 - 3/1/1998
// No changes to this file in this version.
//
// v1.2 - 2/1/1998
// No changes to this file in this version.
//
// v1.1 - 1/1/1998
// Fixed a bug with non-initialisation of restart_interval
// causing decoding more than one JPEG to die.
//
// v1.0 - 23/12/1997
// Initial release.

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h>

#include "JXP_JPEG.h"

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

#define JPG_MARKER		0xFF

#define JPG_SOI			0xD8	// Start of Image
#define JPG_EOI			0xD9	// End of Image
#define JPG_APP0		0xE0	// Application 0

#define JPG_SOF0		0xC0	// Start of Frame 0
#define JPG_SOF1		0xC1	// Start of Frame 1
#define JPG_SOF15		0xCF	// Start of Frame 15

#define JPG_SOS			0xDA	// Start of Scan

#define JPG_DQT			0xDB	// Define Quantisation Table
#define JPG_DHT			0xC4	// Define Huffman Table

#define JPG_DRI			0xDD	// Define Restart Interval

#define MIN(A,B)		((A)<(B)?(A):(B))
#define MAX(A,B)		((A)>(B)?(A):(B))
#define MAX3(A,B,C)		MAX(MAX((A),(B)),(C))

#define A1				0.707106781f
#define A2				0.541196100f
#define A3				0.707106781f
#define A4				1.306562965f
#define A5				0.382683432f

/*********************************************************************************************************/
// Local typedefs
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static function prototypes
/*********************************************************************************************************/

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Functions
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::DecodeJPEG()
*
*	desc:		Main interface function to decode a JPEG
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::DecodeJPEG(JXP_Data *Source, JXP_Image_24b **Destination)
{
	int i;

	// Set up data source
	this->Source=Source;
	CurrentPosition=0;

	// Initialise restart interval
	RestartInterval=0;

	// Initialise quantisation tables
	for (i=0; i<4; i++)
		QTable[i]=NULL;

	// Set Huffman tables to undefined
	for (i=0; i<4; i++)
	{
		DC_HTable[i].Defined=0;
		AC_HTable[i].Defined=0;
	}

	// Decode the image
	if (DecodeImage()!=JXP_SUCCESS)
	{
		JXP_Utils.Output("Error decoding file\n");
		return JXP_FAILURE;
	}

	// Free the quantisation tables
	for (i=0; i<4; i++)
	{
		if (QTable[i])
			free(QTable[i]);
	}

	// Free the Huffman tables
	for (i=0; i<4; i++)
	{
		if (DC_HTable[i].Defined)
		{
			free(DC_HTable[i].Huffval);
			free(DC_HTable[i].Huffsize);
			free(DC_HTable[i].Huffcode);
		}

		if (AC_HTable[i].Defined)
		{
			free(AC_HTable[i].Huffval);
			free(AC_HTable[i].Huffsize);
			free(AC_HTable[i].Huffcode);
		}
	}

	// Now to convert the Y, Cb and Cr planes to RGB

	unsigned char *Data;
	int Y_Hdiv,Y_Vdiv,Cb_Hdiv,Cb_Vdiv,Cr_Hdiv,Cr_Vdiv;
	int Temp,YTemp,CbTemp,CrTemp;
	int Hmax,Vmax;
	int ByteOrder;
	int Stride;
	int x,y;

	// Allocate the destination image
	JXP_Image_24b *Image=new JXP_Image_24b(Samples,Lines);
	Data=Image->GetData();

	// Compute some variables used during conversion

	Hmax=MAX3(Y.H,Cb.H,Cr.H);
	Vmax=MAX3(Y.V,Cb.V,Cr.V);

	Y_Hdiv=Hmax/Y.H;
	Y_Vdiv=Vmax/Y.V;
	Cb_Hdiv=Hmax/Cb.H;
	Cb_Vdiv=Vmax/Cb.V;
	Cr_Hdiv=Hmax/Cr.H;
	Cr_Vdiv=Vmax/Cr.V;

	Stride=Samples;
	Stride+=Hmax*8-1;
	Stride/=Hmax*8;
	Stride*=Hmax*8;

	ByteOrder=JXP_Utils.GetByteOrder();

	// Now do the conversion
	for (y=0; y<Lines; y++)
	{
		for (x=0; x<Samples; x++)
		{
			YTemp=*(Y.Data+(Stride/Y_Hdiv)*(y/Y_Vdiv)+(x/Y_Hdiv));
			CbTemp=*(Cb.Data+(Stride/Cb_Hdiv)*(y/Cb_Vdiv)+(x/Cb_Hdiv));
			CrTemp=*(Cr.Data+(Stride/Cr_Hdiv)*(y/Cr_Vdiv)+(x/Cr_Hdiv));

			Temp=(int)(YTemp+1.402f*(CrTemp-128.0f));
			if (Temp<0) Temp=0; if (Temp>255) Temp=255;
			if (ByteOrder==JXP_BYTEORDER_RGB) *(Data+3*(Samples*y+x))=Temp;
			else *(Data+3*(Samples*y+x)+2)=Temp;

			Temp=(int)(YTemp-0.34414f*(CbTemp-128.0f)-0.71414f*(CrTemp-128.0f));
			if (Temp<0) Temp=0; if (Temp>255) Temp=255;
			*(Data+3*(Samples*y+x)+1)=Temp;

			Temp=(int)(YTemp+1.772f*(CbTemp-128.0f));
			if (Temp<0) Temp=0; if (Temp>255) Temp=255;
			if (ByteOrder==JXP_BYTEORDER_RGB) *(Data+3*(Samples*y+x)+2)=Temp;
			else *(Data+3*(Samples*y+x))=Temp;
		}
	}

	// Free Y, Cb and Cr planes
	free(Y.Data);
	free(Cb.Data);
	free(Cr.Data);

	// Set destination
	*Destination=Image;

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::GetByte()
*
*	desc:		Gets the next byte from the source data
*
*	notes:
*
**********************************************************************************************************/

unsigned char JXP_API_CALL JXP_JPEG_Interface::GetByte(void)
{
	unsigned char Temp;

	if (CurrentPosition>=Source->GetLength())
	{
		// Print a message if we go past the end of the file - but since this
		// function can't report an error, we don't stop decoding.
		JXP_Utils.Output("Error - JPEG decoder has gone past end of file\n");
		return 0;
	}

	Temp=Source->GetData()[CurrentPosition];
	CurrentPosition++;
	return Temp;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::UngetByte()
*
*	desc:		Undoes the previous call to GetByte()
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_JPEG_Interface::UngetByte(void)
{
	CurrentPosition--;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::GetShort()
*
*	desc:		Gets the next 2 bytes in the data stream as a big-endian 16bit int
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::GetShort(void)
{
	int Value;

	Value=GetByte()<<8;
	Value|=GetByte();

	return Value;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::GetMarker()
*
*	desc:		Reads a marker from the JPG
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::GetMarker(int *Marker)
{
	int C;

	// We are reading a marker from the file, so there is an error
	// if the next byte is not 0xFF (JPG_MARKER)
	C=GetByte();
	if (C!=JPG_MARKER)
	{
		JXP_Utils.Output("Error - expected byte 0xFF, found 0x%02X\n",C);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	*Marker=GetByte();

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::DecodeImage()
*
*	desc:		Decodes the image into the Y, Cb and Cr channels
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::DecodeImage(void)
{
	int Marker;

	if (GetMarker(&Marker)!=JXP_SUCCESS)
	{
		JXP_Utils.Output("Error - a marker was expected, but not found\n");
		return JXP_FAILURE;
	}

	if (Marker!=JPG_SOI)
	{
		JXP_Utils.Output("Error - expecting SOI marker, instead found 0x%02X\n",Marker);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	if (GetMarker(&Marker)!=JXP_SUCCESS)
	{
		JXP_Utils.Output("Error - a marker was expected, but not found\n");
		return JXP_FAILURE;
	}

	if (Marker!=JPG_APP0)
	{
		JXP_Utils.Output("Error - expecting APP0 marker, instead found 0x%02X\n",Marker);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	if (InterpretJFIFSegment()!=JXP_SUCCESS)
	{
		JXP_Utils.Output("Error - first APP0 marker is invalid\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Now loop over markers, searching for the SOF0 marker

	for (;;)
	{
		if (GetMarker(&Marker)!=JXP_SUCCESS)
		{
			JXP_Utils.Output("Error - a marker was expected, but not found\n");
			return JXP_FAILURE;
		}

		if (Marker==JPG_SOF0)
		{
			if (DecodeFrame()!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - decode frame failed\n");
				return JXP_FAILURE;
			}

			// Once we've returned from decoding the frame, we have finished!
			break;
		}
		else if (Marker>=JPG_SOF1 && Marker<=JPG_SOF15 && Marker!=JPG_DHT)
		{
			JXP_Utils.Output("Error - this programme can not decode a frame of type 0x%X\n",Marker);
			JXP_Utils.SetLastError(JXP_ERROR_NOT_SUPPORTED);
			return JXP_FAILURE;
		}
		else if (Marker==JPG_DQT)
		{
			if (ReadQuantisationTables()!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - reading quantisation tables failed\n");
				return JXP_FAILURE;
			}
		}
		else if (Marker==JPG_DHT)
		{
			if (ReadHuffmanTables()!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - reading Huffman tables failed\n");
				return JXP_FAILURE;
			}
		}
		else if (Marker==JPG_DRI)
		{
			DefineRestartInterval();
		}
		else
		{
			// Unkown marker
			SkipUnknownSegment(Marker);
		}
	}

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::InterpretJFIFSegment()
*
*	desc:		Interprets the JFIF segment in the JPG file
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::InterpretJFIFSegment(void)
{
	char Header[5];
	int Length;
	int Major;
	int Minor;
	int i;

	Length=GetShort();

	JXP_Utils.Output("Found APP0 marker of length %d\n",Length);

	// Verify Header
	for (i=0; i<5; i++)
		Header[i]=GetByte();

	if (strcmp(Header,"JFIF"))
	{
		JXP_Utils.Output("Error - not a JFIF file\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	JXP_Utils.Output("Verified JFIF format\n");

	// Print out version number
	Major=GetByte();
	Minor=GetByte();
	JXP_Utils.Output("JFIF version: %d.%02d\n",Major,Minor);

	// Skip rest of JFIF header
	for (i=0; i<Length-9; i++)
		GetByte();

	JXP_Utils.Output("\n");

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::SkipUnknownSegment()
*
*	desc:		Skips an unrecognised segment
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_JPEG_Interface::SkipUnknownSegment(int Marker)
{
	int Length;
	int i;

	Length=GetShort();

	JXP_Utils.Output("Found unrecognised marker (0x%02X) of length %d. Skipping\n\n",Marker,Length);

	for (i=0; i<Length-2; i++) GetByte();
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::ReadQuantisationTables()
*
*	desc:		Reads in the quantisation tables
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::ReadQuantisationTables(void)
{
	int Length;

	Length=GetShort();
	JXP_Utils.Output("Found Quantisation tables of length %0d\n",Length);

	for (;;)
	{
		int C,ID,i;

		C=GetByte();
		if (C==JPG_MARKER)
		{
			// End of quantisation table definition
			UngetByte(); // so marker will be read on next read
			JXP_Utils.Output("\n");
			return JXP_SUCCESS;
		}

		if (((C&0xF0)>>4)!=0)
		{
			JXP_Utils.Output("Error - quantisation table must be 8 bits, found a Pq of %0d\n",(C&0xF0)>>4);
			JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
			return JXP_FAILURE;
		}
        
		ID=(C&0x0F);
		if (ID<0 || ID>=4)
		{
			JXP_Utils.Output("Error - quantisation table id must be in range 0 to 3, found %0d\n",ID);
			JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
			return JXP_FAILURE;
		}

		JXP_Utils.Output("Found quantisation table %0d\n",ID);

		QTable[ID]=(unsigned char *)malloc(64);
		if (!QTable[ID])
		{
			JXP_Utils.Output("Error - insufficient memory for quantisation table");
			JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
			return JXP_FAILURE;
		}

		for (i=0; i<64; i++) 
			*(QTable[ID]+i)=GetByte();
	}

	// Shouldn't get to here
	return JXP_FAILURE;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::DecodeFrame()
*
*	desc:		Decodes the baseline DCT frame
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::DecodeFrame(void)
{
	int Length;
	int C;

	// First decode the frame header

	Length=GetShort();
	JXP_Utils.Output("Found start of baseline DCT frame of length %0d\n",Length);

	// Check precision
	C=GetByte();
	if (C!=8)
	{
		JXP_Utils.Output("Error - 8 bit precision allowed only, found %0d\n",C);
		JXP_Utils.SetLastError(JXP_ERROR_NOT_SUPPORTED);
		return JXP_FAILURE;
	}

	// Get image dimensions
	Lines=GetShort();
	Samples=GetShort();
	JXP_Utils.Output("Image dimensions: %0dx%0d\n",Samples,Lines);
    
	// Check number of components (must be 3, one each for Y, Cb and Cr)
	C=GetByte();
	if (C!=3)
	{
		JXP_Utils.Output("Error - 3 components allowed only, found %0d\n",C);
		JXP_Utils.SetLastError(JXP_ERROR_NOT_SUPPORTED);
		return JXP_FAILURE;
	}
    
	// Read component ID info
    
	// Y component:

	C=GetByte();
	Y.ID=C;
	if (Y.ID!=1)
	{
		JXP_Utils.Output("Error - the first component (Y) should have an id of 1, found %0d\n",Y.ID);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	C=GetByte();
	Y.H=(C&0xF0)>>4;
	Y.V=(C&0x0F);
	C=GetByte();
	Y.QT=C;

	// Cb component:

	C=GetByte();
	Cb.ID=C;
	if (Cb.ID!=2)
	{
		JXP_Utils.Output("Error - the second component (Cb) should have an id of 2, found %0d\n",Cb.ID);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	C=GetByte();
	Cb.H=(C&0xF0)>>4;
	Cb.V=(C&0x0F);
	C=GetByte();
	Cb.QT=C;

	// Cr component:

	C=GetByte();
	Cr.ID=C;
	if (Cr.ID!=3)
	{
		JXP_Utils.Output("Error - the third component (Cr) should have an id of 3, found %0d\n",Cr.ID);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	C=GetByte();
	Cr.H=(C&0xF0)>>4;
	Cr.V=(C&0x0F);
	C=GetByte();
	Cr.QT=C;

	JXP_Utils.Output("Y  - id: %0d, H: %0d, V: %0d, qt: %0d\n",Y.ID,Y.H,Y.V,Y.QT);
	JXP_Utils.Output("Cb - id: %0d, H: %0d, V: %0d, qt: %0d\n",Cb.ID,Cb.H,Cb.V,Cb.QT);
	JXP_Utils.Output("Cr - id: %0d, H: %0d, V: %0d, qt: %0d\n",Cr.ID,Cr.H,Cr.V,Cr.QT);
	JXP_Utils.Output("\n");

	// Now parse any more markers, decoding scans as they appear

	for(;;)
	{
		int Marker;

		if (GetMarker(&Marker)!=JXP_SUCCESS)
		{
			JXP_Utils.Output("Error - a marker was expected, but not found\n");
			return JXP_FAILURE;
		}

		if (Marker==JPG_SOS)
		{
			if (DecodeScan()!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - decode scan failed\n");
				return JXP_FAILURE;
			}
		}
		else if (Marker==JPG_EOI)
		{
			// End of image!
			JXP_Utils.Output("\n");
			return JXP_SUCCESS;
		}
		// The rest of these are the same as in DecodeImage()
		else if (Marker==JPG_DQT)
		{
			if (ReadQuantisationTables()!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - reading quantisation tables failed\n");
				return JXP_FAILURE;
			}
		}
		else if (Marker==JPG_DHT)
		{
			if (ReadHuffmanTables()!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - reading Huffman tables failed\n");
				return JXP_FAILURE;
			}
		}
		else if (Marker==JPG_DRI)
		{
			DefineRestartInterval();
		}
		else
		{
			// Unkown marker
			SkipUnknownSegment(Marker);
		}
	}

	// Shouldn't ever get to here
	return JXP_FAILURE;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::ReadHuffmanTables()
*
*	desc:		Reads the Huffman tables
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::ReadHuffmanTables(void)
{
	int Length;

	Length=GetShort();
	JXP_Utils.Output("Found Huffman tables of length %0d\n",Length);
    
	for (;;)
	{
		JXP_JPEG_HuffmanTable *HTable;
		int C,TC,ID;

		C=GetByte();
		if (C==JPG_MARKER)
		{
			// end of Huffman table definition
			UngetByte(); // so marker will be read on next read
			JXP_Utils.Output("\n");
			return JXP_SUCCESS;
		}

		// Get table class
		TC=(C&0xF0)>>4;
		if (TC==0)
		{
			JXP_Utils.Output("Found DC Huffman table ");
		}
		else if (TC==1) 
		{
			JXP_Utils.Output("Found AC Huffman table ");
		}
		else 
		{
			JXP_Utils.Output("Error - table class can only have a value of 0 or 1, found %0d\n",TC);
			JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
			return JXP_FAILURE;
		}

		// Get ID
		ID=(C&0x0F);
		JXP_Utils.Output("%0d\n",ID);

		// Set up pointer to the Huffman table to fill
		HTable=(TC?AC_HTable+ID:DC_HTable+ID);

		HTable->Defined=1;

		int Num=0;
		int i,j,k;

		for (i=0; i<16; i++)
		{
			HTable->Bits[i]=GetByte();
			Num+=HTable->Bits[i];
		}

		// Num is now the number of entires in the table
		HTable->Huffval=(unsigned char *)malloc(Num);
		if (!HTable->Huffval)
		{
			JXP_Utils.Output("Error - insufficient memory to hold Huffman tables\n");
			JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
			return JXP_FAILURE;
		}

		// Read in the table entries
		for (i=0; i<Num; i++)
		{
			HTable->Huffval[i]=GetByte();
		}

		// Finished reading this Huffman table from file. Should now
		// calculate the other Huffman table entries which the decoder
		// uses.

		// First calculate the Huffsize(0..Num) array, which is the
		// size of each entry in Huffval(0..Num-1) in bits, with
		// huffsize(Num)=0

		HTable->Huffsize=(int *)malloc(sizeof(int)*(Num+1));
		if (!HTable->Huffsize)
		{
			JXP_Utils.Output("Error - insufficient memory to hold Huffman tables\n");
			JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
			return JXP_FAILURE;
		}

		i=1;
		j=1; 
		k=0; 

		for (;;)
		{
			if (j>HTable->Bits[i-1])
			{
				i++;
				j=1;
			}
			else
			{
				HTable->Huffsize[k]=i;
				k++;
				j++;
				continue;
			}

			if (i>16)
			{
				HTable->Huffsize[k]=0;
				break;
			}
		}

		// Now to create Huffcode(0..num-1) array, which contains the
		// actual Huffman code corresponding to the entry in 
		// Huffval(0..num-1).

		HTable->Huffcode=(int *)malloc(sizeof(int)*Num);
		if (!HTable->Huffcode)
		{
			JXP_Utils.Output("Error - insufficient memory to hold Huffman tables\n");
			JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
			return JXP_FAILURE;
		}

		int Code=0;
		int SI;

		k=0;

		for (SI=HTable->Huffsize[0]; SI<=16; SI++)
		{
			while (SI==HTable->Huffsize[k])
			{
				HTable->Huffcode[k]=Code;
				Code++;
				k++;
			}

			Code=Code<<1;
		}

		// Now to create the Mincode(0..15), Maxcode(0..15) and
		// Valptr(0..15) arrays. Mincode contains the smallest value
		// code of each code length, Maxcode the largest code of each
		// code length, and Valptr contains the index of the start of
		// the codes in huffcode of each length. These tables are
		// used for decoding.

		j=0;

		for (i=0; i<16; i++)
		{
			if (HTable->Bits[i]==0)
			{
				HTable->Maxcode[i]=-1;
				continue;
			}

			HTable->Valptr[i]=j;
			HTable->Mincode[i]=HTable->Huffcode[j];
			j+=HTable->Bits[i]-1;
			HTable->Maxcode[i]=HTable->Huffcode[j];
			j++;
		}
	}

	// Shouldn't get to here
	return JXP_FAILURE;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::DefineRestartInterval()
*
*	desc:		Defines the restart interval
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_JPEG_Interface::DefineRestartInterval(void)
{
	// Skip the length field, always 4
	GetShort();

	// Read the restart interval
	RestartInterval=GetShort();
	
	JXP_Utils.Output("Found restart interval marker, set to %0d\n\n",RestartInterval);
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::GetBit()
*
*	desc:		Gets the next bit from the data stream
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::GetBit(int *Bit)
{
	int Temp;

	if (BitCount==0)
	{
		BitB=GetByte();
		BitCount=8;

		if (BitB==0xFF)
		{
			// Found 0xFF, will have to skip a stuffed byte
			Temp=GetByte();
			if (Temp!=0)
			{
				JXP_Utils.Output("Error - get bit expected a stuffed byte of 0, found 0x%X\n",Temp);
				JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
				return JXP_FAILURE;
			}
		}
	}
  
	*Bit=BitB>>7;
	BitCount--;
	BitB=(BitB<<1)&0xFF;
  
	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::ResetGetBit()
*
*	desc:		Resets the get bit function
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_JPEG_Interface::ResetGetBit(void)
{
	BitCount=0;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::DecodeScan()
*
*	desc:		Decodes the scan
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::DecodeScan(void)
{
	int Length;
	int C,TS;

	// First interpret the scan header

	Length=GetShort();
	JXP_Utils.Output("Found start of scan of length %0d\n",Length);

	C=GetByte();
	if (C!=3)
	{
		JXP_Utils.Output("Error - expected 3 components in the scan, found %0d\n",C);
		JXP_Utils.SetLastError(JXP_ERROR_NOT_SUPPORTED);
		return JXP_FAILURE;
	}

	// Read table selectors for Y component:

	C=GetByte();
	if (C!=1)
	{
		JXP_Utils.Output("Error - expected a component id of 1 in the scan, found %0d\n",C);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	C=GetByte();
	TS=(C&0xF0)>>4; // DC table selector
	if (!DC_HTable[TS].Defined)
	{
		JXP_Utils.Output("Error - DC Huffman table %0d has not been defined\n",TS);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	Y.DC_HTable=DC_HTable+TS;
	JXP_Utils.Output("Y  - DC: %0d ",TS);

	TS=(C&0x0F); // AC table selector
	if (!AC_HTable[TS].Defined)
	{
		JXP_Utils.Output("Error - AC Huffman table %0d has not been defined\n",TS);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	Y.AC_HTable=AC_HTable+TS;
	JXP_Utils.Output("AC: %0d \n",TS);

	// Read table selectors for Cb component:

	C=GetByte();
	if (C!=2)
	{
		JXP_Utils.Output("Error - expected a component id of 2 in the scan, found %0d\n",C);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	C=GetByte();
	TS=(C&0xF0)>>4; // DC table selector
	if (!DC_HTable[TS].Defined)
	{
		JXP_Utils.Output("Error - DC Huffman table %0d has not been defined\n",TS);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	Cb.DC_HTable=DC_HTable+TS;
	JXP_Utils.Output("Cb - DC: %0d ",TS);

	TS=(C&0x0F);
	if (!AC_HTable[TS].Defined)
	{
		JXP_Utils.Output("Error - AC Huffman table %0d has not been defined\n",TS);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	Cb.AC_HTable=AC_HTable+TS;
	JXP_Utils.Output("AC: %0d \n",TS);

	// Read table selectors for Cr component:

	C=GetByte();
	if (C!=3)
	{
		JXP_Utils.Output("Error - expected a component id of 3 in the scan, found %0d\n",C);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	C=GetByte();
	TS=(C&0xF0)>>4; // DC table selector
	if (!DC_HTable[TS].Defined)
	{
		JXP_Utils.Output("Error - DC Huffman table %0d has not been defined\n",TS);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	Cr.DC_HTable=DC_HTable+TS;
	JXP_Utils.Output("Cr - DC: %0d ",TS);

	TS=(C&0x0F);
	if (!AC_HTable[TS].Defined)
	{
		JXP_Utils.Output("Error - AC Huffman table %0d has not been defined\n",TS);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	Cr.AC_HTable=AC_HTable+TS;
	JXP_Utils.Output("AC: %0d \n",TS);

	// Finished reading the table selectors of the three components. Now setup the
	// quantisation tables which each component uses

	if (!QTable[Y.QT])
	{
		JXP_Utils.Output("Error - quantisation table %0d has not been defined\n",Y.QT);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	Y.QTable=QTable[Y.QT];
	if (!QTable[Cb.QT])
	{
		JXP_Utils.Output("Error - quantisation table %0d has not been defined\n",Cb.QT);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	Cb.QTable=QTable[Cb.QT];
	if (!QTable[Cr.QT])
	{
		JXP_Utils.Output("Error - quantisation table %0d has not been defined\n",Cr.QT);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}
	Cr.QTable=QTable[Cr.QT];

	// The last 3 bytes in the scan header should be 0,63,0 default values
	// for a baseline sequential DCT scan.

	if (GetByte()!=0 || GetByte()!=63 || GetByte()!=0)
	{
		JXP_Utils.Output("Error - last 3 bytes of scan header have invalid values\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
		return JXP_FAILURE;
	}

	// Compute numbers of MCU in horizontal and vertical directions, as well as total
	int Hmax,Vmax,TotalMCU;

	Hmax=MAX3(Y.H,Cb.H,Cr.H);
	Vmax=MAX3(Y.V,Cb.V,Cr.V);

	H_MCU=(Samples+8*Hmax-1)/(8*Hmax);
	V_MCU=(Lines+8*Vmax-1)/(8*Vmax);

	TotalMCU=H_MCU*V_MCU;

	JXP_Utils.Output("MCU: H %0d, V: %0d, total: %0d\n",H_MCU,V_MCU,TotalMCU);

	// Now allocate space for each of the components to write their
	// colour plane to

	Y.Data=(unsigned char *)malloc(TotalMCU*64*Y.H*Y.V);
	if (!Y.Data)
	{
		JXP_Utils.Output("Error - insufficient memory to hold decoded Y component\n");
		JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
		return JXP_FAILURE;
	}

	Cb.Data=(unsigned char *)malloc(TotalMCU*64*Cb.H*Cb.V);
	if (!Cb.Data)
	{
		JXP_Utils.Output("Error - insufficient memory to hold decoded Cb component\n");
		JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
		return JXP_FAILURE;
	}

	Cr.Data=(unsigned char *)malloc(TotalMCU*64*Cr.H*Cr.V);
	if (!Cr.Data)
	{
		JXP_Utils.Output("Error - insufficient memory to hold decoded Cr component\n");
		JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
		return JXP_FAILURE;
	}

	// Now to actually decode the MCU

	// Initialise DC predictors to 0
	Y.Predictor=Cb.Predictor=Cr.Predictor=0;

	ResetGetBit();

	JXP_Utils.Output("\nDecoding MCU ");

	int RestartExpected=0;
	int i;

	if (RestartInterval==0)
	{
		// Just decode all the MCU
		for (i=0; i<TotalMCU; i++)
		{
			// Print out dots as progress
			if (TotalMCU>=40)
			{
				if (i%(TotalMCU/40)==0)
					JXP_Utils.Output(".");
			}

			// Decode MCU
			if (DecodeMCU(i)!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - decoding MCU failed\n");
				return JXP_FAILURE;
			}
		}
	}
	else
	{
		// Need to stop occasionally for restart markers
		for (i=0; i<TotalMCU; i++)
		{
			// Print out dots as progress
			if (TotalMCU>=40)
			{
				if (i%(TotalMCU/40)==0)
					JXP_Utils.Output(".");
			}

			// Decode MCU
			if (DecodeMCU(i)!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - decoding MCU failed\n");
				return JXP_FAILURE;
			}

			// Check if we should expect a restart marker
			if (((i%RestartInterval)==(RestartInterval-1)) && (i!=(TotalMCU-1)))
			{
				// Time for a restart marker

				int Marker;
				if (GetMarker(&Marker)!=JXP_SUCCESS)
				{
					JXP_Utils.Output("Error - a marker was expected, but not found\n");
					return JXP_FAILURE;
				}

				if ((Marker&0xF8)!=0xD0) // RST = 0xD*
				{
					JXP_Utils.Output("Error - expected an RSTm marker, found 0x%02X\n",Marker);
					JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
					return JXP_FAILURE;
				}

				if ((Marker&0x0F)!=RestartExpected)
				{
					JXP_Utils.Output("Error - expected restart marker RST%0d, found RST%0d\n",RestartExpected,(Marker&0x0F));
					JXP_Utils.SetLastError(JXP_ERROR_INVALID_DATA);
					return JXP_FAILURE;
				}

				RestartExpected=(RestartExpected+1)%8;

				// Reset DC predictors
				Y.Predictor=Cb.Predictor=Cr.Predictor=0;

				ResetGetBit();
			}
		}
	}

	// Have now decoded all the MCU
  
	JXP_Utils.Output("\n");

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::DecodeMCU()
*
*	desc:		Decodes an MCU
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::DecodeMCU(int MCU)
{
	int DCT_Coefficients[64];
	int IDCT_Values[64];
	int XOff,YOff,x,y;
	int i,j;

	// First decode the HxV DCT's for the Y component

	for (j=0; j<Y.V; j++)
	{
		for (i=0; i<Y.H; i++)
		{
			if (DecodeDCTCoefficients(&Y,DCT_Coefficients)!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - decode DCT coefficients failed\n");
				return JXP_FAILURE;
			}

			// Do inverse DCT
			InverseDCT(DCT_Coefficients,Y.QTable,IDCT_Values);

			// Compute x and y offset of the top left corner of this MCU in Y.data
			XOff=8*Y.H*(MCU%H_MCU);
			YOff=8*Y.V*(MCU/H_MCU);

			XOff+=8*i;
			YOff+=8*j;

			for (y=0; y<8; y++)
			{
				for (x=0; x<8; x++)
					*(Y.Data+(YOff+y)*8*Y.H*H_MCU+(XOff+x))=*(IDCT_Values+8*y+x);
			}
		}
	}

	// Now the same for Cb

	for (j=0; j<Cb.V; j++)
	{
		for (i=0; i<Cb.H; i++)
		{
			if (DecodeDCTCoefficients(&Cb,DCT_Coefficients)!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - decode DCT coefficients failed\n");
				return JXP_FAILURE;
			}

			// Do inverse DCT
			InverseDCT(DCT_Coefficients,Cb.QTable,IDCT_Values);

			// Compute x and y offset of the top left corner of this MCU in Cb.data
			XOff=8*Cb.H*(MCU%H_MCU);
			YOff=8*Cb.V*(MCU/H_MCU);

			XOff+=8*i;
			YOff+=8*j;

			for (y=0; y<8; y++)
			{
				for (x=0; x<8; x++)
					*(Cb.Data+(YOff+y)*8*Cb.H*H_MCU+(XOff+x))=*(IDCT_Values+8*y+x);
			}
		}
	}

	// And finally for Cr

	for (j=0; j<Cr.V; j++)
	{
		for (i=0; i<Cr.H; i++)
		{
			if (DecodeDCTCoefficients(&Cr,DCT_Coefficients)!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - decode DCT coefficients failed\n");
				return JXP_FAILURE;
			}

			// Do inverse DCT
			InverseDCT(DCT_Coefficients,Cr.QTable,IDCT_Values);

			// Compute x and y offset of the top left corner of this MCU in Cb.data
			XOff=8*Cr.H*(MCU%H_MCU);
			YOff=8*Cr.V*(MCU/H_MCU);

			XOff+=8*i;
			YOff+=8*j;

			for (y=0; y<8; y++)
			{
				for (x=0; x<8; x++)
					*(Cr.Data+(YOff+y)*8*Cr.H*H_MCU+(XOff+x))=*(IDCT_Values+8*y+x);
			}
		}
	}

	// Now finished decoding the MCU

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::DecodeDCTCoefficients()
*
*	desc:		Decodes the DCT coefficients
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::DecodeDCTCoefficients(JXP_JPEG_Component *Component, int *Coefficients)
{
	int Diff;
	int rs,r,s,t,k;

	// First decode the DC value.

	if (DecodeHuffman(Component->DC_HTable,&t)!=JXP_SUCCESS)
	{
		JXP_Utils.Output("Error - Huffman decode failed\n");
		return JXP_FAILURE;
	}

	if (Receive(t,&Diff)!=JXP_SUCCESS)
	{
		JXP_Utils.Output("Error - receive failed\n");
		return JXP_FAILURE;
	}

	Extend(&Diff,t);

	Component->Predictor+=Diff;
	Coefficients[0]=Component->Predictor;

	// Now decode the 63 AC values

	memset(Coefficients+1,0,sizeof(int)*63);

	k=1;

	for (;;)
	{
		if (DecodeHuffman(Component->AC_HTable,&rs)!=JXP_SUCCESS)
		{
			JXP_Utils.Output("Error - Huffman decode failed\n");
			return JXP_FAILURE;
		}

		s=rs%16;
		r=rs>>4;

		if (s==0)
		{
			if (r==15)
			{
				k+=16;
				continue;
			}
			else
			{
				break;
			}
		}
    
		k+=r;

		if (Receive(s,&(Coefficients[k]))!=JXP_SUCCESS)
		{
			JXP_Utils.Output("Error - receive failed\n");
			return JXP_FAILURE;
		}

		Extend(&(Coefficients[k]),s);
    
		if (k==63) break;

		k++;
	}

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::InverseDCT()
*
*	desc:		Performs the inverse DCT (Discrete Cosine Transform)
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_JPEG_Interface::InverseDCT(int *ZigZag, unsigned char *QTable, int *Out)
{
	float Columns[64];
	float Rows[64];
	float Temp[8];
	int i,j;

	// First do an IDCT on the columns:

	// Column 0
	Temp[0]=(float)ZigZag[0]*(float)QTable[0];
	Temp[1]=(float)ZigZag[2]*(float)QTable[2];
	Temp[2]=(float)ZigZag[3]*(float)QTable[3];
	Temp[3]=(float)ZigZag[9]*(float)QTable[9];
	Temp[4]=(float)ZigZag[10]*(float)QTable[10];
	Temp[5]=(float)ZigZag[20]*(float)QTable[20];
	Temp[6]=(float)ZigZag[21]*(float)QTable[21];
	Temp[7]=(float)ZigZag[35]*(float)QTable[35];

	InverseDCT1D(Temp,Columns);

	// Column 1
	Temp[0]=(float)ZigZag[1]*(float)QTable[1];
	Temp[1]=(float)ZigZag[4]*(float)QTable[4];
	Temp[2]=(float)ZigZag[8]*(float)QTable[8];
	Temp[3]=(float)ZigZag[11]*(float)QTable[11];
	Temp[4]=(float)ZigZag[19]*(float)QTable[19];
	Temp[5]=(float)ZigZag[22]*(float)QTable[22];
	Temp[6]=(float)ZigZag[34]*(float)QTable[34];
	Temp[7]=(float)ZigZag[36]*(float)QTable[36];

	InverseDCT1D(Temp,Columns+8);

	// Column 2
	Temp[0]=(float)ZigZag[5]*(float)QTable[5];
	Temp[1]=(float)ZigZag[7]*(float)QTable[7];
	Temp[2]=(float)ZigZag[12]*(float)QTable[12];
	Temp[3]=(float)ZigZag[18]*(float)QTable[18];
	Temp[4]=(float)ZigZag[23]*(float)QTable[23];
	Temp[5]=(float)ZigZag[33]*(float)QTable[33];
	Temp[6]=(float)ZigZag[37]*(float)QTable[37];
	Temp[7]=(float)ZigZag[48]*(float)QTable[48];

	InverseDCT1D(Temp,Columns+16);

	// Column 3
	Temp[0]=(float)ZigZag[6]*(float)QTable[6];
	Temp[1]=(float)ZigZag[13]*(float)QTable[13];
	Temp[2]=(float)ZigZag[17]*(float)QTable[17];
	Temp[3]=(float)ZigZag[24]*(float)QTable[24];
	Temp[4]=(float)ZigZag[32]*(float)QTable[32];
	Temp[5]=(float)ZigZag[38]*(float)QTable[38];
	Temp[6]=(float)ZigZag[47]*(float)QTable[47];
	Temp[7]=(float)ZigZag[49]*(float)QTable[49];

	InverseDCT1D(Temp,Columns+24);

	// Column 4
	Temp[0]=(float)ZigZag[14]*(float)QTable[14];
	Temp[1]=(float)ZigZag[16]*(float)QTable[16];
	Temp[2]=(float)ZigZag[25]*(float)QTable[25];
	Temp[3]=(float)ZigZag[31]*(float)QTable[31];
	Temp[4]=(float)ZigZag[39]*(float)QTable[39];
	Temp[5]=(float)ZigZag[46]*(float)QTable[46];
	Temp[6]=(float)ZigZag[50]*(float)QTable[50];
	Temp[7]=(float)ZigZag[57]*(float)QTable[57];

	InverseDCT1D(Temp,Columns+32);

	// Column 5
	Temp[0]=(float)ZigZag[15]*(float)QTable[15];
	Temp[1]=(float)ZigZag[26]*(float)QTable[26];
	Temp[2]=(float)ZigZag[30]*(float)QTable[30];
	Temp[3]=(float)ZigZag[40]*(float)QTable[40];
	Temp[4]=(float)ZigZag[45]*(float)QTable[45];
	Temp[5]=(float)ZigZag[51]*(float)QTable[51];
	Temp[6]=(float)ZigZag[56]*(float)QTable[56];
	Temp[7]=(float)ZigZag[58]*(float)QTable[58];

	InverseDCT1D(Temp,Columns+40);

	// Column 6
	Temp[0]=(float)ZigZag[27]*(float)QTable[27];
	Temp[1]=(float)ZigZag[29]*(float)QTable[29];
	Temp[2]=(float)ZigZag[41]*(float)QTable[41];
	Temp[3]=(float)ZigZag[44]*(float)QTable[44];
	Temp[4]=(float)ZigZag[52]*(float)QTable[52];
	Temp[5]=(float)ZigZag[55]*(float)QTable[55];
	Temp[6]=(float)ZigZag[59]*(float)QTable[59];
	Temp[7]=(float)ZigZag[62]*(float)QTable[62];

	InverseDCT1D(Temp,Columns+48);

	// Column 7
	Temp[0]=(float)ZigZag[28]*(float)QTable[28];
	Temp[1]=(float)ZigZag[42]*(float)QTable[42];
	Temp[2]=(float)ZigZag[43]*(float)QTable[43];
	Temp[3]=(float)ZigZag[53]*(float)QTable[53];
	Temp[4]=(float)ZigZag[54]*(float)QTable[54];
	Temp[5]=(float)ZigZag[60]*(float)QTable[60];
	Temp[6]=(float)ZigZag[61]*(float)QTable[61];
	Temp[7]=(float)ZigZag[63]*(float)QTable[63];

	InverseDCT1D(Temp,Columns+56);

	// Now on the rows

	for (i=0; i<8; i++)
	{
		for (j=0; j<8; j++)
		{
			Temp[j]=Columns[j*8+i];
		}
		InverseDCT1D(Temp,Rows+8*i);
	}

	// Now *rows is an 8x8 array with the inverse transformed values -
	// convert to integer values

	for (i=0; i<64; i++)
	{
		Rows[i]+=128.0;

		if (Rows[i]<0) 
			Out[i]=0;
		else if (Rows[i]>255) 
			Out[i]=255;
		else 
			Out[i]=(int)Rows[i];
	}
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::InverseDCT1D()
*
*	desc:		Performs a 1D fast inverse DCT
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_JPEG_Interface::InverseDCT1D(float *In, float *Out)
{
	float v11,v21,v31,v41,v51,v61,v71,v81;
	float v52,v62,v72,v82;
	float v33,v43,v63,v83, v9a,v9b;
	float v34,v54,v64,v74;
	float v15,v25,v45;
	float v16,v26,v36,v46,v56,v66,v76,v86;

	v11=In[0]/2.828427125f;
	v21=In[4]/4.0f;
	v31=In[2]/4.0f;
	v41=In[6]/4.0f;
	v51=In[5]/4.0f;
	v61=In[1]/4.0f;
	v71=In[7]/4.0f;
	v81=In[3]/4.0f;

	v52=v51-v81;
	v62=v61+v71;
	v72=v61-v71;
	v82=v51+v81;
  
	v33=v31-v41;
	v43=v31+v41;
	v63=v62-v82;
	v83=v62+v82;
	v9a=-v52-v72;
	v9b=v9a*A5;

	v34=v33*A1;
	v54=-v52*A2+v9b;
	v64=v63*A3;
	v74=v72*A4+v9b;

	v15=v11+v21;
	v25=v11-v21;
	v45=v34+v43;

	v16=v15+v45;
	v26=v25+v34;
	v36=v25-v34;
	v46=v15-v45;
	v56=-v54;
	v66=-v54+v64;
	v76=v64+v74;
	v86=v74+v83;

	Out[0]=v16+v86;
	Out[1]=v26+v76;
	Out[2]=v36+v66;
	Out[3]=v46+v56;
	Out[4]=v46-v56;
	Out[5]=v36-v66;
	Out[6]=v26-v76;
	Out[7]=v16-v86;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::DecodeHuffman()
*
*	desc:		Decodes the next huffman-encoded value in the bitstream
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::DecodeHuffman(JXP_JPEG_HuffmanTable *HTable, int *Value)
{
	int Code,Temp;
	int i,j;


	i=0;
	if (GetBit(&Code)!=JXP_SUCCESS)
	{
		JXP_Utils.Output("Error - get bit failed\n");
		return JXP_FAILURE;
	}

	for (;;)
	{
		if (Code>HTable->Maxcode[i])
		{
			i++;

			if (GetBit(&Temp)!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - get bit failed\n");
				return JXP_FAILURE;
			}

			Code=(Code<<1)+Temp;
		}
		else
		{
			j=HTable->Valptr[i];
			j+=Code-HTable->Mincode[i];
			*Value=HTable->Huffval[j];
			break; // found *Value
		}
	}

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::Receive()
*
*	desc:		Used during Huffman decoding
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_JPEG_Interface::Receive(int Num, int *Value)
{
	int Temp,i;

	*Value=0;
	for (i=0; i<Num; i++)
	{
		if (GetBit(&Temp)!=JXP_SUCCESS)
		{
			JXP_Utils.Output("Error - get bit failed\n");
			return JXP_FAILURE;
		}
		*Value=((*Value)<<1)+Temp;
	}

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_JPEG_Interface::Extend()
*
*	desc:		Used during Huffman decoding
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_JPEG_Interface::Extend(int *V, int T)
{
	int VT;

	VT=(1<<(T-1));

	if ((*V)<VT)
	{
		VT=((-1)<<T)+1;
		(*V)+=VT;
	}
}

/*********************************************************************************************************/
