/*********************************************************************************************************
*
*	JXP_Dither.h
*
*	Handles colour reduction and dithering for the JXP library
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

// Revision history:
// ----------------

// v2.01 - 22/1/2000
// No changes to this file in this version.
//
// v2.0 - 4/1/2000
// Major rewrite, including conversion to C++.
//
// v1.21 - 3/1/1998
// Made max recursion depth for colour reduction a configurable parameter
// (note that in version 2.0 this parameter is no longer available),
// plus added 'progress meter' for colour reduction.
//
// v1.2 - 2/1/1998
// No changes to this file in this version.
//
// v1.1 - 1/1/1998
// Initial inclusion of colour reduction and dithering.

#ifndef __JXP_DITHER_H
#define __JXP_DITHER_H

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include "JXP.h"

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

/*********************************************************************************************************/
// Global typedefs
/*********************************************************************************************************/

// JXP_Dither_RGB - RGB triplet class used during colour reduction
class JXP_Dither_RGB
{
public:

	// The RGB data
	int R,G,B;

	// Constructors
	JXP_Dither_RGB(int R, int G, int B) { this->R=R; this->G=G; this->B=B; }
	JXP_Dither_RGB(void) { R=0; G=0; B=0; }

	// Add and subtract operators
	JXP_Dither_RGB JXP_API_CALL operator+(const JXP_Dither_RGB &t) const { return JXP_Dither_RGB(R+t.R,G+t.G,B+t.B); }
	JXP_Dither_RGB JXP_API_CALL operator-(const JXP_Dither_RGB &t) const { return JXP_Dither_RGB(R-t.R,G-t.G,B-t.B); }

	void JXP_API_CALL operator+=(JXP_Dither_RGB &t) { R+=t.R; G+=t.G; B+=t.B; }
	void JXP_API_CALL operator-=(JXP_Dither_RGB &t) { R-=t.R; G-=t.G; B-=t.B; }

	// Multiply and divide by scalars
	JXP_Dither_RGB JXP_API_CALL operator*(int t) const { return JXP_Dither_RGB(R*t,G*t,B*t); }
	JXP_Dither_RGB JXP_API_CALL operator/(int t) const { return JXP_Dither_RGB(R/t,G/t,B/t); }

	void JXP_API_CALL operator*=(int t) { R*=t; G*=t; B*=t; }
	void JXP_API_CALL operator/=(int t) { R/=t; G/=t; B/=t; }

	// Shift operators
	JXP_Dither_RGB JXP_API_CALL operator>>(int t) const { return JXP_Dither_RGB(R>>t,G>>t,B>>t); }
	JXP_Dither_RGB JXP_API_CALL operator<<(int t) const { return JXP_Dither_RGB(R<<t,G<<t,B<<t); }

	void JXP_API_CALL operator>>=(int t) { R>>=t; G>>=t; B>>=t; }
	void JXP_API_CALL operator<<=(int t) { R<<=t; G<<=t; B<<=t; }

	// SumSquare() - returns sum of each element squared
	int JXP_API_CALL SumSquare(void) { return R*R + G*G + B*B; }

	// Clamp() - ensures all values are within 0 to 255 range
	void JXP_API_CALL Clamp(void) 
	{ 
		if (R<0) R=0; else if (R>255) R=255;
		if (G<0) G=0; else if (G>255) G=255;
		if (B<0) B=0; else if (B>255) B=255;
	}
};

// JXP_Dither_CRTree - class used to hold an octree used for colour reduction
class JXP_Dither_CRTree
{
private:

	// Number of pixels in this node, including children
	int Num1;

	// Number of pixles in this node which aren't defined in children
	int Num2;

	// Sum of the RGB values for all of the Num2 pixels
	JXP_Dither_RGB Sum;

	// Sum of the square of the distance between all of the Num1 pixels of this 
	// node and the node's centre. The centre is calculated as (int)(max-min)/2, 
	// so is not the perfect centre.
	int E;

	// The 8 children of this node
	JXP_Dither_CRTree *Children[8];

public:

	// Constructor
	JXP_Dither_CRTree(void);

	// Destructor
	~JXP_Dither_CRTree(void);

	// AddPixel() - adds a pixel to the tree
	int JXP_API_CALL AddPixel(JXP_Dither_CRTree **Tree, JXP_Dither_RGB *Pixel, JXP_Dither_RGB *Min, JXP_Dither_RGB *Max, int Level);

	// NumColours() - returns the number of unique colours in the tree
	int JXP_API_CALL NumColours(void);

	// PruneChild() - prunes the Child'th child of the tree node it is called on
	void JXP_API_CALL PruneChild(int Child);

	// ApplyThreshold() - applies the threshold to the tree, to reduce its number of colours
	void JXP_API_CALL ApplyThreshold(int CurrentThreshold, int *NextThreshold);

	// AssignPalette() - used to assign the entries in the tree to a palette
	void JXP_API_CALL AssignPalette(JXP_Palette *Palette, int *NextIndex);
};

// JXP_Dither_Interface - interface class which performs colour reduction and dithering functions
class JXP_Dither_Interface
{
public:

	// GeneratePaletteMultiple() - the main palette generation function
	int JXP_API_CALL GeneratePaletteMultiple(JXP_Image_24b **Source, int NumImages, JXP_Palette **Destination, int NumColours, int Flags);

	// FitToPalette() - fits a 24bit image to a palette, producing an 8bit image
	int JXP_API_CALL FitToPalette(JXP_Image_24b *Source, JXP_Palette *Palette, JXP_Image_8b **Destination, JXP_DitherType DitherType);

private:

	// Cache used during fitting of an image to a palette
	unsigned char *IndexCache;

	// The palette the image is currently being fit to
	JXP_Palette *CurrentPalette;

	// The data of the current palette
	unsigned char *CurrentPaletteData;

	// ClosestIndex() - returns the palette index most closely matching the input pixel
	int JXP_API_CALL ClosestIndex(JXP_Dither_RGB *Pixel);
};

/*********************************************************************************************************/
// Global function prototypes
/*********************************************************************************************************/

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

/*********************************************************************************************************/

#endif // __JXP_DITHER_H
