/*********************************************************************************************************
*
*	JXP_Dither.cpp
*
*	Handles colour reduction and dithering for the JXP library
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

// Revision history:
// ----------------

// v2.01 - 22/1/2000
// No changes to this file in this version.
//
// v2.0 - 4/1/2000
// Major rewrite, including conversion to C++.
//
// v1.21 - 3/1/1998
// Made max recursion depth for colour reduction a configurable parameter
// (note that in version 2.0 this parameter is no longer available),
// plus added 'progress meter' for colour reduction.
//
// v1.2 - 2/1/1998
// No changes to this file in this version.
//
// v1.1 - 1/1/1998
// Initial inclusion of colour reduction and dithering.

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "JXP_Dither.h"

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

/*********************************************************************************************************/
// Local typedefs
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static function prototypes
/*********************************************************************************************************/

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Functions
/*********************************************************************************************************/

/*********************************************************************************************************/
// JXP_Dither_CRTree
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Dither_CRTree::JXP_Dither_CRTree()
*
*	desc:		Constructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Dither_CRTree::JXP_Dither_CRTree(void)
{
	int i;
	for (i=0; i<8; i++)
		Children[i]=NULL;

	Num1=0;
	Num2=0;
	E=0;
}

/*********************************************************************************************************
*
*	function:	JXP_Dither_CRTree::~JXP_Dither_CRTree()
*
*	desc:		Destructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Dither_CRTree::~JXP_Dither_CRTree(void)
{
	int i;
	for (i=0; i<8; i++)
	{
		if (Children[i])
			delete Children[i];
	}
}

/*********************************************************************************************************
*
*	function:	JXP_Dither_CRTree::AddPixel()
*
*	desc:		Adds a pixel to the tree
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Dither_CRTree::AddPixel(JXP_Dither_CRTree **Tree, JXP_Dither_RGB *Pixel, JXP_Dither_RGB *Min, JXP_Dither_RGB *Max, int Level)
{
	if (!*Tree)
	{
		// Allocate a new node
		*Tree=new JXP_Dither_CRTree();
		if (!*Tree)
		{
			JXP_Utils.Output("Error - insufficient memory\n");
			JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
			return JXP_FAILURE;
		}
	}

	JXP_Dither_CRTree *STree=*Tree;
	JXP_Dither_RGB Centre=(*Min+*Max)/2;
	JXP_Dither_RGB NewMin,NewMax;

	if (Level==6)
	{
		// Reached maximum recursion depth - add Pixel here

		STree->Num1++;
		STree->Num2++;

		STree->Sum+=*Pixel;

		STree->E+=(*Pixel-Centre).SumSquare();

		// Ensure no E value goes over the 'big number' threshold
		if ((STree->E>=1<<30) || (STree->E<0))
			STree->E=(1<<30)-1;

		return JXP_SUCCESS;
	}
	else
	{
		// Recurse down one of the child's nodes.

		// First update n1 and E for this node

		STree->Num1++;

		STree->E+=(*Pixel-Centre).SumSquare();

		// Ensure no E value goes over the 'big number' threshold
		if ((STree->E>=1<<30) || (STree->E<0))
			STree->E=(1<<30)-1;

		// Now decide which child this pixel is in

		if (Pixel->R<=Centre.R)
		{
			// Children 0,1,2,5
			if (Pixel->G<=Centre.G)
			{
				// Children 0,2
				if (Pixel->B<=Centre.B)
				{
					// Child 0
					NewMin=*Min;
					NewMax=Centre;

					return STree->Children[0]->AddPixel(&STree->Children[0],Pixel,&NewMin,&NewMax,Level+1);
				}
				else
				{
					// Child 2
					NewMin=JXP_Dither_RGB(Min->R,Min->G,(Min->B+Max->B+1)/2);
					NewMax=JXP_Dither_RGB((Min->R+Max->R)/2,(Min->G+Max->G)/2,Max->B);

					return STree->Children[2]->AddPixel(&STree->Children[2],Pixel,&NewMin,&NewMax,Level+1);
				}
			}
			else
			{
				// Children 1,5
				if (Pixel->B<=Centre.B)
				{
					// Child 1
					NewMin=JXP_Dither_RGB(Min->R,(Min->G+Max->G+1)/2,Min->B);
					NewMax=JXP_Dither_RGB((Min->R+Max->R)/2,Max->G,(Min->B+Max->B)/2);

					return STree->Children[1]->AddPixel(&STree->Children[1],Pixel,&NewMin,&NewMax,Level+1);
				}
				else
				{
					// Child 5
					NewMin=JXP_Dither_RGB(Min->R,(Min->G+Max->G+1)/2,(Min->B+Max->B+1)/2);
					NewMax=JXP_Dither_RGB((Min->R+Max->R)/2,Max->G,Max->B);

					return STree->Children[5]->AddPixel(&STree->Children[5],Pixel,&NewMin,&NewMax,Level+1);
				}
			}
		}
		else
		{
			// Children 3,4,6,7
			if (Pixel->G<=Centre.G)
			{
				// Children 3,7
				if (Pixel->B<=Centre.B)
				{
					// Child 3
					NewMin=JXP_Dither_RGB((Min->R+Max->R+1)/2,Min->G,Min->B);
					NewMax=JXP_Dither_RGB(Max->R,(Min->G+Max->G)/2,(Min->B+Max->B)/2);

					return STree->Children[3]->AddPixel(&STree->Children[3],Pixel,&NewMin,&NewMax,Level+1);
				}
				else
				{
					// Child 7
					NewMin=JXP_Dither_RGB((Min->R+Max->R+1)/2,Min->G,(Min->B+Max->B+1)/2);
					NewMax=JXP_Dither_RGB(Max->R,(Min->G+Max->G)/2,Max->B);

					return STree->Children[7]->AddPixel(&STree->Children[7],Pixel,&NewMin,&NewMax,Level+1);
				}
			}
			else
			{
				// Children 4,6
				if (Pixel->B<=Centre.B)
				{
					// Child 6
					NewMin=JXP_Dither_RGB((Min->R+Max->R+1)/2,(Min->G+Max->G+1)/2,Min->B);
					NewMax=JXP_Dither_RGB(Max->R,Max->G,(Min->B+Max->B)/2);

					return STree->Children[6]->AddPixel(&STree->Children[6],Pixel,&NewMin,&NewMax,Level+1);
				}
				else
				{
					// Child 4
					NewMin=JXP_Dither_RGB((Min->R+Max->R+1)/2,(Min->G+Max->G+1)/2,(Min->B+Max->B+1)/2);
					NewMax=*Max;

					return STree->Children[4]->AddPixel(&STree->Children[4],Pixel,&NewMin,&NewMax,Level+1);
				}
			}
		}
	}

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_Dither_CRTree::NumColours()
*
*	desc:		Returns the number of unique colours in the tree
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Dither_CRTree::NumColours(void)
{
	int Temp=0;
	int i;

	if (!this)
		return 0;

	if (Num2) Temp++;

	for (i=0; i<8; i++)
		Temp+=Children[i]->NumColours();

	return Temp;
}

/*********************************************************************************************************
*
*	function:	JXP_Dither_CRTree::PruneChild()
*
*	desc:		Prunes the Child'th child of the tree node it is called on
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_Dither_CRTree::PruneChild(int Child)
{
	int i;

	// First call recursively with each of the 8 children of the Child'th child
	for (i=0; i<8; i++)
	{
		if (Children[Child]->Children[i])
			Children[Child]->PruneChild(i);
	}

	// Now returned from recursion, each of of the 8 children of the Child'th child
	// are now NULL.

	// Now merge the Child'th child's data up the tree
	Sum+=Children[Child]->Sum;
	Num2+=Children[Child]->Num2;

	// Now remove the Child'th child
	delete Children[Child];
	Children[Child]=NULL;
}

/*********************************************************************************************************
*
*	function:	JXP_Dither_CRTree::ApplyThreshold()
*
*	desc:		Applies the threshold to the tree, to reduce its number of colours - any node with
*				an E value <= to the current threshold is pruned, and the threshold for the next
*				pruning is computed
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_Dither_CRTree::ApplyThreshold(int CurrentThreshold, int *NextThreshold)
{
	int i;

	if (!this)
		return;

	// First check this nodes 8 children to see if they can be culled
	for (i=0; i<8; i++)
	{
		if (Children[i])
		{
			if (Children[i]->E<=CurrentThreshold)
			{
				PruneChild(i);
			}
			else
			{
				if (Children[i]->E<*NextThreshold)
					*NextThreshold=Children[i]->E;
			}
		}
	}

	// Now call recursively with each of the 8 children
	for (i=0; i<8; i++)
	{
		Children[i]->ApplyThreshold(CurrentThreshold,NextThreshold);
	}
}

/*********************************************************************************************************
*
*	function:	JXP_Dither_CRTree::AssignPalette()
*
*	desc:		Used to assign the entries in the tree to a palette
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_Dither_CRTree::AssignPalette(JXP_Palette *Palette, int *NextIndex)
{
	int i;

	if (!this)
		return;

	if (Num2>0)
	{
		// This node corresponds to a colour in the output palette
		JXP_Dither_RGB Temp=Sum/Num2;

		Palette->GetData()[3*(*NextIndex)]=Temp.R;
		Palette->GetData()[3*(*NextIndex)+1]=Temp.G;
		Palette->GetData()[3*(*NextIndex)+2]=Temp.B;

		(*NextIndex)++;
	}

	// Call recursively
	for (i=0; i<8; i++)
	{
		Children[i]->AssignPalette(Palette,NextIndex);
	}
}

/*********************************************************************************************************/
// JXP_Dither_Interface
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Dither_Interface::GeneratePaletteMultiple()
*
*	desc:		The main palette generation function
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Dither_Interface::GeneratePaletteMultiple(JXP_Image_24b **Source, int NumImages, JXP_Palette **Destination, int NumColours, int Flags)
{
	JXP_Dither_CRTree *Tree;
	int KeepWhite,KeepBlack;
	int i,j;

	// Print out message
	if (NumImages==1)
		JXP_Utils.Output("Performing colour reduction\n");
	else
		JXP_Utils.Output("Performing colour reduction on %d images\n",NumImages);

	// Check requested number of colours
	if (NumColours<2 || NumColours>256)
	{
		JXP_Utils.Output("Error - number of colours must be in the range 2 to 256, found %d\n",NumColours);
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_NUMBER);
		return JXP_FAILURE;
	}

	// See if we need to keep black and/or white
	KeepWhite=!!(Flags&JXP_GP_KEEP_WHITE);
	KeepBlack=!!(Flags&JXP_GP_KEEP_BLACK);

	// Check for degenerate cases...
	if (KeepWhite && KeepBlack && NumColours==2)
	{
		// Only 2 colours and want to keep black and white - nothing to generate
		JXP_Palette *Palette=new JXP_Palette(NumColours);
		Palette->GetData()[0]=0;
		Palette->GetData()[1]=0;
		Palette->GetData()[2]=0;
		Palette->GetData()[3]=255;
		Palette->GetData()[4]=255;
		Palette->GetData()[5]=255;

		*Destination=Palette;

		return JXP_SUCCESS;
	}

	// Add each pixel in the source images to the tree
	
	JXP_Utils.Output("Performing classification\n");
	JXP_Utils.Output("- adding images ");

	JXP_Dither_RGB Min(0,0,0),Max(255,255,255),Pixel;
	int ByteOrder=JXP_Utils.GetByteOrder();

	Tree=NULL;

	for (i=0; i<NumImages; i++)
	{
		unsigned char *Data;
		int Width,Height;

		JXP_Utils.Output(".");

		// Get info on i'th image
		Width=Source[i]->GetWidth();
		Height=Source[i]->GetHeight();
		Data=Source[i]->GetData();

		for (j=0; j<Width*Height; j++)
		{
			if (ByteOrder=JXP_BYTEORDER_RGB)
			{
				Pixel.R=*(Data+3*j);
				Pixel.G=*(Data+3*j+1);
				Pixel.B=*(Data+3*j+2);
			}
			else
			{
				Pixel.R=*(Data+3*j+2);
				Pixel.G=*(Data+3*j+1);
				Pixel.B=*(Data+3*j);
			}

			if (Tree->AddPixel(&Tree,&Pixel,&Min,&Max,0)!=JXP_SUCCESS)
			{
				JXP_Utils.Output("Error - adding pixel to tree failed\n");
				return JXP_FAILURE;
			}
		}
	}

	JXP_Utils.Output("\n");

	// Now reduce the number of colours in the tree to the desired number
	
	JXP_Utils.Output("Reducing colours\n");

	int ColoursOriginal,ColoursLeft;
	int DotsPrinted,DotsToPrint;
	int CurrentThreshold,NextThreshold;

	ColoursOriginal=ColoursLeft=Tree->NumColours();
	DotsPrinted=0;

	JXP_Utils.Output("- colours in original: %d ",ColoursLeft);

	CurrentThreshold=0;

	while (ColoursLeft>NumColours)
	{
		// Print out dots as a 'progress meter'
		if (ColoursOriginal>NumColours)
		{
			DotsToPrint=30*(ColoursOriginal-ColoursLeft)/(ColoursOriginal-NumColours);
			for (i=0; i<DotsToPrint-DotsPrinted; i++)
				JXP_Utils.Output(".");
			DotsPrinted=DotsToPrint;
		}

		// Apply the current threshold to the tree, to prune some colours
		
		NextThreshold=1<<30; // a big number...

		Tree->ApplyThreshold(CurrentThreshold,&NextThreshold);

		CurrentThreshold=NextThreshold;
		ColoursLeft=Tree->NumColours();
	}

	JXP_Utils.Output("\n");
	JXP_Utils.Output("- colours in reduction: %d\n",ColoursLeft);

	// Now generate the palette from the reduced tree

	JXP_Utils.Output("Creating palette\n");
	
	int NextIndex=0;
	JXP_Palette *Palette=new JXP_Palette(NumColours);

	Tree->AssignPalette(Palette,&NextIndex);

	// Now fiddle with the palette a bit more to ensure that it contains white and black, if requested
	
	if (KeepWhite || KeepBlack)
	{
		// Want to keep white or black - see if they're already in the palette
		int FoundBlack=0,FoundWhite=0;
		for (i=0; i<Palette->GetNumEntries(); i++)
		{
			if (Palette->GetData()[3*i]==255 && 
				Palette->GetData()[3*i+1]==255 && 
				Palette->GetData()[3*i+2]==255)
			{
				FoundWhite=1;
			}
			else if (Palette->GetData()[3*i]==0 && 
				Palette->GetData()[3*i+1]==0 && 
				Palette->GetData()[3*i+2]==0)
			{
				FoundBlack=1;
			}
		}

		// Compute how many more colours to remove
		int ColoursToRemove=0;
		if (KeepWhite && !FoundWhite) ColoursToRemove++;
		if (KeepBlack && !FoundBlack) ColoursToRemove++;

		if (ColoursToRemove==0)
		{
			if (KeepWhite && KeepBlack) JXP_Utils.Output("- palette already contains white and black\n");
			else if (KeepWhite) JXP_Utils.Output("- palette already contains white\n");
			else if (KeepBlack) JXP_Utils.Output("- palette already contains black\n");
		}
		else if (ColoursToRemove==1)
		{
			if (KeepWhite && !FoundWhite)
			{
				JXP_Utils.Output("- palette does not contain white. Removing an additional colour\n");

				while (ColoursLeft>NumColours-1)
				{
					NextThreshold=1<<30; // a big number...

					Tree->ApplyThreshold(CurrentThreshold,&NextThreshold);

					CurrentThreshold=NextThreshold;
					ColoursLeft=Tree->NumColours();
				}

				// Redo the palette assignment
				NextIndex=0;
				Tree->AssignPalette(Palette,&NextIndex);

				// Put white in the last palette entry
				Palette->GetData()[3*NextIndex]=255;
				Palette->GetData()[3*NextIndex+1]=255;
				Palette->GetData()[3*NextIndex+2]=255;
			}
			else if (KeepBlack && !FoundBlack)
			{
				JXP_Utils.Output("- palette does not contain black. Removing an additional colour\n");

				while (ColoursLeft>NumColours-1)
				{
					NextThreshold=1<<30; // a big number...

					Tree->ApplyThreshold(CurrentThreshold,&NextThreshold);

					CurrentThreshold=NextThreshold;
					ColoursLeft=Tree->NumColours();
				}

				// Redo the palette assignment
				NextIndex=0;
				Tree->AssignPalette(Palette,&NextIndex);

				// Put black in the last palette entry
				Palette->GetData()[3*NextIndex]=0;
				Palette->GetData()[3*NextIndex+1]=0;
				Palette->GetData()[3*NextIndex+2]=0;
			}
		}
		else if (ColoursToRemove==2)
		{
			JXP_Utils.Output("- palette does not contain black or white. Removing two additional colours\n");

			while (ColoursLeft>NumColours-2)
			{
				NextThreshold=1<<30; // a big number...

				Tree->ApplyThreshold(CurrentThreshold,&NextThreshold);

				CurrentThreshold=NextThreshold;
				ColoursLeft=Tree->NumColours();
			}

			// Redo the palette assignment
			NextIndex=0;
			Tree->AssignPalette(Palette,&NextIndex);

			// Put black in the second last palette entry
			Palette->GetData()[3*NextIndex]=0;
			Palette->GetData()[3*NextIndex+1]=0;
			Palette->GetData()[3*NextIndex+2]=0;
			NextIndex++;

			// Put white in the last palette entry
			Palette->GetData()[3*NextIndex]=255;
			Palette->GetData()[3*NextIndex+1]=255;
			Palette->GetData()[3*NextIndex+2]=255;
		}
	}

	// Done with the tree

	delete Tree;

	// Return palette
	*Destination=Palette;

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_Dither_Interface::FitToPalette()
*
*	desc:		Fits a 24bit image to a palette, producing an 8bit image
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Dither_Interface::FitToPalette(JXP_Image_24b *Source, JXP_Palette *Palette, JXP_Image_8b **Destination, JXP_DitherType DitherType)
{
	JXP_Utils.Output("Fitting image to palette\n");

	if (DitherType!=JXP_DITHER_NONE && DitherType!=JXP_DITHER_FS)
	{
		JXP_Utils.Output("Error - invalid dither type\n");
		JXP_Utils.SetLastError(JXP_ERROR_INVALID_DITHER);
		return JXP_FAILURE;
	}

	// Allocate index cache
	IndexCache=(unsigned char *)malloc(64*64*64);
	if (!IndexCache)
	{
		JXP_Utils.Output("Error - insufficient memory\n");
		JXP_Utils.SetLastError(JXP_ERROR_NO_MEMORY);
		return JXP_FAILURE;
	}
	memset(IndexCache,255,64*64*64);

	// Set up current palette
	CurrentPalette=Palette;
	CurrentPaletteData=Palette->GetData();

	// Get info on source image
	unsigned char *SourceData;
	int Width,Height;

	Width=Source->GetWidth();
	Height=Source->GetHeight();
	SourceData=Source->GetData();

	// Create new destination image
	JXP_Image_8b *Image8b=new JXP_Image_8b(Width,Height);
	Image8b->SetPalette(new JXP_Palette(Palette));

	// Get info on destination image
	unsigned char *DestinationData;

	DestinationData=Image8b->GetData();

	// Fit image to palette
	int ByteOrder=JXP_Utils.GetByteOrder();

	if (DitherType==JXP_DITHER_NONE)
	{
		// No Dithering
		JXP_Dither_RGB Pixel;
		int i;

		for (i=0; i<Width*Height; i++)
		{
			if (ByteOrder==JXP_BYTEORDER_RGB)
			{
				Pixel.R=SourceData[3*i];
				Pixel.G=SourceData[3*i+1];
				Pixel.B=SourceData[3*i+2];
			}
			else
			{
				Pixel.R=SourceData[3*i+2];
				Pixel.G=SourceData[3*i+1];
				Pixel.B=SourceData[3*i];
			}

			DestinationData[i]=ClosestIndex(&Pixel);
		}
	}
	else
	{
		// Perform Floyd-Steinberg error diffusion dithering

		// Allocate and initialise current and next error arrays
		JXP_Dither_RGB *CurrentError,*NextError,*TempError;
		CurrentError=new JXP_Dither_RGB[Width+2];
		NextError=new JXP_Dither_RGB[Width+2];
		memset(CurrentError,0,sizeof(JXP_Dither_RGB)*(Width+2));
		memset(NextError,0,sizeof(JXP_Dither_RGB)*(Width+2));

		// CurrentError and NextError contain the error values multipled by 16

		JXP_Dither_RGB Error,SourcePixel,Pixel,QuantisedPixel;
		int Index;

		int x,y,_x;

		for (y=0; y<Height; y++)
		{
			for (_x=0; _x<Width; _x++)
			{
				// Note that CurrentError[x+1] and NextError[x+1]
				// are the entries for the xth pixel

				if (y&1)
				{
					// pass left to right - x is just x
					x=_x;
				}
				else
				{
					// pass right to left - replace x with (Width-x-1)
					x=Width-_x-1;
				}

				// *****
				// First compute the index for this pixel

				// Get the error value for this pixel
				Error=CurrentError[x+1];

				// Round to an integer
				Error>>=4;

				// Get source pixel
				if (ByteOrder==JXP_BYTEORDER_RGB)
				{
					SourcePixel.R=SourceData[3*(Width*y+x)];
					SourcePixel.G=SourceData[3*(Width*y+x)+1];
					SourcePixel.B=SourceData[3*(Width*y+x)+2];
				}
				else
				{
					SourcePixel.R=SourceData[3*(Width*y+x)+2];
					SourcePixel.G=SourceData[3*(Width*y+x)+1];
					SourcePixel.B=SourceData[3*(Width*y+x)];
				}

				// Add propagated error
				Pixel=SourcePixel+Error;

				// Clamp pixel if necessary
				Pixel.Clamp();

				// Turn pixel into palette index
				Index=ClosestIndex(&Pixel);

				// Write to destination
				DestinationData[Width*y+x]=Index;

				// *****
				// Now do the error diffusion

				// Compute the error between the actual and quantised output
				QuantisedPixel.R=CurrentPaletteData[3*Index];
				QuantisedPixel.G=CurrentPaletteData[3*Index+1];
				QuantisedPixel.B=CurrentPaletteData[3*Index+2];
				Error=SourcePixel-QuantisedPixel;

				if (y&1)
				{
					// Error diffused to right, *16
					CurrentError[x+2]+=Error*7;

					// Error diffused to below right, *16
					NextError[x+2]+=Error;

					// Error diffused to below, *16
					NextError[x+1]+=Error*5;

					// Error diffused to below left, *16
					NextError[x]+=Error*3;
				}
				else
				{
					// Error diffused to left, *16
					CurrentError[x]+=Error*7;

					// Error diffused to below left, *16
					NextError[x]+=Error;

					// Error diffused to below, *16
					NextError[x+1]+=Error*5;

					// Error diffused to below right, *16
					NextError[x+2]+=Error*3;
				}
			}

			// Swap CurrentError and NextError
			TempError=CurrentError;
			CurrentError=NextError;
			NextError=TempError;

			// Reset NextError
			memset(NextError,0,sizeof(JXP_Dither_RGB)*(Width+2));
		}

		// Release current and next error arrays
		delete [] CurrentError;
		delete [] NextError;
	}

	// Free index cache
	free(IndexCache);

	// Set return image
	*Destination=Image8b;

	return JXP_SUCCESS;
}

/*********************************************************************************************************
*
*	function:	JXP_Dither_Interface::ClosestIndex()
*
*	desc:		Returns the palette index most closely matching the input pixel
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Dither_Interface::ClosestIndex(JXP_Dither_RGB *Pixel)
{
	int Entry;

	// Compute cache entry by demoting colour channels to 6 bits
	Entry=4096*(Pixel->R>>2) + 64*(Pixel->G>>2) + (Pixel->B>>2);

	if (IndexCache[Entry]==255)
	{
		// Entry'th entry in the index cache has not been filled yet - so compute the closest index
		// the long way.

		// Note that 255 is a valid cache entry (if the palette has 256 colours), so calls to this
		// function with a colour which maps to an index of 255 will re-perform the full calculation
		// needlessly. However, to avoid this would require an extra bit per entry, and the speedup
		// is pretty good anyway.

		JXP_Dither_RGB Temp;
		int MinDiff,TempDiff;
		int Pos;
		int i;

		Temp.R=CurrentPaletteData[0];
		Temp.G=CurrentPaletteData[1];
		Temp.B=CurrentPaletteData[2];

		Pos=0;
		MinDiff=(*Pixel-Temp).SumSquare();

		for (i=1; i<CurrentPalette->GetNumEntries(); i++)
		{
			Temp.R=CurrentPaletteData[3*i];
			Temp.G=CurrentPaletteData[3*i+1];
			Temp.B=CurrentPaletteData[3*i+2];

			TempDiff=(*Pixel-Temp).SumSquare();

			if (TempDiff<MinDiff)
			{
				MinDiff=TempDiff;
				Pos=i;
			}
		}

		// Pos is the closest index to Pixel
		IndexCache[Entry]=Pos;
	}

	return IndexCache[Entry];
}

/*********************************************************************************************************/
