/*********************************************************************************************************
*
*	JXP.h
*
*	Interface to the JXP library, plus global definitions
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

// Revision history:
// ----------------

// v2.01 - 22/1/2000
// Fixed a small bug in the resampling code.
//
// v2.0 - 4/1/2000
// Major rewrite, including conversion to C++. Added Targa and resampling
// functions.
//
// v1.21 - 3/1/1998
// Made max recursion depth for colour reduction a configurable parameter, 
// plus added 'progress meter' for colour reduction.
//
// v1.2 - 2/1/1998
// Added support for .MSK mask files.
//
// v1.1 - 1/1/1998
// Added colour reduction and dithering functions.
//
// v1.0 - 23/12/1997
// Initial release, JPEG decoder only.


// Acknowledgements:
// ----------------

// JPG decoder: From the book 'JPEG: still image data compression standard', by
// William B. Pennebaker and Joan L. Mitchell.

// Colour reduction: Based on a description of an algorithm used by ImageMagick,
// a graphics library for Unix.

// Floyd-Steinberg dithering: Originally from a description of John DiCamillo's
// Alpha engine. Bug-finding help from the Independent JPEG Group's quantisation
// source (jquant2.c). Final implementation based on that used by the public
// domain mpeg_play MPEG library's dithering source (fs2.c).

#ifndef __JXP_H
#define __JXP_H

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include <stdio.h>

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

// JXP calling convention for general API functions
#define JXP_API_CALL				__fastcall

// JXP calling convention for variable argument functions (Visual C requires __cdecl)
#define JXP_VARARG_CALL				__cdecl

// JXP function return codes
#define JXP_SUCCESS					1
#define JXP_FAILURE					0

// JXP error codes - as returned by JXP_Utils.GetLastError()
enum JXP_ErrorCode
{
	JXP_ERROR_NONE,					// No error
	JXP_ERROR_NO_MEMORY,			// Insufficient memory to allocate internal data structures
	JXP_ERROR_FILE_NOT_FOUND,		// The file could not be found
	JXP_ERROR_INVALID_DATA,			// Data is corrupt
	JXP_ERROR_NOT_SUPPORTED,		// The file format is not supported by this version of JXP
	JXP_ERROR_INVALID_NUMBER,		// If palette generation is requested with an invalid number of colours
	JXP_ERROR_INVALID_DITHER,		// If an invalid dithering type is requested
	JXP_ERROR_INVALID_PARAMETER,	// A function parameter was invalid
};

// JXP byte orders for RGB components
enum JXP_ByteOrder
{
	JXP_BYTEORDER_RGB,				// Red in lowest byte, then green and blue
	JXP_BYTEORDER_BGR,				// Blue in lowest byte, then green and red
};

// JXP flags for palette generation
enum
{
	JXP_GP_KEEP_WHITE=0x01,			// Ensure pure white is in the palette
	JXP_GP_KEEP_BLACK=0x02,			// Ensure pure black is in the palette
};

// JXP Dither type options
enum JXP_DitherType
{
	JXP_DITHER_NONE,				// No dithering
	JXP_DITHER_FS,					// Floyd-Steinberg error diffusion dithering
};

// JXP supported Targa image types
enum JXP_TargaType
{
	JXP_TGA_INDEXED,				// Indexed colour (ie has a palette)
	JXP_TGA_RGB,					// RGB (24bit)
	JXP_TGA_UNSUPPORTED,			// Unsupported format
};

// JXP supported image resampling types
enum JXP_ResampleType
{
	JXP_RESAMPLE_NN,				// Nearest neighbour
	JXP_RESAMPLE_BILINEAR,			// Bilinear filter
};

/*********************************************************************************************************/
// Global typedefs
/*********************************************************************************************************/

// Forward declarations
class JXP_Image_8b;

// JXP_Data - defines an arbitrary block of data
class JXP_Data
{
private:

	// The data itself
	unsigned char *Data;

	// Its length in bytes
	int Length;

public:

	// Constructor/Destructor
	JXP_Data(int Length);
	~JXP_Data(void);

	// GetData() - returns data pointer
	unsigned char * JXP_API_CALL GetData(void);

	// GetLength() - returns length of data (in bytes)
	int JXP_API_CALL GetLength(void);

	// WriteToFile() - dumps the block of data to the file
	void JXP_API_CALL WriteToFile(FILE *fptr);
};

// JXP_GlobalUtils - various global utilities etc used by JXP
class JXP_GlobalUtils
{
private:

	// Pointer to the function used to 'display' JXP library output - can
	// be NULL, in which case nothing is output. The default is to write
	// to stdout.
	void (JXP_API_CALL *OutputFunction)(char *String);

	// The last error produced by any JXP function
	JXP_ErrorCode LastError;

	// The byte order in which JXP expects and creates all 24bit images (RGB by default)
	JXP_ByteOrder ByteOrder;

public:

	// Constructor
	JXP_GlobalUtils(void);

	// Output() - outputs the formatted message
	void JXP_VARARG_CALL Output(char *Format, ...);

	// SetOutputFunction() - sets the new output function, or NULL for no output
	void JXP_API_CALL SetOutputFunction(void (JXP_API_CALL *Function)(char *String));

	// SetLastError(), GetLastError() - set and get the last JXP error code
	void JXP_API_CALL SetLastError(JXP_ErrorCode Error);
	JXP_ErrorCode JXP_API_CALL GetLastError(void);

	// Returns a 'friendly' description of the error code
	char * JXP_API_CALL ErrorDescription(JXP_ErrorCode Error);

	// SetByteOrder(), GetByteOrder() - set and get the byte order for 24bit images
	void JXP_API_CALL SetByteOrder(JXP_ByteOrder ByteOrder);
	JXP_ByteOrder JXP_API_CALL GetByteOrder(void);
};

// JXP_Image_24b - defines a 24bit image
class JXP_Image_24b
{
private:

	// Dimensions of the image
	int Width,Height;

	// The image data - 3*Width*Height bytes
	// The byte order (RGB or BGR) is configurable
	unsigned char *Data;

public:

	// Constructor/Destructor
	JXP_Image_24b(int Width, int Height);
	JXP_Image_24b(JXP_Image_8b *Image8b);
	~JXP_Image_24b(void);

	// GetData() - returns data pointer
	unsigned char * JXP_API_CALL GetData(void);

	// GetWidth(), GetHeight() - returns the image dimensions
	int JXP_API_CALL GetWidth(void);
	int JXP_API_CALL GetHeight(void);

	// WriteToFile() - dumps the entire image (3*Width*Height bytes) to the file
	void JXP_API_CALL WriteToFile(FILE *fptr);
};

// JXP_Palette() - stores an 8bit palette (up to 256 entries)
class JXP_Palette
{
private:

	// Number of entries - must be between 0 and 256
	int NumEntries;

	// The palette data - stores 256 entries regardless of actual
	// size. Each palette entry has red in the lowest byte, then green and blue.
	unsigned char Data[768];

public:

	// Constructor
	JXP_Palette(int NumEntries);
	JXP_Palette(JXP_Palette *Palette);

	// GetData() - returns the data pointer
	unsigned char * JXP_API_CALL GetData(void);

	// GetNumEntries() - returns the number of entries in the palette
	int JXP_API_CALL GetNumEntries(void);
};

// JXP_Image_8b - defines an 8 bit indexed colour image
class JXP_Image_8b
{
private:

	// Dimensions of the image
	int Width,Height;

	// The image data - Width*Height bytes
	unsigned char *Data;

	// The palette
	JXP_Palette *Palette;

public:

	// Constructor/Destructor
	JXP_Image_8b(int Width, int Height);
	~JXP_Image_8b(void);

	// GetData() - returns data pointer
	unsigned char * JXP_API_CALL GetData(void);

	// GetWidth(), GetHeight() - returns the image dimensions
	int JXP_API_CALL GetWidth(void);
	int JXP_API_CALL GetHeight(void);

	// GetPalette() - returns the palette
	JXP_Palette * JXP_API_CALL GetPalette(void);

	// SetPalette() - replaces the current palette
	void JXP_API_CALL SetPalette(JXP_Palette *Palette);
};

// JXP_Image_GreyScale - defines a greyscale image - an 8bit image without a palette
class JXP_Image_GreyScale
{
private:

	// Dimensions of the image
	int Width,Height;

	// The image data - Width*Height bytes
	unsigned char *Data;

public:

	// Constructor/Destructor
	JXP_Image_GreyScale(int Width, int Height);
	~JXP_Image_GreyScale(void);

	// GetData() - returns data pointer
	unsigned char * JXP_API_CALL GetData(void);

	// GetWidth(), GetHeight() - returns the image dimensions
	int JXP_API_CALL GetWidth(void);
	int JXP_API_CALL GetHeight(void);
};

/*********************************************************************************************************/
// Global function prototypes
/*********************************************************************************************************/

// Note that all interface functions return JXP_SUCCESS or JXP_FAILURE to indicate success or failure
// respectively. If a function fails, use JXP_Utils.GetLastError() to get the error code.

// **** JPEG FUNCTIONS ****

// Decode a JPEG in the data block 'Source', returning it via 'Destination'
int JXP_API_CALL JXP_DecodeJPEG(JXP_Data *Source, JXP_Image_24b **Destination);

// Decode the JPEG file 'Filename', returning it via 'Destination'
int JXP_API_CALL JXP_DecodeJPEGFile(char *Filename, JXP_Image_24b **Destination);

// **** COLOUR REDUCTION AND DITHERING FUNCTIONS ***

// Generate a palette from the 24bit image 'Source', with the desired number of colours (must be
// between 2 and 256), returning it via 'Destination'. 'Flags' is a combination of the JXP_GP_* values.
int JXP_API_CALL JXP_GeneratePalette(JXP_Image_24b *Source, JXP_Palette **Destination, int NumColours, int Flags=0);

// Generate a palette from the 'NumImages' images in the array of 24bit images 'Source', with the desired
// number of colours (must be between 2 and 256), returning it via 'Destination'.
int JXP_API_CALL JXP_GeneratePaletteMultiple(JXP_Image_24b **Source, int NumImages, JXP_Palette **Destination, int NumColours, int Flags=0);

// Fits the 24bit 'Source' image to the supplied palette, returning an 8bit image via 'Destination'
int JXP_API_CALL JXP_FitToPalette(JXP_Image_24b *Source, JXP_Palette *Palette, JXP_Image_8b **Destination, JXP_DitherType DitherType=JXP_DITHER_FS);

// **** TARGA FUNCTIONS ****

// Examines the TGA file in 'Source', returning it's type via 'Format'
int JXP_API_CALL JXP_GetTargaFormat(JXP_Data *Source, JXP_TargaType *Format);

// Examines the TGA file 'Filename', returning its type via 'Format'
int JXP_API_CALL JXP_GetTargaFileFormat(char *Filename, JXP_TargaType *Format);

// Read a 24bit TGA in the data block 'Source', returning it via 'Destination'
int JXP_API_CALL JXP_Read24bTarga(JXP_Data *Source, JXP_Image_24b **Destination);

// Read the 24bit TGA file 'Filename', returning it via 'Destination'
int JXP_API_CALL JXP_Read24bTargaFile(char *Filename, JXP_Image_24b **Destination);

// Read an 8bit TGA in the data block 'Source', returning it via 'Destination'
int JXP_API_CALL JXP_Read8bTarga(JXP_Data *Source, JXP_Image_8b **Destination);

// Read the 8bit TGA file 'Filename', returning it via 'Destination'
int JXP_API_CALL JXP_Read8bTargaFile(char *Filename, JXP_Image_8b **Destination);

// **** MASK FUNCTIONS ****

// Generates a mask file from the black and white image 'Source', returning it as a block of data via
// 'Destination'
int JXP_API_CALL JXP_GenerateMask(JXP_Image_24b *Source, JXP_Data **Destination);

// Read a mask in the data block 'Source', returning it via 'Destination'
int JXP_API_CALL JXP_ReadMask(JXP_Data *Source, JXP_Image_GreyScale **Destination);

// Read the mask file 'Filename', returning it via 'Destination'
int JXP_API_CALL JXP_ReadMaskFile(char *Filename, JXP_Image_GreyScale **Destination);

// **** RESAMPLING FUNCTIONS ****

// Resamples the image 'Source' to the dimensions 'Width'*'Height', using the resampling method 
// 'ResampleType', returning it via 'Destination'
int JXP_API_CALL JXP_ResampleImage(JXP_Image_24b *Source, int Width, int Height, JXP_Image_24b **Destination, JXP_ResampleType ResampleType=JXP_RESAMPLE_BILINEAR);

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

// Global instance of the JXP_GlobalUtils class - which everything which needs to should use to
// access relevant functions/variables.
extern JXP_GlobalUtils JXP_Utils;

/*********************************************************************************************************/

#endif // __JXP_H
