/*********************************************************************************************************
*
*	JXP.cpp
*
*	Interface to the JXP library, plus global definitions
*
*	Author: Saxon Druce
*
*	Copyright + 1997-2000
*
*	Use of this source code is subject to acceptance of the conditions of the
*	license in the accompanying documentation.
*
**********************************************************************************************************/

// Revision history:
// ----------------

// v2.01 - 22/1/2000
// Fixed a small bug in the resampling code.
//
// v2.0 - 4/1/2000
// Major rewrite, including conversion to C++. Added Targa and resampling
// functions.
//
// v1.21 - 3/1/1998
// Made max recursion depth for colour reduction a configurable parameter, 
// plus added 'progress meter' for colour reduction.
//
// v1.2 - 2/1/1998
// Added support for .MSK mask files.
//
// v1.1 - 1/1/1998
// Added colour reduction and dithering functions.
//
// v1.0 - 23/12/1997
// Initial release, JPEG decoder only.


// Acknowledgements:
// ----------------

// JPG decoder: From the book 'JPEG: still image data compression standard', by
// William B. Pennebaker and Joan L. Mitchell.

// Colour reduction: Based on a description of an algorithm used by ImageMagick,
// a graphics library for Unix.

// Floyd-Steinberg dithering: Originally from a description of John DiCamillo's
// Alpha engine. Bug-finding help from the Independent JPEG Group's quantisation
// source (jquant2.c). Final implementation based on that used by the public
// domain mpeg_play MPEG library's dithering source (fs2.c).

/*********************************************************************************************************/
// Include files
/*********************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <memory.h>

#include "JXP.h"

#include "JXP_JPEG.h"
#include "JXP_Dither.h"
#include "JXP_Targa.h"
#include "JXP_Mask.h"
#include "JXP_Resample.h"

/*********************************************************************************************************/
// Defines
/*********************************************************************************************************/

/*********************************************************************************************************/
// Local typedefs
/*********************************************************************************************************/

/*********************************************************************************************************/
// Static function prototypes
/*********************************************************************************************************/

static void JXP_API_CALL StandardOutputFunction(char *String);

/*********************************************************************************************************/
// Global variables
/*********************************************************************************************************/

// Global instance of the JXP_GlobalUtils class - which everything which needs to should use to
// access relevant functions/variables.
JXP_GlobalUtils JXP_Utils;

/*********************************************************************************************************/
// Static variables
/*********************************************************************************************************/

/*********************************************************************************************************/
// Functions
/*********************************************************************************************************/

/*********************************************************************************************************/
// JXP_Data
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Data::JXP_Data()
*
*	desc:		Constructor - Length is the length of the data in bytes
*
*	notes:
*
**********************************************************************************************************/

JXP_Data::JXP_Data(int Length)
{
	this->Length=Length;
	Data=(unsigned char *)malloc(Length);
}

/*********************************************************************************************************
*
*	function:	JXP_Data::~JXP_Data()
*
*	desc:		Destructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Data::~JXP_Data(void)
{
	free(Data);
}

/*********************************************************************************************************
*
*	function:	JXP_Data::GetData()
*
*	desc:		Returns the data pointer, for writing into or reading out of
*
*	notes:
*
**********************************************************************************************************/

unsigned char * JXP_API_CALL JXP_Data::GetData(void)
{
	return Data;
}

/*********************************************************************************************************
*
*	function:	JXP_Data::GetLength()
*
*	desc:		Returns the length of the data, in bytes
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Data::GetLength(void)
{
	return Length;
}

/*********************************************************************************************************
*
*	function:	JXP_Data::WriteToFile()
*
*	desc:		Dumps the block of data to the file
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_Data::WriteToFile(FILE *fptr)
{
	fwrite(Data,sizeof(char),Length,fptr);
}

/*********************************************************************************************************/
// JXP_GlobalUtils
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_GlobalUtils::JXP_GlobalUtils()
*
*	desc:		Constructor
*
*	notes:
*
**********************************************************************************************************/

JXP_GlobalUtils::JXP_GlobalUtils(void)
{
	// Set up the default output function - to write to stdout
	OutputFunction=StandardOutputFunction;

	// Initialise with no error
	LastError=JXP_ERROR_NONE;

	// Default to RGB byte order in 24bit images
	ByteOrder=JXP_BYTEORDER_RGB;
}

/*********************************************************************************************************
*
*	function:	StandardOutputFunction()
*
*	desc:		The default output function - just prints to stdout
*
*	notes:
*
**********************************************************************************************************/

static void JXP_API_CALL StandardOutputFunction(char *String)
{
	printf(String);
	fflush(stdout);
}

/*********************************************************************************************************
*
*	function:	JXP_GlobalUtils::Output()
*
*	desc:		Outputs the formatted message - to the currently set output function (or not at all if it
*				is NULL)
*
*	notes:
*
**********************************************************************************************************/

void JXP_VARARG_CALL JXP_GlobalUtils::Output(char *Format, ...)
{
	if (OutputFunction)
	{
		va_list ArgList;
		char Buf[1024];

		va_start(ArgList,Format);
		vsprintf(Buf,Format,ArgList);

		OutputFunction(Buf);
	}
}

/*********************************************************************************************************
*
*	function:	JXP_GlobalUtils::SetOutputFunction()
*
*	desc:		Sets the output function, or NULL for no output
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_GlobalUtils::SetOutputFunction(void (JXP_API_CALL *Function)(char *String))
{
	OutputFunction=Function;
}

/*********************************************************************************************************
*
*	function:	JXP_GlobalUtils::SetLastError()
*
*	desc:		Sets the last JXP error code
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_GlobalUtils::SetLastError(JXP_ErrorCode Error)
{
	LastError=Error;
}

/*********************************************************************************************************
*
*	function:	JXP_GlobalUtils::GetLastError()
*
*	desc:		Gets the last JXP error code
*
*	notes:
*
**********************************************************************************************************/

JXP_ErrorCode JXP_API_CALL JXP_GlobalUtils::GetLastError(void)
{
	return LastError;
}

/*********************************************************************************************************
*
*	function:	JXP_GlobalUtils::ErrorDescription()
*
*	desc:		Returns a friendly description of the error code
*
*	notes:
*
**********************************************************************************************************/

char * JXP_API_CALL JXP_GlobalUtils::ErrorDescription(JXP_ErrorCode Error)
{
	switch (Error)
	{
		case JXP_ERROR_NONE:
			return "No error";
		case JXP_ERROR_NO_MEMORY:
			return "Insufficient memory";
		case JXP_ERROR_FILE_NOT_FOUND:
			return "File not found";
		case JXP_ERROR_INVALID_DATA:
			return "File data is invalid or corrupt";
		case JXP_ERROR_NOT_SUPPORTED:
			return "File format not supported by this version of JXP";
		case JXP_ERROR_INVALID_NUMBER:
			return "Invalid number of colours requested for palette generation";
		case JXP_ERROR_INVALID_DITHER:
			return "Invalid dithering type requested";
		case JXP_ERROR_INVALID_PARAMETER:
			return "Invalid function parameter";
		default:
			return "Unrecognised error code";
	}
}

/*********************************************************************************************************
*
*	function:	JXP_GlobalUtils::SetByteOrder()
*
*	desc:		Sets the byte order of 24bit images
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_GlobalUtils::SetByteOrder(JXP_ByteOrder ByteOrder)
{
	this->ByteOrder=ByteOrder;
}

/*********************************************************************************************************
*
*	function:	JXP_GlobalUtils::GetByteOrder()
*
*	desc:		Gets the byte order of 24bit images
*
*	notes:
*
**********************************************************************************************************/

JXP_ByteOrder JXP_API_CALL JXP_GlobalUtils::GetByteOrder(void)
{
	return ByteOrder;
}

/*********************************************************************************************************/
// JXP_Image_24b
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Image_24b::JXP_Image_24b()
*
*	desc:		Constructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Image_24b::JXP_Image_24b(int Width, int Height)
{
	this->Width=Width;
	this->Height=Height;
	Data=(unsigned char *)malloc(3*Width*Height);
}

/*********************************************************************************************************
*
*	function:	JXP_Image_24b::JXP_Image_24b()
*
*	desc:		Constructor - creates a 24bit from an 8bit image
*
*	notes:
*
**********************************************************************************************************/

JXP_Image_24b::JXP_Image_24b(JXP_Image_8b *Image8b)
{
	unsigned char *SourceData;
	unsigned char *PaletteData;
	int i;

	Width=Image8b->GetWidth();
	Height=Image8b->GetHeight();
	Data=(unsigned char *)malloc(3*Width*Height);

	// Get source data
	SourceData=Image8b->GetData();
	PaletteData=Image8b->GetPalette()->GetData();

	if (JXP_Utils.GetByteOrder()==JXP_BYTEORDER_RGB)
	{
		for (i=0; i<Width*Height; i++)
		{
			Data[3*i]  =PaletteData[3*SourceData[i]];
			Data[3*i+1]=PaletteData[3*SourceData[i]+1];
			Data[3*i+2]=PaletteData[3*SourceData[i]+2];
		}
	}
	else
	{
		for (i=0; i<Width*Height; i++)
		{
			Data[3*i]  =PaletteData[3*SourceData[i]+2];
			Data[3*i+1]=PaletteData[3*SourceData[i]+1];
			Data[3*i+2]=PaletteData[3*SourceData[i]];
		}
	}
}

/*********************************************************************************************************
*
*	function:	JXP_Image_24b::~JXP_Image_24b()
*
*	desc:		Destructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Image_24b::~JXP_Image_24b(void)
{
	free(Data);
}

/*********************************************************************************************************
*
*	function:	JXP_Image_24b::GetData()
*
*	desc:		Returns the data pointer, so that it can be written to, or read from
*
*	notes:
*
**********************************************************************************************************/

unsigned char * JXP_API_CALL JXP_Image_24b::GetData(void)
{
	return Data;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_24b::GetWidth()
*
*	desc:		Returns the image width (in pixels)
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Image_24b::GetWidth(void)
{
	return Width;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_24b::GetHeight()
*
*	desc:		Returns the image height (in pixels)
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Image_24b::GetHeight(void)
{
	return Height;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_24b::WriteToFile()
*
*	desc:		Dumps the entire image (3*Width*Height bytes) to the file
*
*	notes:
*
**********************************************************************************************************/

void JXP_API_CALL JXP_Image_24b::WriteToFile(FILE *fptr)
{
	fwrite(Data,sizeof(unsigned char),3*Width*Height,fptr);
}

/*********************************************************************************************************/
// JXP_Palette
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Palette::JXP_Palette()
*
*	desc:		Constructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Palette::JXP_Palette(int NumEntries)
{
	this->NumEntries=NumEntries;

	if (this->NumEntries<0)
		this->NumEntries=0;
	else if (this->NumEntries>256)
		this->NumEntries=256;

	// Clear palette to all black
	memset(Data,0,768);
}

/*********************************************************************************************************
*
*	function:	JXP_Palette::JXP_Palette()
*
*	desc:		Constructor - creates a new palette by copying the parameter
*
*	notes:
*
**********************************************************************************************************/

JXP_Palette::JXP_Palette(JXP_Palette *Palette)
{
	NumEntries=Palette->NumEntries;
	memcpy(Data,Palette->Data,768);
}

/*********************************************************************************************************
*
*	function:	JXP_Palette::GetData()
*
*	desc:		Returns the data pointer, for reading from or writing to
*
*	notes:
*
**********************************************************************************************************/

unsigned char * JXP_API_CALL JXP_Palette::GetData(void)
{
	return &Data[0];
}

/*********************************************************************************************************
*
*	function:	JXP_Palette::GetNumEntries()
*
*	desc:		Returns the number of entries in the palette
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Palette::GetNumEntries(void)
{
	return NumEntries;
}

/*********************************************************************************************************/
// JXP_Image_8b
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Image_8b::JXP_Image_8b()
*
*	desc:		Constructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Image_8b::JXP_Image_8b(int Width, int Height)
{
	this->Width=Width;
	this->Height=Height;
	Data=(unsigned char *)malloc(Width*Height);
	Palette=new JXP_Palette(256);
}

/*********************************************************************************************************
*
*	function:	JXP_Image_8b::~JXP_Image_8b()
*
*	desc:		Destructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Image_8b::~JXP_Image_8b(void)
{
	free(Data);
	delete Palette;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_8b::GetData()
*
*	desc:		Returns the data pointer, so that it can be written to, or read from
*
*	notes:
*
**********************************************************************************************************/

unsigned char * JXP_API_CALL JXP_Image_8b::GetData(void)
{
	return Data;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_8b::GetWidth()
*
*	desc:		Returns the image width (in pixels)
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Image_8b::GetWidth(void)
{
	return Width;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_8b::GetHeight()
*
*	desc:		Returns the image height (in pixels)
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Image_8b::GetHeight(void)
{
	return Height;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_8b::GetPalette()
*
*	desc:		Returns the palette
*
*	notes:
*
**********************************************************************************************************/

JXP_Palette * JXP_API_CALL JXP_Image_8b::GetPalette(void)
{
	return Palette;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_8b::SetPalette()
*
*	desc:		Replaces the current palette
*
*	notes:		This method DOES NOT take it's own copy of the new palette
*
**********************************************************************************************************/

void JXP_API_CALL JXP_Image_8b::SetPalette(JXP_Palette *Palette)
{
	delete this->Palette;
	this->Palette=Palette;
}

/*********************************************************************************************************/
// JXP_Image_GreyScale
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_Image_GreyScale::JXP_Image_GreyScale()
*
*	desc:		Constructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Image_GreyScale::JXP_Image_GreyScale(int Width, int Height)
{
	this->Width=Width;
	this->Height=Height;
	Data=(unsigned char *)malloc(Width*Height);
}

/*********************************************************************************************************
*
*	function:	JXP_Image_GreyScale::~JXP_Image_GreyScale()
*
*	desc:		Destructor
*
*	notes:
*
**********************************************************************************************************/

JXP_Image_GreyScale::~JXP_Image_GreyScale(void)
{
	free(Data);
}

/*********************************************************************************************************
*
*	function:	JXP_Image_GreyScale::GetData()
*
*	desc:		Returns the data pointer, so that it can be written to, or read from
*
*	notes:
*
**********************************************************************************************************/

unsigned char * JXP_API_CALL JXP_Image_GreyScale::GetData(void)
{
	return Data;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_GreyScale::GetWidth()
*
*	desc:		Returns the image width (in pixels)
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Image_GreyScale::GetWidth(void)
{
	return Width;
}

/*********************************************************************************************************
*
*	function:	JXP_Image_GreyScale::GetHeight()
*
*	desc:		Returns the image height (in pixels)
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Image_GreyScale::GetHeight(void)
{
	return Height;
}

/*********************************************************************************************************/
// Interface Functions
/*********************************************************************************************************/

/*********************************************************************************************************
*
*	function:	JXP_DecodeJPEG()
*
*	desc:		Decodes the JPEG in the generic data specified by Source, returning it as a 24bit
*				image via the Destination pointer. Returns JXP_SUCCESS or JXP_FAILURE
*
*	notes:		Because the JPEG decoder uses no global variables (all variables are members of the
*				interface class, of which we use an instance here), you could theoretically decode
*				multiple JPEGs in multiple threads at the same time. Why you would want to is an
*				entirely different issue :)
*
**********************************************************************************************************/

int JXP_API_CALL JXP_DecodeJPEG(JXP_Data *Source, JXP_Image_24b **Destination)
{
	JXP_JPEG_Interface *Interface;
	int Ret;

	// Init to NULL
	*Destination=NULL;

	// Get JPEG decoding interface
	Interface=new JXP_JPEG_Interface();

	// Decode JPEG
	Ret=Interface->DecodeJPEG(Source,Destination);

	// Release JPEG decoding interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_DecodeJPEGFile()
*
*	desc:		Decodes the JPEG of the specified source filename, returning it as a 24bit image via
*				the Destination pointer. Returns JXP_SUCCESS or JXP_FAILURE. All it does is read the
*				file into a generic data block, then call JXP_DecodeJPEG() on it.
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_DecodeJPEGFile(char *Filename, JXP_Image_24b **Destination)
{
	JXP_Data *Data;
	FILE *Fptr;
	int Length;
	int Ret;

	// Open file
	Fptr=fopen(Filename,"rb");
	if (!Fptr)
	{
		JXP_Utils.SetLastError(JXP_ERROR_FILE_NOT_FOUND);
		return JXP_FAILURE;
	}

	// Get file length
	fseek(Fptr,0,SEEK_END);
	Length=ftell(Fptr);
	fseek(Fptr,0,SEEK_SET);

	// Create new block of data
	Data=new JXP_Data(Length);

	// Read in file
	fread(Data->GetData(),sizeof(char),Length,Fptr);

	// Close file
	fclose(Fptr);

	// Decode JPEG
	Ret=JXP_DecodeJPEG(Data,Destination);

	// Free data
	delete Data;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_GeneratePalette()
*
*	desc:		Generate a palette from the 24bit image 'Source', with the desired number of colours 
*				(must be between 2 and 256), returning it via 'Destination'.
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_GeneratePalette(JXP_Image_24b *Source, JXP_Palette **Destination, int NumColours, int Flags)
{
	// Just do a multiple-image palette generation with the single image
	return JXP_GeneratePaletteMultiple(&Source,1,Destination,NumColours,Flags);
}

/*********************************************************************************************************
*
*	function:	JXP_GeneratePaletteMultiple()
*
*	desc:		Generate a palette from the 'NumImages' images in the array of 24bit images 'Source', 
*				with the desired number of colours (must be between 2 and 256), returning it via 
*				'Destination'.
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_GeneratePaletteMultiple(JXP_Image_24b **Source, int NumImages, JXP_Palette **Destination, int NumColours, int Flags)
{
	JXP_Dither_Interface *Interface;
	int Ret;

	// Init to NULL
	*Destination=NULL;

	// Get colour reduction interface
	Interface=new JXP_Dither_Interface();

	// Generate palette
	Ret=Interface->GeneratePaletteMultiple(Source,NumImages,Destination,NumColours,Flags);

	// Release colour reduction interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_FitToPalette()
*
*	desc:		Fits the 24bit 'Source' image to the supplied palette, returning an 8bit image via 
*				'Destination'
*
*	notes:
*
**********************************************************************************************************/

int JXP_API_CALL JXP_FitToPalette(JXP_Image_24b *Source, JXP_Palette *Palette, JXP_Image_8b **Destination, JXP_DitherType DitherType)
{
	JXP_Dither_Interface *Interface;
	int Ret;

	// Init to NULL
	*Destination=NULL;

	// Get colour reduction interface
	Interface=new JXP_Dither_Interface();

	// Fit to palette
	Ret=Interface->FitToPalette(Source,Palette,Destination,DitherType);

	// Release colour reduction interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_GetTargaFormat()
*
*	desc:		Examines the TGA file in 'Source', returning it's type via 'Format'
*
*	notes:		Calling the function in the interface class is probably overkill for something this
*				simple...
*
**********************************************************************************************************/

int JXP_API_CALL JXP_GetTargaFormat(JXP_Data *Source, JXP_TargaType *Format)
{
	JXP_Targa_Interface *Interface;
	int Ret;

	// Init to unsupported
	*Format=JXP_TGA_UNSUPPORTED;

	// Get Targa interface
	Interface=new JXP_Targa_Interface();

	// Call method
	Ret=Interface->GetTargaFormat(Source,Format);

	// Release interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_GetTargaFormat()
*
*	desc:		Examines the TGA file 'Filename', returning its type via 'Format'
*
*	notes:		Calling the function in the interface class is probably overkill for something this
*				simple...
*
**********************************************************************************************************/

int JXP_API_CALL JXP_GetTargaFileFormat(char *Filename, JXP_TargaType *Format)
{
	JXP_Targa_Interface *Interface;
	int Ret;

	// Init to unsupported
	*Format=JXP_TGA_UNSUPPORTED;

	// Get Targa interface
	Interface=new JXP_Targa_Interface();

	// Call method
	Ret=Interface->GetTargaFileFormat(Filename,Format);

	// Release interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_Read24bTarga()
*
*	desc:		Read a 24bit TGA in the data block 'Source', returning it via 'Destination'
*
*	notes:		
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Read24bTarga(JXP_Data *Source, JXP_Image_24b **Destination)
{
	JXP_Targa_Interface *Interface;
	int Ret;

	// Init to NULL
	*Destination=NULL;

	// Get Targa interface
	Interface=new JXP_Targa_Interface();

	// Call method
	Ret=Interface->Read24bTarga(Source,Destination);

	// Release interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_Read24bTargaFile()
*
*	desc:		Read the 24bit TGA file 'Filename', returning it via 'Destination'
*
*	notes:		
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Read24bTargaFile(char *Filename, JXP_Image_24b **Destination)
{
	JXP_Data *Data;
	FILE *Fptr;
	int Length;
	int Ret;

	// Open file
	Fptr=fopen(Filename,"rb");
	if (!Fptr)
	{
		JXP_Utils.SetLastError(JXP_ERROR_FILE_NOT_FOUND);
		return JXP_FAILURE;
	}

	// Get file length
	fseek(Fptr,0,SEEK_END);
	Length=ftell(Fptr);
	fseek(Fptr,0,SEEK_SET);

	// Create new block of data
	Data=new JXP_Data(Length);

	// Read in file
	fread(Data->GetData(),sizeof(char),Length,Fptr);

	// Close file
	fclose(Fptr);

	// Read TGA
	Ret=JXP_Read24bTarga(Data,Destination);

	// Free data
	delete Data;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_Read8bTarga()
*
*	desc:		Read an 8bit TGA in the data block 'Source', returning it via 'Destination'
*
*	notes:		
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Read8bTarga(JXP_Data *Source, JXP_Image_8b **Destination)
{
	JXP_Targa_Interface *Interface;
	int Ret;

	// Init to NULL
	*Destination=NULL;

	// Get Targa interface
	Interface=new JXP_Targa_Interface();

	// Call method
	Ret=Interface->Read8bTarga(Source,Destination);

	// Release interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_Read8bTargaFile()
*
*	desc:		Read the 8bit TGA file 'Filename', returning it via 'Destination'
*
*	notes:		
*
**********************************************************************************************************/

int JXP_API_CALL JXP_Read8bTargaFile(char *Filename, JXP_Image_8b **Destination)
{
	JXP_Data *Data;
	FILE *Fptr;
	int Length;
	int Ret;

	// Open file
	Fptr=fopen(Filename,"rb");
	if (!Fptr)
	{
		JXP_Utils.SetLastError(JXP_ERROR_FILE_NOT_FOUND);
		return JXP_FAILURE;
	}

	// Get file length
	fseek(Fptr,0,SEEK_END);
	Length=ftell(Fptr);
	fseek(Fptr,0,SEEK_SET);

	// Create new block of data
	Data=new JXP_Data(Length);

	// Read in file
	fread(Data->GetData(),sizeof(char),Length,Fptr);

	// Close file
	fclose(Fptr);

	// Read TGA
	Ret=JXP_Read8bTarga(Data,Destination);

	// Free data
	delete Data;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_GenerateMask()
*
*	desc:		Generates a mask file from the black and white image 'Source', returning it as a 
*				block of data via 'Destination'
*
*	notes:		
*
**********************************************************************************************************/

int JXP_API_CALL JXP_GenerateMask(JXP_Image_24b *Source, JXP_Data **Destination)
{
	JXP_Mask_Interface *Interface;
	int Ret;

	// Init to NULL
	*Destination=NULL;

	// Get mask interface
	Interface=new JXP_Mask_Interface();

	// Call method
	Ret=Interface->GenerateMask(Source,Destination);

	// Release interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_ReadMask()
*
*	desc:		Read a mask in the data block 'Source', returning it via 'Destination'
*
*	notes:		
*
**********************************************************************************************************/

int JXP_API_CALL JXP_ReadMask(JXP_Data *Source, JXP_Image_GreyScale **Destination)
{
	JXP_Mask_Interface *Interface;
	int Ret;

	// Init to NULL
	*Destination=NULL;

	// Get mask interface
	Interface=new JXP_Mask_Interface();

	// Call method
	Ret=Interface->ReadMask(Source,Destination);

	// Release interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_ReadMaskFile()
*
*	desc:		Read the mask file 'Filename', returning it via 'Destination'
*
*	notes:		
*
**********************************************************************************************************/

int JXP_API_CALL JXP_ReadMaskFile(char *Filename, JXP_Image_GreyScale **Destination)
{
	JXP_Data *Data;
	FILE *Fptr;
	int Length;
	int Ret;

	// Open file
	Fptr=fopen(Filename,"rb");
	if (!Fptr)
	{
		JXP_Utils.SetLastError(JXP_ERROR_FILE_NOT_FOUND);
		return JXP_FAILURE;
	}

	// Get file length
	fseek(Fptr,0,SEEK_END);
	Length=ftell(Fptr);
	fseek(Fptr,0,SEEK_SET);

	// Create new block of data
	Data=new JXP_Data(Length);

	// Read in file
	fread(Data->GetData(),sizeof(char),Length,Fptr);

	// Close file
	fclose(Fptr);

	// Read Mask
	Ret=JXP_ReadMask(Data,Destination);

	// Free data
	delete Data;

	// Return status
	return Ret;
}

/*********************************************************************************************************
*
*	function:	JXP_ResampleImage()
*
*	desc:		Resamples the image 'Source' to the dimensions 'Width'*'Height', using the resampling 
*				method 'ResampleType', returning it via 'Destination'
*
*	notes:		
*
**********************************************************************************************************/

int JXP_API_CALL JXP_ResampleImage(JXP_Image_24b *Source, int Width, int Height, JXP_Image_24b **Destination, JXP_ResampleType ResampleType)
{
	JXP_Resample_Interface *Interface;
	int Ret;

	// Init to NULL
	*Destination=NULL;

	// Get resample interface
	Interface=new JXP_Resample_Interface();

	// Call method
	Ret=Interface->ResampleImage(Source,Width,Height,Destination,ResampleType);

	// Release interface
	delete Interface;

	// Return status
	return Ret;
}

/*********************************************************************************************************/
