/*****************************************************************************

JiNX Network System (JNet) - IPX Network Driver - By Mark Hodson, 1998-99

This library is freeware.  The only provision is that the programme using this
code must display in its credits (if it has any) an acknowledgement of the use
and original source of this code.

ie. "JNet IPX Network System by Mark Hodson".

*****************************************************************************/

// Structures ///////////////////////////////////////////////////////////////

typedef unsigned short IPX_Socket;

typedef _Packed struct {

    unsigned short packet_checksum;
    unsigned short packet_length;
    unsigned char  transport_control;
    unsigned char  packet_type;
    unsigned char  destination_address[10];
    IPX_Socket     destination_socket;
    unsigned char  source_address[10];
    IPX_Socket     source_socket;

    char packet_data[546];

} IPX_Packet;


// Functions and Variables //////////////////////////////////////////////////

extern char *JNet_IPX_ErrorString[];
extern char *JNet_IPX_StatusString[];

// Initialise the JNet system.

// Params : Number of packets you wish to handle at one time.
// Returns: Error code (0 => No error)
// Sets   : Maximum packet size (ignore this the real limit is 546 bytes).
//          IPX Retry counter.
//          Your address on the network.

extern int __syscall JNet_IPX_Initialise(int number_of_packet_handlers);
extern unsigned short __syscall IPX_MaximumPacketSize;
extern unsigned short __syscall IPX_RetryCount;
extern char __syscall IPX_NetworkAddress[10];


// Open Socket opens an IPX socket for sending or receiving data.

// Params : 16bit socket number, or 0x0000 for IPX to choose one for you.
// Returns: Error code (0 => No error)
// Sets   : The successfully opened socket number - save this in a local
//            variable immediately after calling Open Socket.

extern int __syscall JNet_IPX_OpenSocket(IPX_Socket socket);
extern IPX_Socket __syscall IPX_LastSocketOpened;


// Close Socket closes an open IPX socket.

// Returns: Error code (0 => No error)

extern int __syscall JNet_IPX_CloseSocket(IPX_Socket socket);


// Broadcast Packet sends a packet to all local nodes listening on a
// particular socket.

// Params : The 16bit local socket to send from.
//          The 16bit destination socket to send to.
//          The data you wish to send.
//          The length of the data you wish to send (max 546 bytes).
// Returns: Error code (0 => No error)

extern int __syscall JNet_IPX_BroadcastPacket(IPX_Socket socket, IPX_Socket dest_socket,
                                              void *data, int data_length);


// Send Packet sends a packet to a particular socket on a particular local
// node (identified by its network address).

// Params : The 16bit local socket to send from.
//          The full 10 byte destination network address to send to.
//          The 16bit destination socket to send to.
//          The data you wish to send.
//          The length of the data you wish to send (max 546 bytes).
// Returns: Error code (0 => No error)

extern int __syscall JNet_IPX_SendPacket(IPX_Socket socket,
                                         char *dest_address, IPX_Socket dest_socket,
                                         void *data, int data_length);


// Receive Packet sets a packet handler to receive a packet incident on a
// particular local socket.

// Params : The 16bit local socket to check.
// Returns: Error code (0 => No error)

extern int __syscall JNet_IPX_ReceivePacket(IPX_Socket socket);


// Relinquish Control can be called in idle time to give the IPX driver extra
// time to perform its tasks.  This function is called once when you call
// Check Status.

// Returns: Error code (0 => No error)

extern int __syscall JNet_IPX_RelinquishControl(void);


// Check Status looks at the packet handlers to see if any have completed
// their transaction.

// Returns: Status code         00h => No packet handlers have completed.
//                              10h => Successfully received.
//                              20h => Successfully sent.
//                              Others => Various error codes.
// Sets   : Pointer to a complete packet, or NULL if Status Code = 0.  You
//            must deal with the packet before calling Check Status again, as
//            a subsequent call will destroy the packet in order to free the
//            handler for future use.

extern int __syscall JNet_IPX_CheckStatus(void);
extern IPX_Packet * __syscall IPX_LastPacketReported;


// How Many Spare tells you how many packet handlers are currently available.
// How Many Sending tells you how many packet handlers are receiving data.
// How Many Sending tells you how many packet handlers are sending data.

// Returns: Number of packet handlers available/receiving/sending.

extern int __syscall JNet_IPX_HowManySpare(void);
extern int __syscall JNet_IPX_HowManyReceiving(void);
extern int __syscall JNet_IPX_HowManySending(void);


// IPX Address Registry Convenience Functions

int JNet_IPX_InitialiseRegistry(int number_of_entries);

int JNet_IPX_RegisterAddress(char *address);

int JNet_IPX_CheckAddress(char *address);

int JNet_IPX_HowManyAddresses(void);

char *JNet_IPX_ReturnAddress(int index);



