; Yule Get Dizzy
; by 'Diminished'

; v2: 26th Dec 2023

; Assemble using beebasm: https://github.com/stardot/beebasm/
;   beebasm -i source.s -do disc.ssd -boot DEMO8 -v
; to get a disc image that will autoboot.

; Hi and thanks to the Stardot crew.

; Try BEAM_AVOIDANCE 0 or 1, to build with or without flicker correction.

SPACING                           = 8   ; spacing between asterisks
VSYNC_BEFORE_SHAPE                = 0   ; vsync before every full shape redraw (for an FPS penalty)
BEAM_AVOIDANCE                    = 1   ; activate anti-flicker deferral mechanism
BEAM_EVASION_LOWER_BOUND_MODIFIER = 0   ; set > 0 to increase lower bound of deferral stripe range
BEAM_EVASION_UPPER_BOUND_MODIFIER = 0   ; set > 0 to reduce upper bound of deferral stripe range
DISABLE_INTERLACING               = 1

TEST                      = 0
TEST2                     = 0
TEST3                     = 0

OSBYTE                    = &FFF4
OSWRCH                    = &FFEE
VRAM_H                    = &5D
T_MOS_FONT                = &C000
VRAM_TEXT_TARGET          = &5800
videoULAPaletteRegister   = &FE21
VSYNC_OSBYTE              = &13
T_SAVED_COORDINATES_X     = &5700   ; put saved coords just below VRAM
T_SAVED_COORDINATES_Y     = &5780   ; put saved coords just below VRAM

IF BEAM_AVOIDANCE
zp_old_irq1v_L            = &71
zp_old_irq1v_H            = &72
zp_timer_started          = &73
timer_latch               = 312*64
USERVIA_ACR               = &FE6B
USERVIA_IER               = &FE6E
USERVIA_T1VAL_L           = &FE64
USERVIA_T1VAL_H           = &FE65
ENDIF

zp_rot_angle                = &0
zp_rot_angle_next           = &1
zp_ptr_trig_L               = &2
zp_ptr_trig_H               = &3
zp_v_shape_ix_doubled       = &4
zp_v_shape_ix               = &5
zp_shape_num_pts_doubled    = &7
zp_ptr_shape_minus_1_L      = &8
zp_ptr_shape_minus_1_H      = &9
zp_ptr_shape_minus_2_L      = &A
zp_ptr_shape_minus_2_H      = &B
zp_X                        = &C
zp_Y                        = &D
zp_X_voff_block_in_stripe   = &E
zp_leftblock_topline_L      = &F
zp_leftblock_topline_H      = &10
zp_rightblock_topline_L     = &11
zp_rightblock_topline_H     = &12
zp_leftblock_lower_L        = &13
zp_leftblock_lower_H        = &14
zp_rightblock_lower_L       = &15
zp_rightblock_lower_H       = &16
zp_overlap_Y                = &17
zp_t_star_left_L            = &18
zp_t_star_left_H            = &19
zp_t_star_right_L           = &1A
zp_t_star_right_H           = &1B
zp_sub_Y                    = &1C
zp_main_Y                   = &1D
zp_rotations_doubled        = &1E
zp_ptr_mos_font_L           = &1F
zp_ptr_mos_font_H           = &20
zp_ptr_string_L             = &21
zp_ptr_string_H             = &22
zp_cur_char_row             = &23
zp_ptr_text_vram_L          = &24
zp_ptr_text_vram_H          = &25
zp_script_scene             = &26
zp_shape_no_erase_oneshot   = &27
zp_delay_count              = &28
zp_script_jumptable_L       = &29
zp_script_jumptable_H       = &2a
zp_scene_duration           = &2b
zp_colour                   = &2c
IF BEAM_AVOIDANCE
  IF TEST
    zp_TEST_timer_H_at_vsync  = &2d
  ENDIF
  zp_deferred_stars_count   = &2e
ENDIF
zp_ptr_saved_coords_x_L     = &2f
zp_ptr_saved_coords_x_H     = &30
zp_ptr_saved_coords_y_L     = &31
zp_ptr_saved_coords_y_H     = &32


; ----------

org &1900

.top

; LUTs for going ASCII char -> MOS font lookup table pointer
.T_CHAR_TO_FONT_PTR_LOW
FOR X,0,31
  EQUB 0
NEXT
FOR X,32,126
  EQUB &FF AND (&C000 + 8*(X-32))
NEXT
.T_CHAR_TO_FONT_PTR_HIGH
FOR X,0,31
  EQUB 0
NEXT
FOR X,32,126
  EQUB &FF AND ((&C000 + 8*(X-32)) >> 8)
NEXT

; ----------

org &1A00

; STRINGS
; max 32 chars, due to optimisation of display routine.
; Also, no individual string can straddle a page boundary here.

.T_STRING_NULL
EQUB 0
.T_STRING_HELLO
EQUS "Season's Greetings.",0
.T_STRING_LOGIKER
EQUS "Logiker requested this shape:   ", 0
.T_STRING_WONDERING
EQUS "You might be wondering why it   ", 0
.T_STRING_TAKES
EQUS "takes 14K of code to display    ", 0
.T_STRING_ONE_SIMPLE
EQUS "one simple design. Well, maybe  ", 0
.T_STRING_ROTATE
EQUS "we could ... rotate it?         ", 0
.T_STRING_MIRACLE
EQUS "It's a Christmas miracle!       ", 0
.T_STRING_SPENT_8K
EQUS "I spent 8K on one lookup table, ", 0

org &1B00
.T_STRING_ENOUGH_RAM_LEFT
EQUS "but is there enough RAM left for", 0
.T_STRING_SOME_MORE_SHAPES
EQUS "some more shapes? How about a   ", 0
.T_STRING_NICE_STAR
EQUS "nice Christmas star for my      ", 0
.T_STRING_STARDOT
EQUS "associates at Stardot?          ", 0
.T_STRING_SNOWFLAKE
EQUS "Perhaps a snowflake? Although I ", 0
.T_STRING_GOING_TO_RAIN
EQUS "think it's just going to rain on", 0
.T_STRING_CHRISTMAS_DAY
EQUS "Christmas Day this year. Anyway:", 0

org &1d00
.T_STRING_MERRY
EQUS "A very merry Christmas to all!  ", 0
.T_STRING_BLANK
EQUS "                                ",0

; flicker defence :/ (312 x 64) = 19968 = &4e00
;   - each "stripe" is 4px tall
;   (0,0) in M_star coords is like (0,32) [top-left] in real 320x256 ones
;   - so 8 stripes from top
;   - do (zp_Y / 4), get first stripe
; timer high values:
;   &4D is the value measured immediately after vsync ...
;   &3E is the value when zp_Y=0

;  (blanking)
; &46 ======... |
; &45 ======... |
; &44 ======... |
; &43 ======... |
; &42 ======... |
; &41 ======... |
; &40 ======... |
; &3F ======... V
; &3E IS FIRST VISIBLE STRIPE, where zp_Y=0 (measured empirically)

; Star always overlaps exactly 2 stripes (usefully):
; ----------------------
; ---------------------- &3E
; ----------------------
; ---**-----------------
; .******.......**......
; ..****......******.... &3D
; .******......****.....
; ...**.......******....
; --------------**------
; ---------------------- &3C ...
; ----------------------
; ----------------------
; If zp_Y is 0..3  then collides with stripes &3e, &3d
; If zp_Y is 4..7  then collides with stripes &3d, &3c
; If zp_Y is 8..11 then collides with stripes &3c, &3b

; so colliding stripes are (&3e - (zp_Y DIV 4)) and (&3e - (1 + (zp_Y DIV 4)))
; HOWEVER
; drawing the star will take some time ...
;   - one star takes about 1 stripe's time to draw (measured empirically)
;   - we also have to erase it of course, so that's 2 stripes' worth
;   - additionally, between being drawn and erased, the star will move -- so
;     potentially up to another stripe's worth of vertical movement in either
;     direction may occur
; SO each star needs to respect:
; - its origin stripe as given by (zp_Y DIV 4);
; - two stripes above that, to allow drawing time;
; - the stripe below it, with which it overlaps;
; - two more stripes top and bottom, to allow for for vertical movement between draw and erase

; So the proscribed stripes are like so:

; ----------------------
; ---------------------- &3E
; ---------------------- Y = 0-3
; ----------------------
; ......................
; ...................... &3D
; ...................... Y = 4-7
; ......................
; ----------------------
; ---------------------- &3C
; ---------------------- Y = 8-11
; ----------------------
; ......................
; ...................... &3B
; ........**............ Y = 12-15
; ......******..........
; -------****-----------
; ------******---------- &3A
; --------**------------ Y = 16-19
; ----------------------
; ......................
; ...................... &39
; ...................... Y = 20-23
; ......................

; and so
; - if zp_Y is 0..3   then collides with stripes 41, 40, 3F, 3E, 3D, 3C
; - if zp_Y is 4..7   then collides with stripes 40, 3F, 3E, 3D, 3C, 3B
; - if zp_Y is 8..11  then collides with stripes 3F, 3E, 3D, 3C, 3B, 3A
; - if zp_Y is 12..15 then collides with stripes 3E, 3D, 3C, 3B, 3A, 39
; - if zp_Y is 16..19 then collides with stripes 3D, 3C, 3B, 3A, 39, 38
; - if zp_Y is 20..23 then collides with stripes 3C, 3B, 3A, 39, 38, 37
; etc

; ----------
; or, reversed (put this in the LUT)
; stripe &3e excludes zp_Y values  0..15  } careful with these high values
; stripe &3d excludes zp_Y values  0..19  }
; stripe &3c excludes zp_Y values  0..23 <- standard 24 range from here up
; stripe &3b excludes zp_Y values  4..27
; stripe &3a excludes zp_Y values  8..31
; stripe &39 excludes zp_Y values 12..35
; and so on:

IF BEAM_AVOIDANCE
org &1e00
; beam avoidance LUTs
; lower/upper bound on acceptable zp_Y coordinate
; for drawing, given read timer high byte (a.k.a. "stripe")
; if zp_Y coordinate falls in the range _LOWER_ -> _UPPER_ below,
; drawing is not permitted and a deferral will occur
.T_BEAM_EVASION_TMR_H_TO_Y_LOWER_BOUND {
M=BEAM_EVASION_LOWER_BOUND_MODIFIER
FOR S,0,(M + &3C)
  EQUB (((M + &3C)-S) * 4)
NEXT
FOR S,(M+&3D),(M+&41)
  EQUB 0    ; these (stripes 3d-41) are off the top of our "screen"
NEXT        ; but may still collide
FOR X,(M+&42),(M+&4E)
  EQUB 255  ; these ones (42 and up) are irrelevant
NEXT
}

; CAUTION: all values in this table must be +1
; so we can use BCC to check for strictly greater-than!
; (BCC is normally >= )
org &1e80
.T_BEAM_EVASION_TMR_H_TO_Y_UPPER_BOUND {
M=BEAM_EVASION_UPPER_BOUND_MODIFIER
FOR S,0,(2-M)
  EQUB 255
NEXT
FOR S,(3-M),(&3C-M)
  EQUB 1 + (((&3C-M)-S) * 4) + 23
NEXT
FOR S,(&3D-M),(&41-M)
  EQUB 1 + (((&3D-M)-S) * 4) + 19 ; these (stripes 3d-41) are off the top of our "screen"
NEXT                              ; but may still collide
FOR X,(&42-M),(&4E-M)
  EQUB 0  ; these ones (42 and up) are irrelevant
NEXT
}

ENDIF ; BEAM_AVOIDANCE

; ----------

org &2000 ; -- MONSTER TRIG LUT --

MACRO M_TRIG_TERMS_FOR_RADIUS_ID __X, __Y, _A
  _X = SPACING * __X
  _Y = SPACING * __Y
  ___X = 157 + INT(COS((_A*PI)/64.0)*SQR((_X*_X)+(_Y*_Y))) ; X
  ___Y = 128 + INT(SIN((_A*PI)/64.0)*SQR((_X*_X)+(_Y*_Y))) ; Y
  ;PRINT ___X,___Y
  EQUB ___X
  EQUB ___Y
ENDMACRO

NUM_RADIUS_IDS=13

; build 8192-byte LUT for trig
.T_TRIG
FOR A,0,127
  ; Coordinates of various cells in the shape
  ; are used here to build an exhaustive list of every
  ; distinct polar-coordinate radius in the
  ; shape. These are then turned into X and Y
  ; coordinates and stored for every possible
  ; angle from 0 to 127.
  M_TRIG_TERMS_FOR_RADIUS_ID  0, 3, A  ; radius ID 0 or A
  M_TRIG_TERMS_FOR_RADIUS_ID  2, 1, A  ; radius ID 1 or B
  M_TRIG_TERMS_FOR_RADIUS_ID  1, 4, A  ; radius ID 2 or C
  M_TRIG_TERMS_FOR_RADIUS_ID  2, 5, A  ; radius ID 3 or D
  M_TRIG_TERMS_FOR_RADIUS_ID  3, 6, A  ; radius ID 4 or E
  M_TRIG_TERMS_FOR_RADIUS_ID  2, 7, A  ; radius ID 5 or F
  M_TRIG_TERMS_FOR_RADIUS_ID  1, 8, A  ; radius ID 6 or G
  M_TRIG_TERMS_FOR_RADIUS_ID  0, 9, A  ; radius ID 7 or H
  M_TRIG_TERMS_FOR_RADIUS_ID  5, 4, A  ; radius ID 8 or I
  M_TRIG_TERMS_FOR_RADIUS_ID  4, 7, A  ; radius ID 9 or J
  M_TRIG_TERMS_FOR_RADIUS_ID  5, 8, A  ; radius ID 10 or K
  M_TRIG_TERMS_FOR_RADIUS_ID  6, 9, A  ; radius ID 11 or L
  M_TRIG_TERMS_FOR_RADIUS_ID  7, 8, A  ; radius ID 12 or M
  
  M_TRIG_TERMS_FOR_RADIUS_ID  1, 1, A  ; radius ID 13 or N
  M_TRIG_TERMS_FOR_RADIUS_ID  2, 2, A  ; radius ID 14 or O
  M_TRIG_TERMS_FOR_RADIUS_ID  3, 3, A  ; radius ID 15 or P
  M_TRIG_TERMS_FOR_RADIUS_ID  4, 4, A  ; radius ID 16 or Q
  M_TRIG_TERMS_FOR_RADIUS_ID  5, 5, A  ; radius ID 17 or R
  M_TRIG_TERMS_FOR_RADIUS_ID  6, 6, A  ; radius ID 18 or S
  M_TRIG_TERMS_FOR_RADIUS_ID  7, 7, A  ; radius ID 19 or T
  M_TRIG_TERMS_FOR_RADIUS_ID  8, 8, A  ; radius ID 20 or U
  
  M_TRIG_TERMS_FOR_RADIUS_ID  3, 4, A  ; radius ID 21 or V
  M_TRIG_TERMS_FOR_RADIUS_ID  3, 5, A  ; radius ID 22 or W
  M_TRIG_TERMS_FOR_RADIUS_ID  6, 7, A  ; radius ID 23 or X
  M_TRIG_TERMS_FOR_RADIUS_ID  6, 8, A  ; radius ID 24 or Y
  
  EQUB 0,0,0,0,0,0,0,0
  EQUB 0,0,0,0,0,0 ; padding to 64 bytes per angle-unit
NEXT

; ----------

org &4000 ; -- SHAPE LUT --

; inserts a shape's point, as polars: (radius ID), (angle)
; takes:
;   radius ID
;   X    (X,Y) are used purely to
;   Y    compute an angle
;   ROT  additional rotation (in angle-units of 128 per full rotation)
MACRO M_SHAPE_ONE_POINT _RID, __X, __Y, __ROT {
  _X = SPACING * __X
  _Y = SPACING * __Y
  IF _X=0
    _UNTAN = PI/2
  ELSE
    _UNTAN = ATN(_Y / _X)
  ENDIF
  _A = INT((_UNTAN * 63.0) / PI)
  EQUB _RID*2, (_A + __ROT)   ; bake RIDx2 into the table, to save an ASL later
} ENDMACRO

; similar to _ONE_POINT, but lacks the extra ROT argument
; and duplicates the point three times, every 90 degrees around the circle
MACRO M_SHAPE_FOUR_POINTS _RID, __X, __Y {
  _X = SPACING * __X
  _Y = SPACING * __Y
  IF _X=0
    _UNTAN = PI/2
  ELSE
    _UNTAN = ATN(_Y / _X)
  ENDIF
  _A = INT((_UNTAN * 63.0) / PI)
  FOR Q, 0, 96, 32
    EQUB _RID*2, (Q + _A)   ; bake RIDx2 into the table, to save an ASL later
  NEXT
} ENDMACRO

.T_SHAPE_1
  ; (radius ID, X, Y) => (RID, ANGLE), (RID, ANGLE+32), (RID, ANGLE+64), (RID, ANGLE+96)
  ; NOTE that radius ID pasted into the table will be doubled, to avoid having to do an ASL when it's used
  M_SHAPE_FOUR_POINTS   0, 0, 3 ; A
  M_SHAPE_FOUR_POINTS   1, 1, 2 ; B
  M_SHAPE_FOUR_POINTS   1, 2, 1 ; B
  M_SHAPE_FOUR_POINTS   2, 4, 1 ; C
  M_SHAPE_FOUR_POINTS   2, 1, 4 ; C
  M_SHAPE_FOUR_POINTS   3, 5, 2 ; D
  M_SHAPE_FOUR_POINTS   3, 2, 5 ; D
  M_SHAPE_FOUR_POINTS   4, 6, 3 ; E
  M_SHAPE_FOUR_POINTS   4, 3, 6 ; E
  M_SHAPE_FOUR_POINTS   5, 7, 2 ; F
  M_SHAPE_FOUR_POINTS   5, 2, 7 ; F
  M_SHAPE_FOUR_POINTS   6, 8, 1 ; G
  M_SHAPE_FOUR_POINTS   6, 1, 8 ; G
  M_SHAPE_FOUR_POINTS   7, 0, 9 ; H
  M_SHAPE_FOUR_POINTS   8, 4, 5 ; I
  M_SHAPE_FOUR_POINTS   8, 5, 4 ; I
  M_SHAPE_FOUR_POINTS   9, 4, 7 ; J
  M_SHAPE_FOUR_POINTS   9, 7, 4 ; J
  M_SHAPE_FOUR_POINTS  10, 8, 5 ; K
  M_SHAPE_FOUR_POINTS  10, 5, 8 ; K
  M_SHAPE_FOUR_POINTS  11, 8, 6 ; L
  M_SHAPE_FOUR_POINTS  11, 6, 9 ; L
  M_SHAPE_FOUR_POINTS  12, 8, 7 ; M
  M_SHAPE_FOUR_POINTS  12, 7, 8 ; M
.shape1_end
SHAPE_1_NUM_POINTS = (shape1_end - T_SHAPE_1) / 2
PRINT "shape 1 has",SHAPE_1_NUM_POINTS,"points"

.T_SHAPE_2
  ; (radius ID, X, Y) => (RID, ANGLE), (RID, ANGLE+32), (RID, ANGLE+64), (RID, ANGLE+96)
  ; NOTE that radius ID pasted into the table will be doubled, to avoid having to do an ASL when it's used
  M_SHAPE_FOUR_POINTS  13, 0, 1 ; N
  M_SHAPE_FOUR_POINTS  14, 0, 1 ; O
  M_SHAPE_FOUR_POINTS  15, 0, 1 ; P
  M_SHAPE_FOUR_POINTS  16, 0, 1 ; Q
  M_SHAPE_FOUR_POINTS  17, 0, 1 ; R
  M_SHAPE_FOUR_POINTS  18, 0, 1 ; S
  M_SHAPE_FOUR_POINTS  19, 0, 1 ; T
  M_SHAPE_FOUR_POINTS  20, 0, 1 ; U
  
  M_SHAPE_FOUR_POINTS  13, 1, 1 ; N
  M_SHAPE_FOUR_POINTS  14, 1, 1 ; O
  M_SHAPE_FOUR_POINTS  15, 1, 1 ; P
  M_SHAPE_FOUR_POINTS  16, 1, 1 ; Q
  M_SHAPE_FOUR_POINTS  17, 1, 1 ; R
  M_SHAPE_FOUR_POINTS  18, 1, 1 ; S
  M_SHAPE_FOUR_POINTS  19, 1, 1 ; T
  M_SHAPE_FOUR_POINTS  20, 1, 1 ; U
{
U=15
V=6
  M_SHAPE_FOUR_POINTS  15, V, U ; P
  M_SHAPE_FOUR_POINTS  16, V, U ; Q
  M_SHAPE_FOUR_POINTS  17, V, U ; R
  M_SHAPE_FOUR_POINTS  18, V, U ; S
  
  M_SHAPE_FOUR_POINTS  15, U, V ; P
  M_SHAPE_FOUR_POINTS  16, U, V ; Q
  M_SHAPE_FOUR_POINTS  17, U, V ; R
  M_SHAPE_FOUR_POINTS  18, U, V ; S
}
.shape2_end
SHAPE_2_NUM_POINTS = (shape2_end - T_SHAPE_2) / 2
PRINT "shape 2 has",SHAPE_2_NUM_POINTS,"points"

.T_SHAPE_3 {
FOR R,0,(128.0/6.0)*5.0,(128.0/6.0)
  M_SHAPE_ONE_POINT  13, 1, 1, R
  M_SHAPE_ONE_POINT  14, 1, 1, R
  M_SHAPE_ONE_POINT  15, 1, 1, R
  M_SHAPE_ONE_POINT  21, 3, 4, R
  M_SHAPE_ONE_POINT  22, 3, 5, R
  M_SHAPE_ONE_POINT  21, 4, 3, R
  M_SHAPE_ONE_POINT  22, 5, 3, R
  M_SHAPE_ONE_POINT  16, 1, 1, R
  M_SHAPE_ONE_POINT  17, 1, 1, R
  M_SHAPE_ONE_POINT  18, 1, 1, R
  
  M_SHAPE_ONE_POINT  23, 6, 7, R
  M_SHAPE_ONE_POINT  24, 6, 8, R
  M_SHAPE_ONE_POINT  23, 7, 6, R
  M_SHAPE_ONE_POINT  24, 8, 6, R
  
  M_SHAPE_ONE_POINT  19, 1, 1, R
  M_SHAPE_ONE_POINT  20, 1, 1, R
  
NEXT
}
.shape3_end
SHAPE_3_NUM_POINTS = (shape3_end - T_SHAPE_3) / 2
PRINT "shape 3 has",SHAPE_3_NUM_POINTS,"points"

; ----------

org &4300 ; -- MISC LUTs --
  
; asterisk graphics, based on degree of X-shift
; must be page-aligned
.T_STAR_LEFT
  EQUB &30,&FC,&78,&FC,&30,0,0,0
  EQUB &18,&7E,&3C,&7E,&18,0,0,0
  EQUB &0C,&3F,&1E,&3F,&0C,0,0,0
  EQUB 6,&1F,&0F,&1F,6,0,0,0
  EQUB 3,&F,7,&F,3,0,0,0
  EQUB 1,7,3,7,1,0,0,0
  EQUB 0,3,1,3,0,0,0,0
  EQUB 0,1,0,1,0,0,0,0
  EQUB 0,0,0,0,0,0,0,0
  EQUB 0,0,0,0,0,0,0,0
  EQUB 0,0,0,0,0,0,0,0
  EQUB 0,&80,0,&80,0,0,0,0
  EQUB 0,&C0,&80,&C0,0,0,0,0
  EQUB &80,&E0,&C0,&E0,&80,0,0,0
  EQUB &C0,&F0,&E0,&F0,&C0,0,0,0
  EQUB &60,&F8,&F0,&F8,&60,0,0,0
  
.T_X320
FOR X,0,31
  EQUW X*320
NEXT
  
org &4400

; must be page-aligned
.T_STAR_RIGHT
  EQUB 0,0,0,0,0,0,0,0
  EQUB 0,0,0,0,0,0,0,0
  EQUB 0,0,0,0,0,0,0,0
  EQUB 0,&80,0,&80,0,0,0,0
  EQUB 0,&C0,&80,&C0,0,0,0,0
  EQUB &80,&E0,&C0,&E0,&80,0,0,0
  EQUB &C0,&F0,&E0,&F0,&C0,0,0,0
  EQUB &60,&F8,&F0,&F8,&60,0,0,0
  EQUB &30,&FC,&78,&FC,&30,0,0,0
  EQUB &18,&7E,&3C,&7E,&18,0,0,0
  EQUB &C,&3F,&1E,&3F,&C,0,0,0
  EQUB &6,&1F,&F,&1F,&6,0,0,0
  EQUB &3,&F,&7,&F,&3,0,0,0
  EQUB 1,7,3,7,1,0,0,0
  EQUB 0,3,1,3,0,0,0,0
  EQUB 0,1,0,1,0,0,0,0

; ostensibly multiplies by 64, but adds &20 to result
; because that's where the 8K lookup table begins (&2000)
.T_ANGLE_TO_HIGH_T_TRIG_PTR
FOR M,0,127
  EQUB ((M>>2) AND &1F) OR &20    ; 8K LUT ptr high byte
NEXT

.T_ANGLE_TO_LOW_T_TRIG_PTR
FOR M,0,127
  EQUB ((M<<6) AND &FF)           ; 8K LUT ptr low byte
NEXT

.T_X8_BITS_345
FOR X,0,255
  EQUB ((X<<3) AND &38)
NEXT

; ----------
  
; set zp_shape_no_erase_oneshot=1 to draw shape initially;
; subsequent inclusions should set this to 0
; to erase and then draw the shape
.S_shape

IF VSYNC_BEFORE_SHAPE
  LDA   #VSYNC_OSBYTE
  JSR   OSBYTE
ENDIF

IF BEAM_AVOIDANCE
  LDA   #0
  STA   zp_deferred_stars_count
ENDIF

  INC   zp_delay_count

  ; process shape, point-by-point
  
  LDY   zp_shape_num_pts_doubled
  ;BEQ   L_shape_done              ; shape is NULL, quit
  
  STY   zp_v_shape_ix_doubled     ; shape_ix = num points in shape
  TYA
  LSR   A
  STA   zp_v_shape_ix
  
  ; FIXME: could just replace with "INC zp_rot_angle" now
  ; and eliminate zp_rot_angle_next?
  LDX   zp_rot_angle
  INX
  STX   zp_rot_angle_next
  
.P_shape

  ; go point-by-point; erase each point, then plot each point
  LDX   zp_shape_no_erase_oneshot
  BNE   L_shape_once
  
  ; twice?

  ;LDA   zp_rot_angle
  ;JSR   S_shape_recompute_coords
  LDY   zp_v_shape_ix
  ; retrieve saved X and Y coordinates from previous plot for this shape ix
  LDA   (zp_ptr_saved_coords_x_L),Y
  STA   zp_X
  LDA   (zp_ptr_saved_coords_y_L),Y
  STA   zp_Y
  
  ; only do beam avoidance for "twice" (erase+draw) cases, not one-shots
IF BEAM_AVOIDANCE
  ;LDA   zp_Y
  LDY   USERVIA_T1VAL_H           ; get current timer value and check against table
  CMP   T_BEAM_EVASION_TMR_H_TO_Y_LOWER_BOUND,Y ; compare zp_Y against T_*_LOWER_BOUND[t1val]
  BCC   L_not_deferred            ; if zp_Y < lower bound then plot is OK
  CMP   T_BEAM_EVASION_TMR_H_TO_Y_UPPER_BOUND,Y ; compare zp_Y against T_*_UPPER_BOUND[t1val]
  BCS   L_not_deferred            ; RECALL: the upper bound table is all +1, so we can use BCC like >
  ;PHA                             ; push Y-coordinate ...
  ;LDA   zp_X
  ;PHA                             ; push X-coordinate ...
  ;TYA
  LDA   zp_v_shape_ix
  PHA                             ; push shape index.
  INC   zp_deferred_stars_count
  BNE   L_shape_draw_done         ; having deferred this point, skip it and tackle the next one
.L_not_deferred
ENDIF

  JSR   S_shape_point                 ; erase prev
  LDA   zp_rot_angle_next
  JSR   S_shape_recompute_coords      ; returns Xc in X, Yc in A
  JSR   S_shape_point                 ; draw next

  CLC
  BCC   L_shape_draw_done
  
.L_shape_once

  ; or once

  LDA   zp_rot_angle_next ; ???
  JSR   S_shape_recompute_coords
  JSR   S_shape_point
  
.L_NOP
  NOP
  
.L_shape_draw_done

  DEC   zp_v_shape_ix
  DEC   zp_v_shape_ix_doubled             ; twice because table field is (X,Y)
  DEC   zp_v_shape_ix_doubled             ; FIXME: eliminate doubled ...
  
  BNE   P_shape
  

  
IF BEAM_AVOIDANCE
  ; do deferred cleanup!
  LDA   zp_deferred_stars_count
  BEQ   L_shape_done
.P_deferred_cleanup
  PLA                                     ; retrieve shape index
  STA   zp_v_shape_ix
  TAY
  ASL   A
  STA   zp_v_shape_ix_doubled
  LDA   (zp_ptr_saved_coords_x_L),Y
  STA   zp_X
  LDA   (zp_ptr_saved_coords_y_L),Y
  STA   zp_Y
  JSR   S_shape_point                     ; erase, using saved coordinates
  LDA   zp_rot_angle_next
  JSR   S_shape_recompute_coords          ; recompute (and save) new coordinates
  JSR   S_shape_point                     ; draw using new coordinates
  DEC   zp_deferred_stars_count
  BNE   P_deferred_cleanup
ENDIF
  
.L_shape_done
  RTS
  
; -----------

.S_delay
  LDA   #VSYNC_OSBYTE
  PHA
  PHA
  JSR   OSBYTE
IF TEST AND BEAM_AVOIDANCE
  LDA   USERVIA_T1VAL_H
  STA   zp_TEST_timer_H_at_vsync
ENDIF
  PLA
  JSR   OSBYTE
  PLA
  JSR   OSBYTE
  RTS

; ----------

MACRO M_set_shape _T, _NP {
  _T1 = _T-1
  LDY   #LO(_T1)
  STY   zp_ptr_shape_minus_1_L
  DEY
  STY   zp_ptr_shape_minus_2_L
  LDY   #HI(_T1)
  STY   zp_ptr_shape_minus_1_H
  LDY   #HI(_T1-1)
  STY   zp_ptr_shape_minus_2_H
  LDA   #(2*_NP)
  STA   zp_shape_num_pts_doubled
} ENDMACRO


IF BEAM_AVOIDANCE AND TEST
MACRO M_test_timer_stripes
  LDA   USERVIA_T1VAL_H         ; get timer high byte
IF TEST2
  CMP   #&3e                    ; if &3e then white, else black
  BEQ   L_TEST2_A
  LDA   #0
  JMP   L_TEST2_B
.L_TEST2_A
  LDA   #7
.L_TEST2_B
ELSE
  AND   #&7
ENDIF
  M_set_bgcolour                ; VDU 19
ENDMACRO
ELSE
MACRO M_test_timer_stripes
ENDMACRO
ENDIF

; ----------


; set _FAST_RTS_EXIT to 1 to allow faster early exits
; via auto RTS
{
MACRO M_star _FAST_RTS_EXIT

IF TEST3
  LDA   #1
  M_set_bgcolour
ENDIF
  
  LDA   zp_X
  AND   #&F8
  STA   zp_X_voff_block_in_stripe ; "V_BASE" VRAM X-offset within each stripe
  
  ; populate SUB_Y
  LDA   zp_Y
  AND   #7
  STA   zp_sub_Y
  SEC
  SBC   #3                        ; OVERLAP_Y=(SUB_Y-3)
  BPL   L_offset_Y_nonzero
  LDA   #0                        ; IF OVERLAP_Y<0 OVERLAP_Y=0
.L_offset_Y_nonzero
  STA   zp_overlap_Y
  EOR   #&ff
  SEC
  ADC   #5
  STA   zp_main_Y
  
  LDA   zp_Y
  LSR   A
  LSR   A                         ; /4, not /8, because we must ix into T_X320 which is 16-bit
  AND   #&FE
  TAY
  
  CLC
  ; theoretical max low byte &C0:
  LDA   T_X320,Y                  ; (zp_Y / 8) * 320, low byte, start of upper stripe
  ; theoretical max &7 (Y-shift within block)
  ADC   zp_sub_Y                  ; carry cannot be set yet (&C0 + 7), so keep going!
  ; theoretical max &F8:
  ADC   zp_X_voff_block_in_stripe ; add X-offset
  STA   zp_leftblock_topline_L    ; store low byte
  LDA   T_X320+1,Y                ; (zp_Y / 8) * 320, high byte
  ADC   #VRAM_H                   ; add VRAM start (+ carry)
  STA   zp_leftblock_topline_H    ; store high byte
  
  ; FIXME: is it faster to precompute this or do it in the loop?
  ; Need to add 8 to the VRAM address in order to get across from the
  ; left block to the right block.
  LDA   zp_leftblock_topline_L
  CLC
  ADC   #8
  STA   zp_rightblock_topline_L
  LDA   zp_leftblock_topline_H
  ADC   #0                        ; yes, sadly this is necessary
  STA   zp_rightblock_topline_H
  
  CLC
  LDA   T_X320+2,Y                ; ((zp_Y / 8) + 1) * 320, low byte, start of lower stripe
  ADC   zp_X_voff_block_in_stripe ; add X-offset
  STA   zp_leftblock_lower_L      ; store low byte
  LDA   T_X320+3,Y                ; ((zp_Y / 8) + 1) * 320, high byte
  ADC   #VRAM_H                   ; add VRAM start (+ carry)
  STA   zp_leftblock_lower_H      ; store high byte
  
  LDA   zp_leftblock_lower_L
  CLC
  ADC   #8
  STA   zp_rightblock_lower_L
  LDA   zp_leftblock_lower_H
  ADC   #0
  STA   zp_rightblock_lower_H
  
  ; handle X-shift
  ; another 256-byte lookup table saves 4 cycles
  ; over doing ASL:ASL:ASL:AND #&38
  LDY   zp_X
  LDA   T_X8_BITS_345,Y
  STA   zp_t_star_left_L
  STA   zp_t_star_right_L

  LDY   #0
  LDX   #0
.P_star_draw_top_pair
  LDA   (zp_t_star_left_L,X)
  EOR   (zp_leftblock_topline_L),Y
  STA   (zp_leftblock_topline_L),Y
  LDA   (zp_t_star_right_L,X)
  EOR   (zp_rightblock_topline_L),Y
  STA   (zp_rightblock_topline_L),Y
  INC   zp_t_star_left_L
  INC   zp_t_star_right_L
  INY
  M_test_timer_stripes
  DEC   zp_main_Y
  BNE   P_star_draw_top_pair
  
  LDA   zp_overlap_Y
  BNE   L_star_draw_bottom_pair   ; skip bottom pair
  
IF TEST3
  LDA   #0
  M_set_bgcolour
ENDIF

IF _FAST_RTS_EXIT
  RTS                             ; fast RTS exit
ELSE
  CLC
  BCC L_star_exit                 ; or slow, linear fall-through exit
ENDIF
  
.L_star_draw_bottom_pair
  LDY   #0
.P_star_draw_bottom_pair
  LDA   (zp_t_star_left_L,X)
  EOR   (zp_leftblock_lower_L),Y
  STA   (zp_leftblock_lower_L),Y
  LDA   (zp_t_star_right_L,X)
  EOR   (zp_rightblock_lower_L),Y
  STA   (zp_rightblock_lower_L),Y
  INC   zp_t_star_left_L
  INC   zp_t_star_right_L
  INY
  M_test_timer_stripes
  DEC   zp_overlap_Y
  BNE   P_star_draw_bottom_pair
  
IF TEST3
  LDA   #2
  M_set_bgcolour
ENDIF
  
IF _FAST_RTS_EXIT
  RTS
ELSE
.L_star_exit
ENDIF

ENDMACRO }


MACRO M_text {

  LDX   #0
.P_text_need_new_char
  STX   zp_cur_char_row
  LDA   (zp_ptr_string_L,X) ; fetch next character
  BEQ   L_string_done       ; null terminator; if encountered just quit
  INC   zp_ptr_string_L     ; next char -- careful -- no page wrapping!
  
  ; establish appropriate pointer to MOS ROM
  TAY
  LDA   T_CHAR_TO_FONT_PTR_LOW,Y
  STA   zp_ptr_mos_font_L
  LDA   T_CHAR_TO_FONT_PTR_HIGH,Y
  STA   zp_ptr_mos_font_H
  
  LDY   zp_cur_char_row
.P_text_skip_new_char
  LDA   (zp_ptr_mos_font_L),Y
  STA   (zp_ptr_text_vram_L,X)  ; X=0
  INC   zp_ptr_text_vram_L      ; warning: won't wrap
  INY
  CPY   #8
  BNE   P_text_skip_new_char
  BEQ   P_text_need_new_char

.L_string_done

} ENDMACRO

MACRO M_change_text _T {
  ; reset "cursor"
  LDA   #LO(VRAM_TEXT_TARGET)
  STA   zp_ptr_text_vram_L
  ; change string
  LDA   #LO(_T)
  STA   zp_ptr_string_L
  LDA   #HI(_T)
  STA   zp_ptr_string_H
} ENDMACRO


IF (TEST OR TEST3) AND BEAM_AVOIDANCE
MACRO M_set_bgcolour {
  EOR   #7
  STA   videoULAPaletteRegister ; 000
  ORA   #&10
  STA   videoULAPaletteRegister ; 001
  ORA   #&20
  STA   videoULAPaletteRegister ; 011
  ORA   #&40
  STA   videoULAPaletteRegister ; 111
  EOR   #&10
  STA   videoULAPaletteRegister ; 110
  EOR   #&20
  STA   videoULAPaletteRegister ; 100
  ORA   #&10
  STA   videoULAPaletteRegister ; 101
  EOR   #&70
  STA   videoULAPaletteRegister ; 010
  ;RTS
} ENDMACRO
ENDIF ; TEST AND BEAM_AVOIDANCE


org &4800
guard &5800
; ----------


; =========*======*======*=========
; ********** ENTRY POINT **********
; =========*======*======*=========

.start

IF TEST2=0 AND TEST3=0
  ; MODE 4
  LDA   #22
  JSR   OSWRCH
  LDA   #4
  JSR   OSWRCH
ENDIF

  
  LDA   #HI(L_script_jumptable)
  STA   zp_script_jumptable_H
  LDA   #0                        ; start on scene 0
  STA   zp_script_jumptable_L
  
  LDA   T_SCENE_DURATIONS         ; set up first scene duration
  STA   zp_scene_duration
  
  ; cursor off
  ;   VDU 23;8202;0;0;0;
  LDA   #23
  JSR   OSWRCH
  LDA   #0
  JSR   OSWRCH
  LDA   #10
  JSR   OSWRCH
  LDA   #32
  JSR   OSWRCH
  LDA   #6
  STA   zp_ptr_mos_font_H         ; just borrow this
.L_disable_cursor
  JSR   OSWRCH
  DEC   zp_ptr_mos_font_H
  BNE   L_disable_cursor
  
  LDA   #HI(T_MOS_FONT)
  STA   zp_ptr_mos_font_H
  LDA   #LO(T_MOS_FONT)
  STA   zp_ptr_mos_font_L
  
  ; zp_ptr_text_vram_H will never change, so just set it up once:
  LDA   #HI(VRAM_TEXT_TARGET)
  STA   zp_ptr_text_vram_H

  LDA   #HI(T_STAR_LEFT)
  STA   zp_t_star_left_H
  LDA   #HI(T_STAR_RIGHT)
  STA   zp_t_star_right_H
  
IF DISABLE_INTERLACING
  SEI
  ; DISABLE interlacing
  LDA #8
  STA &FE00
  LDA #0
  STA &FE01
  CLI
ENDIF

  LDA   #1
  STA   zp_delay_count

  LDA   #0
  STA   zp_script_scene         ; initialise script state machine
  
  LDA   #LO(T_SAVED_COORDINATES_X)
  STA   zp_ptr_saved_coords_x_L
  LDA   #HI(T_SAVED_COORDINATES_X)
  STA   zp_ptr_saved_coords_x_H
  LDA   #LO(T_SAVED_COORDINATES_Y)
  STA   zp_ptr_saved_coords_y_L
  LDA   #HI(T_SAVED_COORDINATES_Y)
  STA   zp_ptr_saved_coords_y_H
  
IF BEAM_AVOIDANCE
  LDA   #&13
  LDA   #&40                    ; parkour mode ...
  STA   USERVIA_ACR             ; ACR

  LDA   #&40                    ; no timer interrupts, thanks
  STA   USERVIA_IER             ; IER
  LDA   #(timer_latch - 2) AND 255
  STA   USERVIA_T1VAL_L
  ; GO!
  LDA   #VSYNC_OSBYTE
  JSR   OSBYTE
  LDA   #(timer_latch - 2) DIV 256
  STA   USERVIA_T1VAL_H
ENDIF ; BEAM_AVOIDANCE

; ---------
; MAIN LOOP
; ---------

.P_main_new
IF BEAM_AVOIDANCE
  LDA   #0
  STA   zp_deferred_stars_count
ENDIF
  JSR   S_script_new
  JMP   P_main_new
  
; ----------

; expect colour in A
.S_set_colour
  AND   #7
  BNE   L_set_colour_not_black
  ORA   #1
.L_set_colour_not_black
  EOR   #7
  ORA   #&80
  STA   videoULAPaletteRegister ; 000
  ORA   #&10
  STA   videoULAPaletteRegister ; 001
  ORA   #&20
  STA   videoULAPaletteRegister ; 011
  ORA   #&40
  STA   videoULAPaletteRegister ; 111
  EOR   #&10
  STA   videoULAPaletteRegister ; 110
  EOR   #&20
  STA   videoULAPaletteRegister ; 100
  ORA   #&10
  STA   videoULAPaletteRegister ; 101
  EOR   #&70
  STA   videoULAPaletteRegister ; 010
  RTS



; -----------

.S_script_new
IF TEST3
  LDA #0
  STA zp_X
  STA zp_Y
  LDA VSYNC_OSBYTE
  JSR OSBYTE
  M_star 0                        ; 0 = do not RTS on exit
  RTS
ELSE
  DEC   zp_scene_duration
  BEQ   L_script_new_scene        ; timer timed out?
  JMP   (zp_script_jumptable_L)   ; no? jump to current selected scene
.L_script_new_scene
  INC   zp_script_scene
  LDY   zp_script_scene
  LDA   T_SCENE_DURATIONS,Y
  STA   zp_scene_duration         ; set up new scene duration
  TYA
  ASL   A
  ASL   A                         ; (zp_script_scene * 4), set up jump target
  STA   zp_script_jumptable_L     ; page-aligned so nothing further to add
  JMP   (zp_script_jumptable_L)
  ; (no RTS)
ENDIF
  
; -----------
org &4900
.T_SCENE_DURATIONS
  EQUB  &2    ; 01 text=HELLO
  EQUB  &2F   ; 02 draw+hold text
  EQUB  &1    ; 03 text=LOGIKER
  EQUB  &1F   ; 04 draw+hold text
  EQUB  &1    ; 05 draw SHAPE_1
  EQUB  &3F   ; 06 hold SHAPE_1
  EQUB  &1    ; 07 text=WONDERING
  EQUB  &2F   ; 08 draw+hold text
  EQUB  &1    ; 09 text=TAKES
  EQUB  &2F   ; 10 draw+hold text
  EQUB  &1    ; 11 text=ONE_SIMPLE
  EQUB  &2F   ; 12 draw+hold text
  EQUB  &1    ; 13 text=ROTATE
  EQUB  &2F   ; 14 draw+hold text, set up for rotation
  EQUB  &3f   ; 15 spin SHAPE_1, text=BLANK
  EQUB  &7f   ; 16 spin SHAPE_1, draw+hold BLANK text
  EQUB  &1    ; 17 spin SHAPE_1, text=MIRACLE
  EQUB  &2F   ; 18 spin SHAPE_1, draw+hold MIRACLE
  EQUB  &1    ; 19 spin SHAPE_1, text=SPENT_8K
  EQUB  &2F   ; 20 spin SHAPE_1, draw+hold SPENT_8K
  EQUB  &1    ; 21 spin SHAPE_1, text=ENOUGH_RAM_LEFT
  EQUB  &2F   ; 22 spin SHAPE_1, draw+hold ENOUGH_RAM_LEFT
  EQUB  &1    ; 23 spin SHAPE_1, text=SOME_MORE_SHAPES
  EQUB  &2F   ; 24 spin SHAPE_1, draw+hold SOME_MORE_SHAPES
  EQUB  &1    ; 25 spin SHAPE_1, text=NICE_STAR
  EQUB  &2F   ; 26 spin SHAPE_1, draw+hold NICE_STAR
  EQUB  &1    ; 27 spin SHAPE_1, text=STARDOT
  EQUB  &2F   ; 28 spin SHAPE_1, draw+hold STARDOT
  EQUB  &1    ; 29 undraw SHAPE_1
  EQUB  &1    ; 30 draw SHAPE_2
  EQUB  &ff   ; 31 spin SHAPE_2
  EQUB  &1    ; 32 undraw SHAPE_2, text=SNOWFLAKE
  EQUB  &2f   ; 33 draw+hold SNOWFLAKE
  EQUB  &1    ; 34 text=GOING_TO_RAIN
  EQUB  &2f   ; 35 draw+hold GOING_TO_RAIN
  EQUB  &1    ; 36 draw SHAPE_3, text=CHRISTMAS_DAY
  EQUB  &2f   ; 37 spin SHAPE_3, draw+hold CHRISTMAS_DAY
  EQUB  &1    ; 38 spin SHAPE_3, text=MERRY
  EQUB  &7f   ; 39 spin SHAPE_3, draw+hold MERRY
  
  EQUB  1     ; 40 erase SHAPE_3, draw SHAPE_1
  EQUB  &7f   ; 41 hodl
  EQUB  1     ; 42 erase SHAPE_1, draw SHAPE_2
  EQUB  &7f   ; 43 hodl
  EQUB  1     ; 44 erase SHAPE_2, draw SHAPE_3
  EQUB  &7f   ; 45 hodl
  EQUB  1     ; 46 revert to script ID 40
  
  
; ----------

  
; must be page-aligned
org &4A00
.L_script_jumptable
  JMP   S_script_scene_01 : NOP    ; NOP so that entries are 4 bytes apart
  JMP   S_script_scene_02 : NOP
  JMP   S_script_scene_03 : NOP
  JMP   S_script_scene_04 : NOP
  JMP   S_script_scene_05 : NOP
  JMP   S_script_scene_06 : NOP
  JMP   S_script_scene_07 : NOP
  JMP   S_script_scene_08 : NOP
  JMP   S_script_scene_09 : NOP
  JMP   S_script_scene_10 : NOP
  JMP   S_script_scene_11 : NOP
  JMP   S_script_scene_12 : NOP
  JMP   S_script_scene_13 : NOP
  JMP   S_script_scene_14 : NOP
  JMP   S_script_scene_15 : NOP
  JMP   S_script_scene_16 : NOP
  JMP   S_script_scene_17 : NOP
  JMP   S_script_scene_18 : NOP
  JMP   S_script_scene_19 : NOP
  JMP   S_script_scene_20 : NOP
  JMP   S_script_scene_21 : NOP
  JMP   S_script_scene_22 : NOP
  JMP   S_script_scene_23 : NOP
  JMP   S_script_scene_24 : NOP
  JMP   S_script_scene_25 : NOP
  JMP   S_script_scene_26 : NOP
  JMP   S_script_scene_27 : NOP
  JMP   S_script_scene_28 : NOP
  JMP   S_script_scene_29 : NOP
  JMP   S_script_scene_30 : NOP
  JMP   S_script_scene_31 : NOP
  JMP   S_script_scene_32 : NOP
  JMP   S_script_scene_33 : NOP
  JMP   S_script_scene_34 : NOP
  JMP   S_script_scene_35 : NOP
  JMP   S_script_scene_36 : NOP
  JMP   S_script_scene_37 : NOP
  JMP   S_script_scene_38 : NOP
  JMP   S_script_scene_39 : NOP
  JMP   S_script_scene_40 : NOP
  JMP   S_script_scene_41 : NOP
  JMP   S_script_scene_42 : NOP
  JMP   S_script_scene_43 : NOP
  JMP   S_script_scene_44 : NOP
  JMP   S_script_scene_45 : NOP
  JMP   S_script_scene_46 : NOP
  RTS
  
.S_script_scene_01
  M_change_text T_STRING_HELLO
  RTS
.S_script_scene_02
  M_text
  JSR   S_delay
  RTS
.S_script_scene_03
  M_change_text T_STRING_LOGIKER
  RTS
.S_script_scene_04
  M_text
  JSR   S_delay
  RTS
.S_script_scene_05
  LDA   #&0
  STA   zp_rot_angle
  M_set_shape T_SHAPE_1, SHAPE_1_NUM_POINTS
  DEC   zp_shape_no_erase_oneshot
  JSR   S_shape                   ; draw once, no erase
  RTS
.S_script_scene_06
  JSR   S_delay
  RTS
.S_script_scene_07
  M_change_text T_STRING_WONDERING
  RTS
.S_script_scene_08
  M_text
  JSR   S_delay
  RTS
.S_script_scene_09
  M_change_text T_STRING_TAKES
  RTS
.S_script_scene_10
  M_text
  JSR   S_delay
  RTS
.S_script_scene_11
  M_change_text T_STRING_ONE_SIMPLE
  RTS
.S_script_scene_12
  M_text
  JSR   S_delay
  RTS
.S_script_scene_13
  M_change_text T_STRING_ROTATE
  RTS
.S_script_scene_14
  LDA   #0
  STA   zp_shape_no_erase_oneshot ; put this back
  LDA   #&1
  STA   zp_rot_angle              ; put this back too
  M_text
  JSR   S_delay
  RTS
.S_script_scene_15
  JSR   S_shape
  INC   zp_rot_angle
  M_change_text T_STRING_BLANK
  RTS
.S_script_scene_16
  JSR   S_shape
  INC   zp_rot_angle
  M_text
  RTS
.S_script_scene_17
  JSR   S_shape
  INC   zp_rot_angle
  M_change_text T_STRING_MIRACLE
  RTS
.S_script_scene_18
  JSR   S_shape
  INC   zp_rot_angle
  M_text
  RTS
.S_script_scene_19
  JSR   S_shape
  INC   zp_rot_angle
  M_change_text T_STRING_SPENT_8K
  RTS
.S_script_scene_20
  JSR   S_shape
  INC   zp_rot_angle
  M_text
  RTS
.S_script_scene_21
  JSR   S_shape
  INC   zp_rot_angle
  M_change_text T_STRING_ENOUGH_RAM_LEFT
  RTS
.S_script_scene_22
  JSR   S_shape
  INC   zp_rot_angle
  M_text
  RTS
.S_script_scene_23
  JSR   S_shape
  INC   zp_rot_angle
  M_change_text T_STRING_SOME_MORE_SHAPES
  RTS
.S_script_scene_24
  JSR   S_shape
  INC   zp_rot_angle
  M_text
  RTS
.S_script_scene_25
  JSR   S_shape
  INC   zp_rot_angle
  M_change_text T_STRING_NICE_STAR
  RTS
.S_script_scene_26
  JSR   S_shape
  INC   zp_rot_angle
  M_text
  RTS
.S_script_scene_27
  JSR   S_shape
  INC   zp_rot_angle
  M_change_text T_STRING_STARDOT
  RTS
.S_script_scene_28
  JSR   S_shape
  INC   zp_rot_angle
  M_text
  RTS
.S_script_scene_29
  DEC   zp_rot_angle
  DEC   zp_shape_no_erase_oneshot ; go one-shot
  JSR   S_shape                   ; erase SHAPE_1
  M_set_shape T_SHAPE_2, SHAPE_2_NUM_POINTS
  RTS
.S_script_scene_30
  JSR   S_shape                   ; one-shot draw SHAPE_2
  INC   zp_rot_angle
  INC   zp_shape_no_erase_oneshot ; cancel one-shot
  RTS
.S_script_scene_31
  JSR   S_shape
  INC   zp_rot_angle
  RTS
.S_script_scene_32
  M_change_text T_STRING_SNOWFLAKE
  ;DEC   zp_rot_angle
  ;DEC   zp_shape_no_erase_oneshot ; go one-shot
  ;JSR   S_shape                   ; erase SHAPE_2
  ;M_set_shape T_SHAPE_3, SHAPE_3_NUM_POINTS
  JSR   S_shape
  INC   zp_rot_angle
  RTS
.S_script_scene_33
  M_text
  JSR   S_shape
  INC   zp_rot_angle
  ;JSR   S_delay                   ; back to manual delay again
  RTS
.S_script_scene_34
  JSR   S_shape
  INC   zp_rot_angle
  M_change_text T_STRING_GOING_TO_RAIN
  RTS
.S_script_scene_35
  M_text
  JSR   S_shape
  INC   zp_rot_angle
  ;JSR   S_delay                   ; manual delay
  RTS
.S_script_scene_36
  DEC   zp_rot_angle
  DEC   zp_shape_no_erase_oneshot ; go one-shot
  JSR   S_shape                   ; erase SHAPE_2
  M_set_shape T_SHAPE_3, SHAPE_3_NUM_POINTS
  JSR   S_shape                   ; draw SHAPE_3
  INC   zp_rot_angle
  INC   zp_shape_no_erase_oneshot ; cancel one-shot
  M_change_text T_STRING_CHRISTMAS_DAY
  RTS
.S_script_scene_37
  M_text
  JSR   S_shape
  INC   zp_rot_angle
  RTS
.S_script_scene_38
  M_change_text T_STRING_MERRY
  JSR   S_shape
  INC   zp_rot_angle
  RTS
.S_script_scene_39
  M_text
  JSR   S_shape
  INC   zp_rot_angle
  RTS
; infinite loop begins here
.S_script_scene_40
  INC   zp_colour
  LDA   zp_colour
  JSR   S_set_colour
  DEC   zp_rot_angle
  DEC   zp_shape_no_erase_oneshot ; go one-shot
  JSR   S_shape ; erase SHAPE_3
  M_set_shape T_SHAPE_1, SHAPE_1_NUM_POINTS
  RTS
.S_script_scene_41
  JSR   S_shape
  LDA   #0
  STA   zp_shape_no_erase_oneshot
  INC   zp_rot_angle
  RTS
.S_script_scene_42
  DEC   zp_rot_angle
  DEC   zp_shape_no_erase_oneshot ; go one-shot
  JSR   S_shape ; erase SHAPE_1
  M_set_shape T_SHAPE_2, SHAPE_2_NUM_POINTS
  RTS
.S_script_scene_43
  JSR   S_shape
  LDA   #0
  STA   zp_shape_no_erase_oneshot
  INC   zp_rot_angle
  RTS
.S_script_scene_44
  DEC   zp_rot_angle
  DEC   zp_shape_no_erase_oneshot ; go one-shot
  JSR   S_shape ; erase SHAPE_2
  M_set_shape T_SHAPE_3, SHAPE_3_NUM_POINTS
  RTS
.S_script_scene_45
  JSR   S_shape
  LDA   #0
  STA   zp_shape_no_erase_oneshot
  INC   zp_rot_angle
  RTS
.S_script_scene_46
  JSR   S_shape
  INC   zp_rot_angle
  LDA   #38                       ; what? why 38? don't know
  STA   zp_script_scene
  RTS
.LA : JMP LA


; --------------

; stores in zp_X, zp_Y
; takes rotation angle in accumulator
.S_shape_recompute_coords

  LDY   zp_v_shape_ix_doubled

  ; shape points are stored (RID, angle); so we pull the angle first
  ; -1 here because of shenanigans;
  ; zp_ptr_shape_minus_1_* actually points to one BEFORE the start
  ; of the shape; zp_v_shape_ix_doubled's final valid value is 2;
  ; when it's decremented to 0, the loop is terminated
  ; (so Y-reg contains <real index> + 2)
  CLC
  ADC   (zp_ptr_shape_minus_1_L),Y
  AND   #&7F                      ; v6: angles are modulo 128 now ...
  TAX                             ; stash this, for use in a minute
  
  ; based on post-rotated angle, find "row" within 8K lookup table
  ; (each "row" is 32 bytes long)
  ; (*32 + &2000) lookup tables cost 256 bytes of RAM each and save just 4 instructions >_>
  
  TAY                             ; index into LUT
  LDA   T_ANGLE_TO_HIGH_T_TRIG_PTR,Y ; 128 byte LUT
  STA   zp_ptr_trig_H
  
  TXA
  TAY
  LDA   T_ANGLE_TO_LOW_T_TRIG_PTR,Y  ; 128 byte LUT
  STA   zp_ptr_trig_L
  LDY   zp_v_shape_ix_doubled     ; have to retrieve this again
  
  LDA   (zp_ptr_shape_minus_2_L),Y ; get radius ID from T_SHAPE (-2 because more Y shenanigans)
                                  ; NOTE that this comes out of the table pre-doubled, for indexing into zp_ptr_trig_*
  TAX
  TAY
  
  LDA   (zp_ptr_trig_L),Y         ; ZP gives row in trig table, Y gives entry in row
  STA   zp_X                      ; X-coord
;IF BEAM_AVOIDANCE
  TAX
;ENDIF
  
  ; Y-coord last
  ;TXA
  ;TAY
  INY
  LDA   (zp_ptr_trig_L),Y         ; ZP gives row in trig table, Y gives entry in row
  STA   zp_Y                      ; final Y-coord
  
  ; preserve coordinates for future generations
  LDY   zp_v_shape_ix
  STA   (zp_ptr_saved_coords_y_L),Y   ; save computed Y-coordinate
  TXA
  STA   (zp_ptr_saved_coords_x_L),Y   ; save computed X-coordinate
  
  RTS
  
; -----------


.S_shape_point

  M_star 1                        ; 1 = auto-RTS on macro exit

.end

SAVE "DEMO8", top, end, start

