// Import external functions here:

// Moonlander is a library for integrating Processing with Rocket
// Rocket is a tool for synchronizing music and visuals in demoscene projects.
import moonlander.library.*;
// Minim is needed for the music playback.
import ddf.minim.*;

// Global variables

// These control how big the opened window is.
final int CANVAS_WIDTH = 1920; // We want 1920 pixels wide... 
final int CANVAS_HEIGHT = 1080; // and 1080 pixels tall canvas to get FullHD aka 1080p.

final int FPS = 60; // We want  to run in 60 FPS for broadcast reasons

float ASPECT_RATIO = (float)CANVAS_WIDTH/CANVAS_HEIGHT;  // Aspect ratio of the canvas

float time = 0.0; // Time in seconds from Moonlander / GNURocket

Moonlander moonlander; // Moonlander instance

// Needed for audio
Minim minim;
AudioPlayer song;

PFont font;


// Array for storing texts we want to display
String[] texts = {
  "assembly winter 2021",
  "one scene demo compo",
  "gfx: hajame",
  "music: \"Alarm - 105 BPM\" by hajame",
  "let's..",
  "go!",
  "",
  "well, that's it then.",
  "hope the alarm sound didn't...",
  "trigger any PTSD symptoms",
  "thank you:",
  "eimink",
  "for the One Scene workshop repo",
  "codeman38 for Press Start 2P Font",
  "...and eh...",
  "everyone organizing assembly!",
  "bye!"
};

// Now lets setup audio playing
void setupAudio() {
  minim = new Minim(this);

  // This is where we load the song and play it.
  // song = minim.loadFile("beat.mp3");
  // song.play();
}


void settings() {
  boolean fullscreen = true;
  String renderer = P3D; // Renderer to use either P2D,P3D or JAVA2D.
  if (fullscreen) {
    // Set the window fullscreen in desktop's native resolution
    fullScreen(renderer);
  }
  else {
    // Open as window
    size(CANVAS_WIDTH, CANVAS_HEIGHT, renderer);
  }
  
}

void setup() {
  // Load shapes
  // shape1 = loadShape("Love.obj");
  
  // Load the font
  // NOTE: You can use system available fonts like "MS Comic Sans" but that usually makes your production less cross-platform
  //       hence we'd recommend to save the font as an asset to the production.
  //       Just remember that font creators have copyrights, so the license needs to be appropriate to use in a demo.
  
  font = createFont("PressStart2P-vaV7.ttf", 130);
  
  //hide mouse cursor
  noCursor();

  // Init & start moonlander
  int bpm = 105; // Tune's beats per minute
  int rowsPerBeat = 4; // How many rows one beat consists of in the sync editor (Rocket or so)
  moonlander = Moonlander.initWithSoundtrack(this, "hajame-alarm-105_BPM.mp3", bpm, rowsPerBeat);
  moonlander.start();

  frameRate(FPS); // Set the frame rate
  
  // Load assets used in demo
}


void drawCube() {
  pushMatrix();
  stroke(255);
  strokeWeight(4); 
  translate((float)moonlander.getValue("cube:x"), (float)moonlander.getValue("cube:y"), (float)moonlander.getValue("cube:z"));
  
  rotateX(radians((float)moonlander.getValue("cube:rotateX")));
  rotateY(radians((float)moonlander.getValue("cube:rotateY")));
  rotateZ(radians((float)moonlander.getValue("cube:rotateZ")));
  
  fill((float)moonlander.getValue("cube:r")*255,(float)moonlander.getValue("cube:g")*255,(float)moonlander.getValue("cube:b")*255,(int)(moonlander.getValue("cube:line_alpha") * 255));
  box((float)moonlander.getValue("cube:width"), (float)moonlander.getValue("cube:height"), (float)moonlander.getValue("cube:depth")); 
  popMatrix();
}

void drawLines() {
  pushMatrix();
  
  strokeWeight((float)moonlander.getValue("line:width")); 
  translate((float)moonlander.getValue("line:x"), (float)moonlander.getValue("line:y"), (float)moonlander.getValue("line:z"));
  
  stroke((float)moonlander.getValue("line:r")*255,(float)moonlander.getValue("line:g")*255,(float)moonlander.getValue("line:b")*255,(int)(moonlander.getValue("line:alpha") * 255));
  float x1 = (float)moonlander.getValue("line:x1");
  float y1 = (float)moonlander.getValue("line:y1");
  float x2 = - x1;
  float y2 = - y1;
  line(x1, y1, -500, x2, y2, -500);
  popMatrix();
}

void drawText() {
  if (moonlander.getValue("font:text") >= 0) {
    pushMatrix();
    scale((float)moonlander.getValue("font:scale"));
    textAlign(CENTER, CENTER);
    textFont(font);
    fill((int)(moonlander.getValue("font:r") * 255),(int)(moonlander.getValue("font:g") * 255),(int)(moonlander.getValue("font:b") * 255),(int)(moonlander.getValue("font:a") * 255));
    text(texts[(int)moonlander.getValue("font:text")%texts.length], (int)moonlander.getValue("font:x"), (int)moonlander.getValue("font:y"));
    popMatrix();
  }
}

/*
 * This function is called every time a screen is drawn, ideally that would be 60 times per second
 */
void draw() {
  // update Rocket sync data  
  moonlander.update();

  time = (float)moonlander.getCurrentTime();
  float end = 60.0; //end production after 60 secs which is the maximum time allowed by the One Scene Compo
  if (time > end) {
    exit();
  }
  
  // Set the background color
  background((int)(moonlander.getValue("bg:r") * 255),(int)(moonlander.getValue("bg:g") * 255),(int)(moonlander.getValue("bg:b") * 255),(int)(moonlander.getValue("bg:a") * 255));
  
  /*
   * Center coordinates to screen and make the window and canvas resolution independent
   * This is because actual window in full screen on a 4K monitor has more pixels than FullHD resolution
   * so scaling is needed to ensure that all objects (3D and 2D) are in correct places regardless of the desktop resolution
   */
  translate(width/2, height/2, 0);
  scale(width/CANVAS_WIDTH,height/CANVAS_HEIGHT,width/CANVAS_WIDTH);

  // Enable lights and depth testing to ensure that 3D meshes are drawn in correct order
  lights();
  hint(ENABLE_DEPTH_TEST);

  drawCube();
  
  drawLines();

  // disable lights and depth testing so that 2D overlays and text can be draw on top of 3D
  noLights();
  hint(DISABLE_DEPTH_TEST);

  drawText();

}

// Handle keypresses
void keyPressed() {
  if (key == CODED) {
    // if ESC was pressed we exit from demo. 
    // This is a requirement in Assembly compo rules for desktop platforms.
    if (keyCode == ESC) {
      exit();
    }
  }
}
