using System;
using System.Collections.Generic;
using System.Linq;
using SharpDX;
using SharpDX.Direct3D;
using SharpDX.Direct3D11;
using Framefield.Core.Rendering;
using Buffer = SharpDX.Direct3D11.Buffer;

namespace Framefield.Core.ID1b6d97b4_af5f_455b_a1a4_406f1611f78f
{
    public class Class_ReplicateToGrid : FXSourceCodeFunction
    {
        #region Renderer
        public class Renderer : BaseRenderer
        {
            public override void SetupEffect(OperatorPartContext context)
            {
                base.SetupEffect(context);
                try
                {
                    SetupMaterialConstBuffer(context);
                    SetupFogSettingsConstBuffer(context);
                    SetupPointLightsConstBuffer(context);
                }
                catch (Exception e)
                {
                    Logger.Error(ParentFunc, "Error building constant buffer: {0} - Source: {1}", e.Message, e.Source);
                }
            }
            public OperatorPart.Function ParentFunc {get;set;}            
        }
        #endregion

        public Class_ReplicateToGrid()
        {
            _renderer = new Renderer(){ParentFunc = this};
        }

        public override void Dispose()
        {
            Utilities.DisposeObj(ref _renderer);
            base.Dispose();
        }

        //>>> _inputids
        private enum InputId
        {
            Code = 0,
            CountX = 1,
            CountY = 2,
            CountZ = 3,
            SpreadX = 4,
            SpreadY = 5,
            ScatterX = 6,
            SpreadZ = 7,
            ScatterY = 8,
            ScatterZ = 9,
            ScaleRandom = 10,
            Orientation = 11,
            SpinX = 12,
            SpinRandomX = 13,
            Seed = 14,
            MeshInput = 15,
            Scale = 16,
            Pattern = 17,
            CenterX = 18,
            CenterY = 19,
            CenterZ = 20,
            SpinEffectX = 21,
            ScaleEffect = 22,
            EffectCenterX = 23,
            EffectCenterY = 24,
            EffectCenterZ = 25,
            EffectRangeRadius = 26,
            EffectRangeFallOff = 27,
            SpinY = 28,
            SpinZ = 29,
            SpinRandomY = 30,
            SpinRandomZ = 31,
            SpinEffectY = 32,
            SpinEffectZ = 33,
            EffectRangeScatter = 34
        }
        //<<< _inputids


        private bool _firstEval = true;
        public override OperatorPartContext Eval(OperatorPartContext context, List<OperatorPart> inputs, int outputIdx)
        {
            //>>> _params
            var Code = inputs[(int)InputId.Code].Eval(context).Text;
            var CountX = inputs[(int)InputId.CountX].Eval(context).Value;
            var CountY = inputs[(int)InputId.CountY].Eval(context).Value;
            var CountZ = inputs[(int)InputId.CountZ].Eval(context).Value;
            var Count = new Vector3(CountX, CountY, CountZ);
            var SpreadX = inputs[(int)InputId.SpreadX].Eval(context).Value;
            var SpreadY = inputs[(int)InputId.SpreadY].Eval(context).Value;
            var SpreadZ = inputs[(int)InputId.SpreadZ].Eval(context).Value;
            var Spread = new Vector3(SpreadX, SpreadY, SpreadZ);
            var ScatterX = inputs[(int)InputId.ScatterX].Eval(context).Value;
            var ScatterY = inputs[(int)InputId.ScatterY].Eval(context).Value;
            var ScatterZ = inputs[(int)InputId.ScatterZ].Eval(context).Value;
            var Scatter = new Vector3(ScatterX, ScatterY, ScatterZ);
            var ScaleRandom = inputs[(int)InputId.ScaleRandom].Eval(context).Value;
            var Orientation = (int) inputs[(int)InputId.Orientation].Eval(context).Value;
            var SpinX = inputs[(int)InputId.SpinX].Eval(context).Value;
            var SpinY = inputs[(int)InputId.SpinY].Eval(context).Value;
            var SpinZ = inputs[(int)InputId.SpinZ].Eval(context).Value;
            var Spin = new Vector3(SpinX, SpinY, SpinZ);
            var SpinRandomX = inputs[(int)InputId.SpinRandomX].Eval(context).Value;
            var SpinRandomY = inputs[(int)InputId.SpinRandomY].Eval(context).Value;
            var SpinRandomZ = inputs[(int)InputId.SpinRandomZ].Eval(context).Value;
            var SpinRandom = new Vector3(SpinRandomX, SpinRandomY, SpinRandomZ);
            var Seed = inputs[(int)InputId.Seed].Eval(context).Value;
            var MeshInput = inputs[(int)InputId.MeshInput].Eval(context).Mesh;
            var Scale = inputs[(int)InputId.Scale].Eval(context).Value;
            var Pattern = (int) inputs[(int)InputId.Pattern].Eval(context).Value;
            var CenterX = inputs[(int)InputId.CenterX].Eval(context).Value;
            var CenterY = inputs[(int)InputId.CenterY].Eval(context).Value;
            var CenterZ = inputs[(int)InputId.CenterZ].Eval(context).Value;
            var Center = new Vector3(CenterX, CenterY, CenterZ);
            var SpinEffectX = inputs[(int)InputId.SpinEffectX].Eval(context).Value;
            var SpinEffectY = inputs[(int)InputId.SpinEffectY].Eval(context).Value;
            var SpinEffectZ = inputs[(int)InputId.SpinEffectZ].Eval(context).Value;
            var SpinEffect = new Vector3(SpinEffectX, SpinEffectY, SpinEffectZ);
            var ScaleEffect = inputs[(int)InputId.ScaleEffect].Eval(context).Value;
            var EffectCenterX = inputs[(int)InputId.EffectCenterX].Eval(context).Value;
            var EffectCenterY = inputs[(int)InputId.EffectCenterY].Eval(context).Value;
            var EffectCenterZ = inputs[(int)InputId.EffectCenterZ].Eval(context).Value;
            var EffectCenter = new Vector3(EffectCenterX, EffectCenterY, EffectCenterZ);
            var EffectRangeRadius = inputs[(int)InputId.EffectRangeRadius].Eval(context).Value;
            var EffectRangeFallOff = inputs[(int)InputId.EffectRangeFallOff].Eval(context).Value;
            var EffectRangeScatter = inputs[(int)InputId.EffectRangeScatter].Eval(context).Value;
            var EffectRange = new Vector3(EffectRangeRadius, EffectRangeFallOff, EffectRangeScatter);
            //<<< _params        


            var intCountX = (int)(Utilities.Clamp(CountX, 0, 1000));
            var intCountY = (int)(Utilities.Clamp(CountY, 0, 1000));
            var intCountZ = (int)(Utilities.Clamp(CountZ, 0, 1000));

            var numInstances = intCountX * intCountY * intCountZ;

            if (MeshInput == null || numInstances == 0 )
            {
                //_bakedMesh.NumTriangles = 0;            
                context.Mesh = null;
                return context;
            }

            if (_firstEval)
            {
                for (int i = 0; i < NumCodes(); ++i) {
                    Compile(i);
                    
                }
                _firstEval = false;
                Changed = true;
            }

            const float toRad = (float) Math.PI/180f;

            var prevTransform = context.ObjectTWorld;
            var rand = new Random((int) Seed);

            var instancedMesh = MeshInput;
            
            var alwaysNeedsUpdate = Orientation > 0.5;

            if(!Changed && !alwaysNeedsUpdate) {
                context.Mesh = _bakedMesh;
                return context;
            }


            // Compute size of required vertex buffer and create it
            _bakedMesh.NumTriangles = instancedMesh.NumTriangles * numInstances;

            _bakedMesh.AttributesSize = instancedMesh.AttributesSize;
            _bakedMesh.InputElements = instancedMesh.InputElements;
            if (_bakedMesh.Vertices == null || _bakedMesh.Vertices.Description.SizeInBytes != instancedMesh.Vertices.Description.SizeInBytes * numInstances)
            {
                Utilities.DisposeObj(ref _bakedMesh.Vertices);
                _bakedMesh.Vertices = new Buffer(D3DDevice.Device, new BufferDescription()
                                                                  {
                                                                      BindFlags = BindFlags.StreamOutput | BindFlags.VertexBuffer,
                                                                      CpuAccessFlags = CpuAccessFlags.None,
                                                                      OptionFlags = ResourceOptionFlags.None,
                                                                      SizeInBytes = instancedMesh.Vertices.Description.SizeInBytes * numInstances,
                                                                      Usage = ResourceUsage.Default
                                                                  });
            }

            // instance data buffer
            const int instanceDataSize = 4*16; // float4x4
            var streamSize = numInstances * instanceDataSize;
            if (_instanceDataBuffer == null || _instanceDataBuffer.Description.SizeInBytes != streamSize)
            {
                Utilities.DisposeObj(ref _instanceDataBuffer);
                _instanceDataBuffer = new Buffer(context.D3DDevice, streamSize, ResourceUsage.Dynamic, BindFlags.VertexBuffer,
                                                 CpuAccessFlags.Write, ResourceOptionFlags.None, instanceDataSize);
            }

            DataStream instanceDataStream;
            context.D3DDevice.ImmediateContext.MapSubresource(_instanceDataBuffer, MapMode.WriteDiscard, MapFlags.None, out instanceDataStream);
            using (instanceDataStream)
            {
                instanceDataStream.Position = 0;

                for(var indexX = 0; indexX < intCountX; indexX ++) {
                    for(var indexY = 0; indexY < intCountY; indexY ++) {
                        for(var indexZ = 0; indexZ < intCountZ; indexZ ++) 
                        {
                            var fx = (indexX + 0.5f) / CountX;
                            var fy = (indexY + 0.5f) /CountY;
                            var fz = (indexZ + 0.5f) / CountZ;
                            
                            var t = Vector3.Zero;
                            
                            

                            //Matrix rotation;
                            Matrix transform;
                            var rotX = SpinX * toRad;
                            var rotY = SpinY * toRad;
                            var rotZ = SpinZ * toRad;
                            
                            // Triangular-pattern
                            if(Pattern == 1)
                            {
                                var isOdd = indexX % 2 > 0;
                                var verticalOffset= isOdd
                                                    ? (0.331f * SpreadY) 
                                                    : 0;
                                                    
                                const float TriangleScale = 0.581f;
                                t = new Vector3((float) ((indexX - intCountX/2 + 0.5f) * SpreadX * TriangleScale),
                                                (float) ((indexY - intCountY/2 + 0.5f) * SpreadY + verticalOffset),
                                                (float) (0));
                                rotZ+=  isOdd ? 120f*3.141578f / 360f : 0;
    
                            }
                            // Hexa-pattern
                            else if(Pattern == 2)
                            {
                                var isOddColumn = indexX % 2 == 0;
                                var isOddRow = indexY % 2 > 0;
                                var verticalOffset= isOddColumn ^ isOddRow? (0.331f * SpreadY) : 0;
                                                    
                                const float HexScale = 0.58f;
                                t = new Vector3((float) ((indexX - intCountX/2 + 0.5f) * SpreadX * HexScale),
                                                (float) ((indexY - intCountY/2 + 0.5f) * SpreadY + verticalOffset),
                                                (fz - 0.5f) * (SpreadZ * (CountZ)));
                                rotZ+=  (isOddColumn^isOddRow ) ? 120f*3.141578f / 360f : 0;
    
                            }                            // Spiral
                            else if (Pattern == 3) {
                                var n = indexX + indexY * intCountX ;
                                var SpiralOffset = SpreadY;
                                var SpiralSpin = 0;
                                
                                var angle =  (double)n * (137.5095 + (double)SpiralOffset / 100.0) + SpiralSpin * Math.PI/180; //137.508;
                                var radius = (double)SpreadX * Math.Sqrt(n);
                                t = new Vector3((float) ( Math.Sin( angle) * radius )  ,
                                                (float) ( Math.Cos( angle) * radius),
                                                (fz - 0.5f) * (SpreadZ * (CountZ)));
                                rotZ -= (float)angle;
                            
                            }
                            // Grid
                            else {
                                t = new Vector3( 
                                    (fx - 0.5f) * (SpreadX * (CountX)),   
                                    (fy - 0.5f) * (SpreadY * (CountY)),   
                                    (fz - 0.5f) * (SpreadZ * (CountZ))   
                                    );
                            }
                            
                            // Effects
                            var effectR = (t-EffectCenter).Length() + (float)(rand.NextDouble()-0.5f) * EffectRangeScatter;
                            t+= Center;
                            var effectF = Fade( Utilities.Clamp( (effectR - EffectRangeRadius) / -EffectRangeFallOff, 0,1));
                            

                            float s = (Scale + (float)rand.NextDouble() * ScaleRandom) * (effectF * ScaleEffect+1);
                            var scale = new Vector3(s, s, s);
                            
                            rotX += effectF * SpinEffectX * toRad;
                            rotY += effectF * SpinEffectY * toRad;
                            rotZ += effectF * SpinEffectZ * toRad;
                            

                            Matrix.RotationYawPitchRoll(rotX + (float)(rand.NextDouble()) * SpinRandomX * toRad,
                                                        rotY + (float)(rand.NextDouble()) * SpinRandomY * toRad,
                                                        rotZ + (float)(rand.NextDouble()) * SpinRandomZ * toRad,
                                                        out transform);
                                                                            
                            var tScale = Matrix.Scaling( scale.X, scale.Y, scale.Z);
                            transform= tScale * transform;
        
                            // Rotation to camera
                            if(Orientation > 0.5) {                            
                                var cameraToWorld = context.WorldToCamera;
                                cameraToWorld.Invert();
                                
                                var objectToWorldWithoutTranslation = context.ObjectTWorld;
                                var newObjectToWorld = cameraToWorld;
                                newObjectToWorld.Row4 = context.ObjectTWorld.Row4;
                                //Matrix.Translation(0,0,0, out newObjectToWorld);
                                transform *= newObjectToWorld;                        
                            }

                            var translation= Matrix.Translation(new Vector3( (float)(rand.NextDouble() -0.5f) * ( ScatterX),
                                                                    (float)(rand.NextDouble() -0.5f) * ( ScatterY),
                                                                    (float)(rand.NextDouble() -0.5f) * ( ScatterZ)));
                            transform =  translation * transform;

        
                            transform *= Matrix.Transformation(new Vector3(), new Quaternion(), new Vector3(1,1,1), new Vector3(), new Quaternion(), t);
                            instanceDataStream.Write(transform);
                        }
                    }
                }
            }
            context.D3DDevice.ImmediateContext.UnmapSubresource(_instanceDataBuffer, 0);

            var prevEffect = context.Effect;
            var prevRenderer = context.Renderer;
            context.Effect = _effect;
            context.Renderer = _renderer;

            try
            {
                _renderer.SetupEffect(context);

                if (context.DepthStencilView != null)
                    context.D3DDevice.ImmediateContext.OutputMerger.SetTargets(context.DepthStencilView, context.RenderTargetView);
                else
                    context.D3DDevice.ImmediateContext.OutputMerger.SetTargets(context.RenderTargetView);

                if (context.BlendState != null)
                {
                    context.D3DDevice.ImmediateContext.OutputMerger.BlendState = context.BlendState;
                }

                if (context.DepthStencilState != null)
                {
                    context.D3DDevice.ImmediateContext.OutputMerger.DepthStencilState = context.DepthStencilState;
                }

                if (context.RasterizerState != null)
                {
                    context.D3DDevice.ImmediateContext.Rasterizer.State = context.RasterizerState;
                }

                var technique = _effect.GetTechniqueByIndex(0);
                var pass = technique.GetPassByIndex(0);

                context.D3DDevice.ImmediateContext.Rasterizer.SetViewport(context.Viewport);
                context.D3DDevice.ImmediateContext.InputAssembler.InputLayout = new InputLayout(context.D3DDevice, pass.Description.Signature, _instanceDataInputElements);
                context.D3DDevice.ImmediateContext.InputAssembler.PrimitiveTopology = PrimitiveTopology.TriangleList;
                context.D3DDevice.ImmediateContext.InputAssembler.SetVertexBuffers(0, new [] { instancedMesh.Vertices, _instanceDataBuffer}, new [] {76, instanceDataSize}, new [] {0,0} );
                
                // Configure Pipeline for stream output
                context.D3DDevice.ImmediateContext.StreamOutput.SetTargets(new [] { new StreamOutputBufferBinding(_bakedMesh.Vertices, 0) });
                pass.Apply(context.D3DDevice.ImmediateContext);
    
                // Draw mesh into vertex buffer
                context.D3DDevice.ImmediateContext.DrawInstanced(instancedMesh.NumTriangles*3, numInstances, 0, 0);
                context.D3DDevice.ImmediateContext.StreamOutput.SetTargets(new [] { new StreamOutputBufferBinding(null, 0) });
            }
            catch (Exception exception)
            {
                Logger.Error(this,"Replicate2 - An error occured during evaluation: {0}", exception.Message);
            }

            context.Effect = prevEffect;
            context.Renderer = prevRenderer;
            
            context.Mesh = _bakedMesh;
            Changed =false;
            return context;
        }


        #region helpers

        public float Noise(int x, int seed)
        {
            int n = x + seed*137;
            n = (n << 13) ^ n;
            return (float) (1.0 - ((n*(n*n*15731 + 789221) + 1376312589) & 0x7fffffff)/1073741824.0);
        }

        public float Lerp(float a, float b, float t)
        {
            return a + t*(b - a);
        }

        public float Fade(float t)
        {
            return t*t*t*(t*(t*6 - 15) + 10);
        }

        public float Interpolate(float a, float b, float t)
        {
            float ft = t*3.1415927f;
            float f = (float) (1.0 - Math.Cos(ft))*0.5f;
            return a*(1.0f - f) + b*f;
        }

        private float getNoise(float value)
        {
            float noiseSum = 0.0f;
            value *= _frequency;
            value += _noiseTime+10000;


            noiseSum = Lerp(Noise((int) value, SEED),
                            Noise((int) value + 1, SEED),
                            Fade(value - (float) Math.Floor(value)));
            return noiseSum;
        }

        #endregion

        private Mesh _bakedMesh = new Mesh();
        private const int SEED = 0;
        private float _frequency = 1;
        private float _noiseTime;
        private OperatorPart.CollectOpPartFunctionsOfType<Framefield.Core.OperatorPartTraits.IMeshSupplier> _meshCollector = new OperatorPart.CollectOpPartFunctionsOfType<Framefield.Core.OperatorPartTraits.IMeshSupplier>();
        private Renderer _renderer;
        private Buffer _instanceDataBuffer;
        private InputElement[] _instanceDataInputElements 
            = new []
            {
                new InputElement("POSITION", 0, SharpDX.DXGI.Format.R32G32B32A32_Float, 0, 0),
                new InputElement("NORMAL", 0, SharpDX.DXGI.Format.R32G32B32_Float, 16, 0),
                new InputElement("COLOR", 0, SharpDX.DXGI.Format.R32G32B32A32_Float, 28, 0),
                new InputElement("TEXCOORD", 0, SharpDX.DXGI.Format.R32G32_Float, 44, 0),
                new InputElement("TANGENT", 0, SharpDX.DXGI.Format.R32G32B32_Float, 52, 0),
                new InputElement("BINORMAL", 0, SharpDX.DXGI.Format.R32G32B32_Float, 64, 0),
                new InputElement("INSTANCE_OBJ_TO_WORLD_ROW", 0, SharpDX.DXGI.Format.R32G32B32A32_Float, 0,
                                 1, InputClassification.PerInstanceData, 1),
                new InputElement("INSTANCE_OBJ_TO_WORLD_ROW", 1, SharpDX.DXGI.Format.R32G32B32A32_Float, 16,
                                 1, InputClassification.PerInstanceData, 1),
                new InputElement("INSTANCE_OBJ_TO_WORLD_ROW", 2, SharpDX.DXGI.Format.R32G32B32A32_Float, 32,
                                 1, InputClassification.PerInstanceData, 1),
                new InputElement("INSTANCE_OBJ_TO_WORLD_ROW", 3, SharpDX.DXGI.Format.R32G32B32A32_Float, 48,
                                 1, InputClassification.PerInstanceData, 1),
            };
    }
}

