//>>> _using
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SharpDX;
using SharpDX.Direct3D11;
using SharpDX.Windows;
//<<< _using
using SharpDX.Direct3D;
using Framefield.Core.OperatorPartTraits;
using Framefield.Core.Rendering;
using Buffer = SharpDX.Direct3D11.Buffer;
using Newtonsoft.Json;
using System.IO;

namespace Framefield.Core.ID9dfaf626_143e_43f0_898f_2d70fd41cc43
{
    public class Class_MidiInputPresetManager : OperatorPart.Function
    {
        //>>> _inputids
        private enum InputId
        {
            Code = 0,
            AddPresetTrigger = 1,
            UpdatePresetTrigger = 2,
            ApplyPresetTrigger = 3,
            UpdatePresetIndex = 4,
            PresetFolderPath = 5,
            Subtree = 6,
            CollectMidiInputsTrigger = 7,
            ApplyPresetIndex = 8,
            ApplyPresetWhenIndexChanges = 9
        }
        //<<< _inputids
        
        //>>> _outputids
        private enum OutputId
        {
            Output = 0,
            Index = 1
        }
        //<<< _outputids

        public override void Dispose() 
        {
            Utilities.DisposeObj(ref _bakedMesh);
            base.Dispose();
        }

        
        
        public override OperatorPartContext Eval(OperatorPartContext context, List<OperatorPart> inputs, int outputIdx)
        {
            //>>> _params
            var Code = inputs[(int)InputId.Code].Eval(context).Text;
            var AddPresetTrigger = inputs[(int)InputId.AddPresetTrigger].Eval(context).Value;
            var UpdatePresetTrigger = inputs[(int)InputId.UpdatePresetTrigger].Eval(context).Value;
            var ApplyPresetTrigger = inputs[(int)InputId.ApplyPresetTrigger].Eval(context).Value;
            var UpdatePresetIndex = inputs[(int)InputId.UpdatePresetIndex].Eval(context).Value;
            var PresetFolderPath = inputs[(int)InputId.PresetFolderPath].Eval(context).Text;
            var Subtree = inputs[(int)InputId.Subtree];
            var CollectMidiInputsTrigger = inputs[(int)InputId.CollectMidiInputsTrigger].Eval(context).Value;
            var ApplyPresetIndex = inputs[(int)InputId.ApplyPresetIndex].Eval(context).Value;
            var ApplyPresetWhenIndexChanges = (int) inputs[(int)InputId.ApplyPresetWhenIndexChanges].Eval(context).Value;
            //<<< _params

            _presetFolderPath = PresetFolderPath;
            
            var collectInputsTriggered = CollectMidiInputsTrigger > _collectMidiInputsTrigger;
            _collectMidiInputsTrigger = CollectMidiInputsTrigger;
            
            var applyTriggered= ApplyPresetTrigger > _applyPresetTrigger 
                                || (ApplyPresetWhenIndexChanges > 0.5f && (int)ApplyPresetIndex != _applyPresetIndex);
            _applyPresetTrigger = ApplyPresetTrigger;
            
            var addPresetTriggered = AddPresetTrigger > _addPresetTrigger; 
            _addPresetTrigger = AddPresetTrigger;

            var updatePresetTriggered = UpdatePresetTrigger > _updatePresetTrigger; 
            _updatePresetTrigger = UpdatePresetTrigger;


            if(collectInputsTriggered || _midiInputs == null) 
            {
                CollectCurrentSettings(inputs[(int)InputId.Subtree]);
            } 
            
            
            if( addPresetTriggered) 
            {
                _updatePresetIndex++;
                AddPreset(inputs[(int)InputId.UpdatePresetIndex]);
            }
            else {
                _updatePresetIndex = (int)UpdatePresetIndex;
            }
            _applyPresetIndex = (int)ApplyPresetIndex;
            
            if( applyTriggered) 
            {
                ApplyPreset();
            }
            
            if( updatePresetTriggered ) {
                _updatePresetIndex =  (int)ApplyPresetIndex;
                Logger.Info("Saving preset " + _updatePresetIndex);
                SavePreset();
            }
            
            inputs[(int)InputId.Subtree].Eval(context);
            context.Value = _updatePresetIndex;
            
            
            return context;
        }


        private void CollectCurrentSettings(OperatorPart input) 
        {
            var midiInputCollector = new OperatorPart.CollectOpPartFunctionsOfType<OperatorPartTraits.IMidiInput>();
            input.TraverseWithFunction(midiInputCollector, null);
            _midiInputs = new List<IMidiInput>();
            foreach (var opPartFunction in midiInputCollector.CollectedOpPartFunctions)
            {
                _midiInputs.Add( opPartFunction as IMidiInput);
            }   
        }
        
        // --- SAVING PRESETS -------------------------------------------
        private void AddPreset(OperatorPart presetIndexInput) 
        {   
            // Increment preset index
            lock (this)
            {
                 presetIndexInput.Func = Utilities.CreateValueFunction(new Float((float)_updatePresetIndex));
            }       
            
            SavePreset();
        }
        
        private void SavePreset() 
        {
            var newPreset = new Preset() { Settings= new List<PresetSetting>() };
            foreach(var midiInput in _midiInputs) 
            {
                newPreset.Settings.Add(new PresetSetting() {
                    DeviceId= midiInput.Device,
                    Channel = (int)midiInput.Channel,
                    Control = (int)midiInput.Control,
                    MidiValue = (int)midiInput.CurrentMidiValue,
                });
            }
            
            using (var sw = new StreamWriter(GetFilepathForCurrentIndex(_updatePresetIndex)))
            {
                sw.Write(JsonConvert.SerializeObject(newPreset, Formatting.Indented));
            }
        }
        
        private string GetFilepathForCurrentIndex(int index) 
        {
            return _presetFolderPath + "midiPreset-" + index + ".json";
        }
        
        
        // --- APPLYING PRESETS --------------------------------------------------
        private void ApplyPreset() 
        {
            var filepath = GetFilepathForCurrentIndex(_applyPresetIndex);
            Preset preset;
            if (!File.Exists(filepath))
            {
                Logger.Warn(this, "File doesn't exist:  " + filepath);
                return;
            }
            
            using (var reader = new StreamReader(filepath))
            {
                var jsonString = reader.ReadToEnd();
                preset = JsonConvert.DeserializeObject<Preset>(jsonString);
                if (preset == null)
                {
                    Logger.Warn("Loading preset failed:" + filepath);
                    return;
                }        
                Logger.Info(this, "Loaded preset with settings: " + preset.Settings.Count());
            }
            
            int appliedSettingsCount = 0;
            foreach(var midiInput in _midiInputs) 
            {
                foreach(var setting in preset.Settings) 
                {
                    if( midiInput.Device == setting.DeviceId
                        && (int)midiInput.Channel == setting.Channel
                        && (int)midiInput.Control == setting.Control
                        && midiInput.AllowPresets
                    )
                    {
                        appliedSettingsCount ++;
                        midiInput.TargetMidiValue = setting.MidiValue;
                        midiInput.CurrentMidiValue = setting.MidiValue;
                        break;
                    }
                }
            }
            Logger.Info(this, "Applied " + appliedSettingsCount + " settings");
        }
        
        
        struct PresetSetting {
            public string DeviceId;
            public int Channel;
            public int Control;
            public int MidiValue;            
        }
        
        class Preset {
            public List<PresetSetting> Settings;
        }       
        
        private string _presetFolderPath = "./";
        private List<IMidiInput> _midiInputs;
        private int _updatePresetIndex;
        private int _applyPresetIndex;
        private bool _firstEval = true;
        private float _addPresetTrigger;
        private float _applyPresetTrigger;
        private float _updatePresetTrigger;
        private float _collectMidiInputsTrigger;

        private Mesh _bakedMesh = new Mesh();
        private MeshCollector _meshCollector;
    }
}


