package game;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import org.json.simple.JSONArray;
import org.json.simple.JSONObject;

import bot.Bot.Direction;
import bot.BotManager;
import game.Map.ObstacleType;
import util.GaussianBlur;
import util.Position;
import util.Util;

public class Map {

	public enum ObstacleType {
		FLOOR, WALL, DOOR, PELLET, 
		SUPERPELLET, GHOST, PACMAN, ME
	}
	
	public ObstacleType[][] obstacles;
	
	public HashMap<Integer, Ghost> opponents;
	
	public boolean canMove = false;
	
	public Map() {
		this.opponents = new HashMap<Integer, Ghost>();
	}
	
	public Map(Map map) {
		this.obstacles = Util.deepCopy(map.obstacles);
		this.opponents = (HashMap<Integer, Ghost>) map.opponents.clone();
	}
	
	public void update(JSONObject map) {
		JSONArray array = (JSONArray) map.get("content");
		int height = ((Long) map.get("height")).intValue();
		int pelletsleft = ((Long) map.get("pelletsleft")).intValue();
		int width = ((Long) map.get("width")).intValue();
		
		ObstacleType[][] obstacles = new ObstacleType[width][height];
		
		for (int h = 0; h < height; h++) {
			String row = (String) array.get(h);
			for (int w = 0; w < width; w++) {
				obstacles[w][h] = getObstacleType(row.charAt(w));
			}
		}
		this.obstacles = obstacles;		
		this.canMove = true;
	}
	
	public void updateOpponents(JSONArray opponents) {
		if (opponents == null) {
			return;
		}
		clearGhostsFromMap();
		for (int i = 0; i < opponents.size(); i++) {
			JSONObject opponent = (JSONObject) opponents.get(i);
				
			Ghost ghost = new Ghost(((Long) opponent.get("id")).intValue());
			ghost.score = ((Long) opponent.get("score")).intValue();
			ghost.position = new Position<Integer, Integer>(
					((Long) opponent.get("x")).intValue(), ((Long) opponent.get("y")).intValue());
			ghost.id = ((Long) opponent.get("id")).intValue();
			ghost.isDangerous = (boolean) opponent.get("isdangerous");
				
			obstacles[ghost.position.x][ghost.position.y] = ObstacleType.GHOST;
			
			if (this.opponents.containsKey(ghost.id)) {
				this.opponents.replace(ghost.id, ghost);
			} else {
				this.opponents.put(ghost.id, ghost);
			}
		}
	}

	private void clearGhostsFromMap() {
		for (int x = 0; x < obstacles.length; x++) {
			for (int y = 0; y < obstacles[x].length; y++) {
				if (obstacles[x][y] != ObstacleType.GHOST) {
					continue;
				}
				obstacles[x][y] = ObstacleType.FLOOR;
			}
		}
	}
	
	public List<Direction> getAvailableMoves(Position<Integer, Integer> position) {
		List<Direction> directions = new ArrayList<Direction>();
		try {
			if (obstacles[position.x][position.y - 1] != ObstacleType.WALL) {
				directions.add(Direction.UP);
			}
		} catch (Exception e) {}
		try {
			if (obstacles[position.x][position.y + 1] != ObstacleType.WALL) {
				directions.add(Direction.DOWN);
			} 
		} catch (Exception e) {}
		try {
			if (obstacles[position.x - 1][position.y] != ObstacleType.WALL) {
				directions.add(Direction.LEFT);
			}
		} catch (Exception e) {}
		try {
			if (obstacles[position.x + 1][position.y] != ObstacleType.WALL) {
				directions.add(Direction.RIGHT);
			}
		} catch (Exception e) {}
		return directions;
	}
	
	public static Position<Integer, Integer> getPosition(Position<Integer, Integer> position, Direction direction) {
		if (direction == Direction.DOWN) {
			return new Position<Integer, Integer>(position.x, position.y + 1);
		} else if (direction == Direction.UP) {
			return new Position<Integer, Integer>(position.x, position.y - 1);
		} else if (direction == Direction.LEFT) {
			return new Position<Integer, Integer>(position.x - 1, position.y);
		} else if (direction == Direction.RIGHT) {
			return new Position<Integer, Integer>(position.x + 1, position.y);
		}
		return null;
	}
	
	private ObstacleType getObstacleType(char c) {
		switch (c) {
		
			case '|':
				return ObstacleType.WALL;
				
			case '_':
				return ObstacleType.FLOOR;
				
			case '.':
				return ObstacleType.PELLET;
				
			case 'o':
				return ObstacleType.SUPERPELLET;
				
			case '-':
				return ObstacleType.DOOR;
		}
		return ObstacleType.FLOOR;
	}
	
	public float calculateScore(Position<Integer, Integer> position) {
		int r = 3;
		
		float[] pascal = new float[r];
		for (int i = 0; i < r; i++) {
			pascal[i] = Util.pascalTriangle(r, i + 1);
		}
		
		float score = 0.0f;
		for (int x = 0; x < r; x++) {
			for (int y = 0; y < r; y++) {
				score += getScore(
					new Position<Integer, Integer>(position.x - r/2 + x, position.y - r/2 + y))
					 * pascal[x] * pascal[y];
			}
		}

		return score + getScore(position);
	}
	
	private boolean hasDeadlyGhostInRange(Position<Integer, Integer> position) {
		for (Ghost ghost : opponents.values()) {
			if (Math.sqrt(Math.pow(ghost.position.x - position.x, 2) + 
					Math.pow(ghost.position.y - position.y, 2)) > 2.0) {
				continue;
			}
			if (ghost.isDangerous) {
				return true;
			}
		}
		return false;
	}
	
	private float getScore(Position<Integer, Integer> position) {
		try {
			switch (obstacles[position.x][position.y]) {
				case DOOR:
					return 0.0f;
					
				case FLOOR:
					return 0.0f;
					
				case GHOST:
					if (hasDeadlyGhostInRange(position)) {
						return -25.0f;
					}
					if (BotManager.getInstance().bot.ghost.isDangerous) {
						return 25.0f;
					}
					return -0.5f;
					
				case PELLET:
					return 1.0f;
					
				case SUPERPELLET:
					return 10.0f;
	
				case WALL:
					return 0.0f;
					
				default:
					break;
			}
		} catch (Exception e) {}
		return 0;
	}
	
	public void clear(Position<Integer, Integer> position) {
		obstacles[position.x][position.y] = ObstacleType.FLOOR;
	}
}
