﻿/*
The MIT License

Copyright (c) 2010 Christoph Husse

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using SettlersEngine;

namespace GhostlyAi
{
    public interface IPathNode<in TUserContext>
    {
        Boolean IsWalkable(TUserContext inContext);
        int X { get; set; }
        int Y { get; set; }
        TileType TileType { get; set; }
        double G { get; set; }
        double H { get; set; }
        double F { get; set; }
        double Points { get; set; }
    }

    public interface IIndexedObject
    {
        int Index { get; set; }
    }

    /// <summary>
    /// Uses about 50 MB for a 1024x1024 grid.
    /// </summary>
    public class SpatialAStar<TPathNode, TUserContext> where TPathNode : IPathNode<TUserContext>
    {
        private OpenCloseMap _mClosedSet;
        private OpenCloseMap _mOpenSet;
        private PriorityQueue<PathNode> _mOrderedOpenSet;
        private PathNode[,] _mCameFrom;
        private OpenCloseMap _mRuntimeGrid;
        private PathNode[,] _mSearchSpace;

        public TPathNode[,] SearchSpace
        {
            get { return _searchSpace; }
            set
            {
                _searchSpace = value;
                Width = value.GetLength(0);
                Height = value.GetLength(1);
                _mSearchSpace = new PathNode[Width, Height];
                _mClosedSet = new OpenCloseMap(Width, Height);
                _mOpenSet = new OpenCloseMap(Width, Height);
                _mCameFrom = new PathNode[Width, Height];
                _mRuntimeGrid = new OpenCloseMap(Width, Height);
//                _mOrderedOpenSet = new PriorityQueue<PathNode>(PathNode.Comparer);
                _mOrderedOpenSet = new PriorityQueue<PathNode>(new PathNode(0, 0, default(TPathNode)));

                for (int x = 0; x < Width; x++)
                {
                    for (int y = 0; y < Height; y++)
                    {
                        if (value[x, y] == null)
                            throw new ArgumentNullException();

                        _mSearchSpace[x, y] = new PathNode(x, y, value[x, y]);
                    }
                }
            }
        }

        public int Width { get; set; }
        public int Height { get; set; }

        public class PathNode : IPathNode<TUserContext>, IComparer<PathNode>, IIndexedObject
        {
            // ReSharper disable once StaticMemberInGenericType
            //public static readonly PathNode Comparer = new PathNode(0, 0, default(TPathNode));

            public TPathNode UserContext { get; internal set; }
            public double G { get; set; }
            public double H { get; set; }
            public double F { get; set; }
            public double Points { get; set; }
            public int Index { get; set; }

            public bool IsWalkable(TUserContext inContext)
            {
                return UserContext.IsWalkable(inContext);
            }

            public int X { get; set; }
            public int Y { get; set; }
            public TileType TileType { get; set; }

            public int Compare(PathNode x, PathNode y)
            {
                if (x != null && x.F < y.F)
                    return -1;
                return x != null && x.F > y.F ? 1 : 0;
            }

            public PathNode(int inX, int inY, TPathNode inUserContext)
            {
                X = inX;
                Y = inY;
                UserContext = inUserContext;
            }
        }

        public SpatialAStar(TPathNode[,] inGrid)
        {
            if(inGrid != null)
                SearchSpace = inGrid;
        }

        public virtual double Heuristic(Point inStart, Point inEnd, TUserContext inUserContext)
        {
            return Math.Sqrt((inStart.X - inEnd.X) * (inStart.X - inEnd.X) + (inStart.Y - inEnd.Y) * (inStart.Y - inEnd.Y));
        }

        private readonly double _sqrt2 = Math.Sqrt(2);
        private TPathNode[,] _searchSpace;

        protected virtual double NeighborDistance(PathNode inFrom, PathNode inTo, PathNode endPathNode, TUserContext inUserContext)
        {
            int diffX = Math.Abs(inFrom.X - inTo.X);
            int diffY = Math.Abs(inFrom.Y - inTo.Y);

            switch (diffX + diffY)
            {
                case 1: return 1;
                case 2: return _sqrt2;
                case 0: return 0;
                default:
                    throw new ApplicationException();
            }
        }

        //private List<Int64> elapsed = new List<long>();

        /// <summary>
        /// Returns null, if no path is found. Start- and End-Node are included in returned path. The user context
        /// is passed to IsWalkable().
        /// </summary>
        public LinkedList<TPathNode> Search(Point inStartNode, Point inEndNode, TUserContext inUserContext)
        {
            PathNode startNode = _mSearchSpace[inStartNode.X, inStartNode.Y];
            PathNode endNode = _mSearchSpace[inEndNode.X, inEndNode.Y];

            //System.Diagnostics.Stopwatch watch = new System.Diagnostics.Stopwatch();
            //watch.Start();

            if (startNode == endNode)
                return new LinkedList<TPathNode>(new[] { startNode.UserContext });

            PathNode[] neighborNodes = new PathNode[4];

            _mClosedSet.Clear();
            _mOpenSet.Clear();
            _mRuntimeGrid.Clear();
            _mOrderedOpenSet.Clear();

            for (int x = 0; x < Width; x++)
            {
                for (int y = 0; y < Height; y++)
                {
                    _mCameFrom[x, y] = null;
                }
            }

            startNode.G = 0;
            startNode.UserContext.G = startNode.G;

            startNode.H = Heuristic(new Point(startNode.X, startNode.Y), new Point(endNode.X, endNode.Y),   inUserContext);
            startNode.UserContext.H = startNode.H;

            startNode.F = startNode.H;
            startNode.UserContext.F = startNode.F;


            startNode.UserContext.Points = 0;

            

            _mOpenSet.Add(startNode);
            _mOrderedOpenSet.Push(startNode);

            _mRuntimeGrid.Add(startNode);

            while (!_mOpenSet.IsEmpty)
            {
                var x = _mOrderedOpenSet.Pop();

                if (x == endNode)
                {
                    // watch.Stop();

                    //elapsed.Add(watch.ElapsedMilliseconds);

                    LinkedList<TPathNode> result = ReconstructPath(_mCameFrom, _mCameFrom[endNode.X, endNode.Y]);

                    result.AddLast(endNode.UserContext);

                    //remove self
                    //result.RemoveFirst();

                    foreach (var node in result.ToList())
                    {
                        if (node.X == startNode.X && node.Y == startNode.Y)
                            result.Remove(node);
                    }

                    //fix points
                    var points = 0;
                    foreach (var pathNode in result)
                    {
                        if (pathNode.TileType == TileType.Pellet) points += 1;
                        else if (pathNode.TileType == TileType.SuperPellet) points += 10;

                        pathNode.Points = points;
                    }


                    return result;
                }

                _mOpenSet.Remove(x);
                _mClosedSet.Add(x);

                StoreNeighborNodes(x, neighborNodes);

                foreach (var y in neighborNodes)
                {
                    bool tentativeIsBetter;

                    if (y == null)
                        continue;

                    if (!y.UserContext.IsWalkable(inUserContext))
                        continue;

                    if (_mClosedSet.Contains(y))
                        continue;

                    

                    var tentativeGScore = _mRuntimeGrid[x].G + NeighborDistance(x, y, endNode, inUserContext);

                    var point = 0;
                    if (y.UserContext.TileType == TileType.Pellet)
                        point = 1;
                    else if (y.UserContext.TileType == TileType.SuperPellet)
                        point = 10;
//                    else if (y.UserContext.TileType == TileType.Player && ((Gamestate) inUserContext).You.Isdangerous)
//                        point = 3;

                    var tentativePoint = _mRuntimeGrid[x].UserContext.Points + point;

                    var wasAdded = false;

                    if (!_mOpenSet.Contains(y))
                    {
                        _mOpenSet.Add(y);
                        tentativeIsBetter = true;
                        wasAdded = true;
                    }
                    else if (tentativeGScore < _mRuntimeGrid[y].G)
                    {
                        tentativeIsBetter = true;
                    }
                    else
                    {
                        tentativeIsBetter = false;
                    }

                    if (tentativeIsBetter)
                    {
                        _mCameFrom[y.X, y.Y] = x;

                        if (!_mRuntimeGrid.Contains(y))
                            _mRuntimeGrid.Add(y);

                        _mRuntimeGrid[y].G = tentativeGScore;
                        _mRuntimeGrid[y].UserContext.G = _mRuntimeGrid[y].G;

//                        if(endNode.UserContext.TileType == TileType.SuperPellet)
//                            _mRuntimeGrid[y].H = Heuristic(new Point(y.X, y.Y), new Point(endNode.X, endNode.Y), inUserContext)*-1;
//                        else
                        _mRuntimeGrid[y].H = Heuristic(new Point(y.X, y.Y), new Point(endNode.X, endNode.Y), inUserContext);

                        _mRuntimeGrid[y].UserContext.H = _mRuntimeGrid[y].H;

                        _mRuntimeGrid[y].F = _mRuntimeGrid[y].G + _mRuntimeGrid[y].H;
                        _mRuntimeGrid[y].UserContext.F = _mRuntimeGrid[y].F;


                        _mRuntimeGrid[y].UserContext.Points = tentativePoint;



                        if (wasAdded)
                            _mOrderedOpenSet.Push(y);
                        else
                            _mOrderedOpenSet.Update(y);
                    }
                }
            }

            return null;
        }

        private LinkedList<TPathNode> ReconstructPath(PathNode[,] cameFrom, PathNode currentNode)
        {
            LinkedList<TPathNode> result = new LinkedList<TPathNode>();

            ReconstructPathRecursive(cameFrom, currentNode, result);

            return result;
        }

        private void ReconstructPathRecursive(PathNode[,] cameFrom, PathNode currentNode, LinkedList<TPathNode> result)
        {
            PathNode item = cameFrom[currentNode.X, currentNode.Y];

            if (item != null)
            {
                ReconstructPathRecursive(cameFrom, item, result);

                result.AddLast(currentNode.UserContext);
            }
            else
                result.AddLast(currentNode.UserContext);
        }

        private void StoreNeighborNodes(PathNode inAround, PathNode[] inNeighbors)
        {
            int x = inAround.X;
            int y = inAround.Y;

//            if ((x > 0) && (y > 0))
//                inNeighbors[0] = m_SearchSpace[x - 1, y - 1];
//            else
//                inNeighbors[0] = null;

            if (y > 0)
                inNeighbors[0] = _mSearchSpace[x, y - 1];
            else
                //inNeighbors[0] = null;
                inNeighbors[0] = _mSearchSpace[x, Height - 1];

            //            if ((x < Width - 1) && (y > 0))
            //                inNeighbors[2] = m_SearchSpace[x + 1, y - 1];
            //            else
            //                inNeighbors[2] = null;

            if (x > 0)
                inNeighbors[1] = _mSearchSpace[x - 1, y];
            else
                //inNeighbors[1] = null;
                inNeighbors[1] = _mSearchSpace[Width - 1,y];

            if (x < Width - 1)
                inNeighbors[2] = _mSearchSpace[x + 1, y];
            else
                //inNeighbors[2] = null;
                inNeighbors[2] = _mSearchSpace[Width - 1, y];

            //            if ((x > 0) && (y < Height - 1))
            //                inNeighbors[5] = m_SearchSpace[x - 1, y + 1];
            //            else
            //                inNeighbors[5] = null;

            if (y < Height - 1)
                inNeighbors[3] = _mSearchSpace[x, y + 1];
            else
                inNeighbors[3] = _mSearchSpace[x, Height - 1];

            //            if ((x < Width - 1) && (y < Height - 1))
            //                inNeighbors[7] = m_SearchSpace[x + 1, y + 1];
            //            else
            //                inNeighbors[7] = null;
        }

        private class OpenCloseMap
        {
            private readonly PathNode[,] _mMap;
            private int Width { get; }
            private int Height { get; }
            private int Count { get; set; }

            public PathNode this[PathNode node]
            {
                get
                {
                    return _mMap[node.X, node.Y];
                }

            }

            public bool IsEmpty
            {
                get
                {
                    return Count == 0;
                }
            }

            public OpenCloseMap(int inWidth, int inHeight)
            {
                _mMap = new PathNode[inWidth, inHeight];
                Width = inWidth;
                Height = inHeight;
            }

            public void Add(PathNode inValue)
            {
                Count++;
                _mMap[inValue.X, inValue.Y] = inValue;
            }

            public bool Contains(PathNode inValue)
            {
                PathNode item = _mMap[inValue.X, inValue.Y];

                if (item == null)
                    return false;

#if DEBUG
                if (!inValue.Equals(item))
                    throw new ApplicationException();
#endif

                return true;
            }

            public void Remove(PathNode inValue)
            {
                Count--;
                _mMap[inValue.X, inValue.Y] = null;
            }

            public void Clear()
            {
                Count = 0;

                for (int x = 0; x < Width; x++)
                {
                    for (int y = 0; y < Height; y++)
                    {
                        _mMap[x, y] = null;
                    }
                }
            }
        }
    }
}
