#include "camera.h"
#include "globals.h"



static const float MaxVerticalAngle = 85.0f; //must be less than 90 to avoid gimbal lock

struct Camera SetCamera()
{
	struct Camera camera;
	camera.fov = 70.0f;
	camera.horizontalAngle = 0.0f;
	camera.verticalAngle = 0.0f;
	camera.nearPlane = 0.01f;
	camera.farPlane = 200.0f;
	camera.viewportAspectRatio = 16.0f / 9.0f;

	camera.position = vec3(0.0f, 0.0f, 4.0f);

	camera.cameraOrientation = m4_identity();
	camera.cameraMatrix = m4_identity();
	camera.cameraView = m4_identity();
	camera.cameraProjection = m4_identity();

	return camera;
}

void SetCameraPosition(struct Camera *camera, vec3_t newPosition)
{
	camera->position = newPosition;
}

void SetCameraViewportAspectRatio(struct Camera *camera, float newAspectRatio)
{
	camera->viewportAspectRatio = newAspectRatio;
}

void OffsetCameraPosition(struct Camera *camera, vec3_t offset)
{
	camera->position = v3_add(camera->position, offset);
}

vec3_t GetCameraForwardVector(struct Camera *camera)
{
	mat4_t invertedOrientation = m4_invert_affine( CameraOrientation(camera) );
	vec3_t forward = m4_mul_dir(invertedOrientation, vec3(0,0,-1));
	return forward;
}

vec3_t GetCameraRightVector(struct Camera *camera)
{
	mat4_t invertedOrientation = m4_invert_affine( CameraOrientation(camera) );
	vec3_t right = m4_mul_dir(invertedOrientation, vec3(1,0,0));
	return right;
}

vec3_t GetCameraUpVector(struct Camera *camera)
{
	mat4_t invertedOrientation = m4_invert_affine( CameraOrientation(camera) );
	vec3_t up = m4_mul_dir(invertedOrientation, vec3(0,1,0));
	return up;
}

mat4_t CameraOrientation(struct Camera *camera)
{	
	mat4_t rotationX = m4_rotation_x( DEGREES_TO_RADS * (camera->verticalAngle) );
	mat4_t rotationY = m4_rotation_y( DEGREES_TO_RADS * (camera->horizontalAngle) );
	mat4_t orientation = m4_mul(rotationX, rotationY);

	camera->cameraOrientation = orientation;
	return orientation;
}

mat4_t CameraView(struct Camera *camera)
{
	mat4_t view;
	mat4_t translationMatrix = m4_translation( camera->position );
	view = m4_mul(CameraOrientation( camera ), translationMatrix );

	camera->cameraView = view;
	return view;
}

mat4_t CameraProjection(struct Camera *camera)
{
	mat4_t projection = m4_perspective( camera->fov, camera->viewportAspectRatio, camera->nearPlane, camera->farPlane);

	camera->cameraProjection = projection;
	return projection;
}

mat4_t CameraMatrix(struct Camera *camera)
{
	mat4_t cameraMatrix = m4_mul( CameraProjection(camera), CameraView(camera) );

	camera->cameraMatrix = cameraMatrix;
	return cameraMatrix;
}

void CameraNormalizeAngles(struct Camera *camera)
{
	camera->horizontalAngle = fmodf(camera->horizontalAngle, 360.0f);
	if (camera->horizontalAngle < 0.0f) {
		camera->horizontalAngle += 360.0f;
	}

	if(camera->verticalAngle > MaxVerticalAngle)
        camera->verticalAngle = MaxVerticalAngle;
    else if(camera->verticalAngle < -MaxVerticalAngle)
        camera->verticalAngle = -MaxVerticalAngle;

}

void UpdateCameraMatrices(struct Camera *camera)
{
	
}

// jotkut näistä matriiseista vois olla muuttujinakin. aina ei välttämättä tarttis laskea uudestaan.
// muuttujina tuolla camera structin sisällä. ja sitten nuo funktiot muuttaa sitä kyseistä muuttujaa.

CameraKeyFrame NewCameraKeyFrame(vec3_t position,  float horizontalAngle, float verticalAngle, int demoTime)
{
	CameraKeyFrame newCameraKeyFrame;
	newCameraKeyFrame.position = position;
	newCameraKeyFrame.horizontalAngle = horizontalAngle;
	newCameraKeyFrame.verticalAngle = verticalAngle;
	newCameraKeyFrame.demoTime = demoTime;
	return newCameraKeyFrame;
}