//-----------------------------LICENSE NOTICE------------------------------------
//  This file is part of Hair Boy
//  Copyright (C) 2016 Carlos Sevila (@CarlosSevila)
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//------------------------------------------------------------------------------

#include "game.h"

#include "../util/util.h"

u8 map[MAP_WIDTH][MAP_HEIGHT];
u8 currentMap;
TPlayer player;
TBlock block;
TDeformBlock deformBlocks[MAX_DEFORM_BLOCKS];
u8 numDeformBlocks;
u8 deadCount;
u8 winCount;
u16 totalDefeats;
u8 timeCount;
u8 seconds;
u16 minutes;

void InitializeGame(void) {
	currentMap = 0;
	deadCount = 0;
	winCount = 0;
	totalDefeats = 0;
	seconds = 0;
	minutes = 0;
	timeCount = 50;

	cpct_waitVSYNC();

	LoadMap(currentMap);
}

void GameLoop(void) {
	u8 ticks = 0;

	while (1) {

		// Update.
		if (!CheckDeath() && !CheckWin()) {
			CheckInput(&ticks);
			UpdatePlayer(&ticks);
			if (block.active) {
				UpdateBlock(&ticks);
			}
			UpdateDeformBlocks();

			if (deadCount > 0) {
				SetAnimationFrame(&player.entity, kPlayerDeath, 0);
			} else if (winCount > 0) {
				SetAnimationFrame(&player.entity, kPlayerWin, 1);
			}
		}

		// End game.
		if (cpct_isKeyPressed(Key_Esc) || currentMap == NUM_MAPS) {
			break;
		}

		UpdateTotalTime();

		cpct_waitVSYNC();

		// Draw.
		DrawEntity(&player.entity);
		if (block.active) {
			DrawEntity(&block.entity);
		}
		DrawTotalTime(0);

		ticks++;
	}
}

void ResetPlayer(void) {
	player.entity.x = player.startX;
	player.entity.y = player.startY;
	player.entity.prevX = player.entity.x;
	player.entity.prevY = player.entity.y;
	player.entity.right = player.entity.x + TILE_WIDTH;
	player.entity.bottom = player.entity.y + TILE_HEIGHT;
	player.state = FALLING;
	player.jumpFactor = JUMP_SIZE - 1;
	player.jumpBlocked = 0;
	player.leftSpeed = 0;
	player.rightSpeed = 0;
}

void CheckInput(u8 *ticks) {
	u8 jumpPressed = 0;
	u8 leftPressed = 0;
	u8 rightPressed = 0;

	cpct_scanKeyboard_f();

	jumpPressed = cpct_isKeyPressed(Key_Space);
	leftPressed = cpct_isKeyPressed(Key_CursorLeft);
	rightPressed = cpct_isKeyPressed(Key_CursorRight);

	if (jumpPressed == 0) {
		player.jumpBlocked = 0;
	}

	player.entity.prevX = player.entity.x;
	player.entity.prevY = player.entity.y;

	if (rightPressed) {
		if (player.leftSpeed > 0) {
			player.leftSpeed -= 2;
			if (player.leftSpeed < 2 || player.leftSpeed > MAX_SPEED) {
				player.leftSpeed = 0;
			}
		} else if (player.rightSpeed < MAX_SPEED) {
			player.rightSpeed++;
		}
	} else if (leftPressed) {
		if (player.rightSpeed > 0) {
			player.rightSpeed -= 2;
			if (player.rightSpeed < 2 || player.rightSpeed > MAX_SPEED) {
				player.rightSpeed = 0;
			}
		} else if (player.leftSpeed < MAX_SPEED) {
			player.leftSpeed++;
		}
	} else {
		if (player.rightSpeed > 0) {
			if (player.state == GROUNDED) {
				player.rightSpeed -= 2;

				if (player.rightSpeed < 2 || player.rightSpeed > MAX_SPEED) {
					player.rightSpeed = 0;
				}
			} else if ((*ticks & 1) == 0) {
				player.rightSpeed--;
			}
		} else if (player.leftSpeed > 0) {
			if (player.state == GROUNDED) {
				player.leftSpeed -= 2;

				if (player.leftSpeed < 2 || player.leftSpeed > MAX_SPEED) {
					player.leftSpeed = 0;
				}
			} else if ((*ticks & 1) == 0) {
				player.leftSpeed--;
			}
		}
	}

	if (player.state == GROUNDED && jumpPressed) {
		if (player.jumpBlocked == 0) {
			player.jumpBlocked = 1;
			player.state = JUMPING;
			cpct_akp_SFXPlay(1, 13, 56, 0, 0, AY_CHANNEL_B);
			// Parameters: instrument number, volume (0-15), note, speed (0 = original), inverted pitch, channel number (0-2)
		}
	}

	if (player.jumpBlocked == 0) {
		if (CheckMapCollision(player.entity.right, player.entity.y) || CheckMapCollision(player.entity.right, player.entity.bottom - 1)) {
			if (leftPressed || rightPressed) {
				if (jumpPressed) {
					player.jumpBlocked = 1;
					player.state = JUMPING;
					player.leftSpeed = MAX_SPEED;
					player.rightSpeed = 0;
					player.jumpFactor = 0;
					cpct_akp_SFXPlay(1, 13, 56, 0, 0, AY_CHANNEL_B);
				}
			}
		} else if (CheckMapCollision(player.entity.x - 1, player.entity.y) && CheckMapCollision(player.entity.x - 1, player.entity.bottom - 1)) {
			if (leftPressed || rightPressed) {
				if (jumpPressed) {
					player.jumpBlocked = 1;
					player.state = JUMPING;
					player.rightSpeed = MAX_SPEED;
					player.leftSpeed = 0;
					player.jumpFactor = 0;
					cpct_akp_SFXPlay(1, 13, 56, 0, 0, AY_CHANNEL_B);
				}
			}
		}
	}
}

void UpdatePlayer(u8 *ticks) {
	u8 hasToMove = 1;

	// Horizontal movement.
	if (player.rightSpeed > 0) {
		player.lastSpeedIsRight = 1;
		if (!CheckMapCollision(player.entity.right, player.entity.y) && !CheckMapCollision(player.entity.right, player.entity.bottom - 1)) {
			if (player.rightSpeed < HALF_MAX_SPEED) {
				if ((*ticks & 1) == 0) {
					hasToMove = 0;
				}
			}

			if (hasToMove) {
				player.entity.x++;
				player.entity.right++;
			}
		} else {
			player.rightSpeed = 0;
		}
	} else if (player.leftSpeed > 0) {
		player.lastSpeedIsRight = 0;
		if (!CheckMapCollision(player.entity.x - 1, player.entity.y) && !CheckMapCollision(player.entity.x - 1, player.entity.bottom - 1)) {
			if (player.leftSpeed < HALF_MAX_SPEED) {
				if ((*ticks & 1) == 0) {
					hasToMove = 0;
				}
			}

			if (hasToMove) {
				player.entity.x--;
				player.entity.right--;
			}
		} else {
			player.leftSpeed = 0;
		}
	}

	// Vertical movement.
	if (player.state == JUMPING) {
		if (!CheckMapCollision(player.entity.x, player.entity.y - kJumpValues[player.jumpFactor]) && !CheckMapCollision(player.entity.right - 1, player.entity.y - kJumpValues[player.jumpFactor])) {
			player.entity.y -= kJumpValues[player.jumpFactor];
			player.entity.bottom -= kJumpValues[player.jumpFactor];

			if (player.jumpFactor < JUMP_SIZE - 1) {
				player.jumpFactor++;
			} else {
				player.state = FALLING;
				player.jumpFactor = JUMP_SIZE - 1;
			}
		} else {
			player.state = FALLING;
			player.jumpFactor = JUMP_SIZE - 1;

			// Move until it touches.
			while (!CheckMapCollision(player.entity.x, player.entity.y - 1) && !CheckMapCollision(player.entity.right - 1, player.entity.y - 1)) {
				player.entity.y -= 1;
				player.entity.bottom -= 1;
			}
		}
	} else if (player.state == FALLING) {
		if (!CheckMapCollision(player.entity.x, player.entity.bottom + kJumpValues[player.jumpFactor] - 1) && !CheckMapCollision(player.entity.right - 1, player.entity.bottom + kJumpValues[player.jumpFactor] - 1)) {
		    player.entity.y += kJumpValues[player.jumpFactor];
			player.entity.bottom += kJumpValues[player.jumpFactor];

			if (player.jumpFactor > 0) {
				player.jumpFactor--;
			}
		} else {
			player.state = GROUNDED;
			player.jumpFactor = 0;

			// Move until it touches.
			while (!CheckMapCollision(player.entity.x, player.entity.bottom) && !CheckMapCollision(player.entity.right - 1, player.entity.bottom)) {
				player.entity.y += 1;
				player.entity.bottom += 1;
			}
		}
	} else if (player.state == GROUNDED) {
		if (!CheckMapCollision(player.entity.x, player.entity.bottom) && !CheckMapCollision(player.entity.right - 1, player.entity.bottom)) {
			player.state = FALLING;
			player.jumpFactor = JUMP_SIZE - 5;
		}
	}

	// Animations.
	if (player.state == GROUNDED) {
		if (player.rightSpeed > 0) {
			SetAnimationFrame(&player.entity, kPlayerMoveRight, 1);
		} else if (player.leftSpeed > 0) {
			SetAnimationFrame(&player.entity, kPlayerMoveLeft, 1);
		} else {
			if (player.lastSpeedIsRight) {
				SetAnimationFrame(&player.entity, kPlayerIdleRight, 1);
			} else {
				SetAnimationFrame(&player.entity, kPlayerIdleLeft, 1);
			}
		}
	} else if (player.state == JUMPING) {
		if (player.rightSpeed > 0) {
			SetAnimationFrame(&player.entity, kPlayerJumpRight, 1);
		} else if (player.leftSpeed > 0) {
			SetAnimationFrame(&player.entity, kPlayerJumpLeft, 1);
		} else {
			if (player.lastSpeedIsRight) {
				SetAnimationFrame(&player.entity, kPlayerJumpRight, 1);
			} else {
				SetAnimationFrame(&player.entity, kPlayerJumpLeft, 1);
			}
		}
	} else if (player.state == FALLING) {
		if (player.rightSpeed > 0) {
			SetAnimationFrame(&player.entity, kPlayerFallRight, 1);
		} else if (player.leftSpeed > 0) {
			SetAnimationFrame(&player.entity, kPlayerFallLeft, 1);
		} else {
			if (player.lastSpeedIsRight) {
				SetAnimationFrame(&player.entity, kPlayerFallRight, 1);
			} else {
				SetAnimationFrame(&player.entity, kPlayerFallLeft, 1);
			}
		}
	}
}

u8 CheckMapCollision(u8 x, u8 y) {
	x = x >> 2;
	y = y >> 4;

	// Bounds.
	if (x == 0xFF) {
		return 1;
	}

	if (y == 0xFF) {
		return 1;
	}

	if (x >= MAP_WIDTH) {
		return 1;
	}

	if (y >= MAP_HEIGHT) {
		return 1;
	}

	// Special tiles.
	if (map[x][y] == MAP_END) {
		winCount = WIN_TIME;	// Wait and reset.
	} else if (map[x][y] == MAP_LASER_HORIZONTAL || map[x][y] == MAP_LASER_VERTICAL) {
		deadCount = DEATH_TIME;	// Wait and reset.
	} else if (map[x][y] == MAP_DEFORM_BLOCK) {
		ActivateDeformBlock(x, y);
	}

	return map[x][y];
}

void ChangeMap() {
	currentMap++;

	if (currentMap < NUM_MAPS) {
		LoadMap(currentMap);
	}
}

void LoadMap(u8 m) {
	u8 i = 0, j = 0, leftPixel = 0, rightPixel = 0;

	cpct_clearScreen_f64(0);
	block.active = 0;
	numDeformBlocks = 0;

	// Reset deform blocks.
	for (i = 0; i < MAX_DEFORM_BLOCKS; i++) {
		deformBlocks[i].time = 0;
	}

	// Each byte contains to tiles.
	for (i = 0; i < MAP_HEIGHT; i++) {
		for (j = 0; j < MAP_HALF_WIDTH; j++) {
			leftPixel = GetLeftPixelValue((u8)kMaps[m][i][j]);
			rightPixel = GetRightPixelValue((u8)kMaps[m][i][j]);

			map[j * 2][i] = leftPixel;
			map[j * 2 + 1][i] = rightPixel;
		}
	}

	// Draw tiles.
	for (i = 0; i < MAP_WIDTH; i++) {
		for (j = 0; j < MAP_HEIGHT; j++) {
			if (map[i][j] > 0) {
				if (map[i][j] < 14) {
					cpct_drawTileAligned4x8_f(GetTile(map[i][j]), CPCT_VMEM_START + i * 4 + SCR_WIDTH * j * 2);
					cpct_drawTileAligned4x8_f(GetTile(map[i][j]) + 32, CPCT_VMEM_START + i * 4 + SCR_WIDTH * j * 2 + SCR_WIDTH);

					if (map[i][j] == MAP_DEFORM_BLOCK) {
						if (numDeformBlocks < MAX_DEFORM_BLOCKS) {
							deformBlocks[numDeformBlocks].x = i;
							deformBlocks[numDeformBlocks].y = j;
							deformBlocks[numDeformBlocks].time = 0;
							numDeformBlocks++;
						}
					}
				} else if (map[i][j] == MAP_MOVING_BLOCK) {
					block.active = 1;
					block.startX = i * TILE_WIDTH;
					block.startY = j * TILE_HEIGHT;
					block.xSpeed = 0;
					block.ySpeed = 0;
					map[i][j] = MAP_VOID;	// No collide.
					ResetBlock();
				} else if (map[i][j] == MAP_PLAYER_START) {
					player.startX = i * TILE_WIDTH;
					player.startY = j * TILE_HEIGHT;
					map[i][j] = MAP_VOID;	// No collide.
				}

			}
		}
	}

	// Check block neighbors to set its speed.
	if (!CheckMapCollisionBasic(block.entity.right, block.entity.y)) {
		block.xSpeed = 1;
	}
	if (!CheckMapCollisionBasic(block.entity.x - 1, block.entity.y)) {
		block.xSpeed = -1;
	}
	if (!CheckMapCollisionBasic(block.entity.x, block.entity.y - 5)) {
		block.ySpeed = -4;
	}
	if (!CheckMapCollisionBasic(block.entity.x, block.entity.bottom + 4)) {
		block.ySpeed = 4;
	}

	DrawMapProgressBar();
	DrawHUDBackground();
	DrawTotalDefeats();
	DrawTotalTime(1);

	ResetPlayer();
}

u8* GetTile(u8 i) {
	switch (i) {
		case MAP_BASIC_01: 			return kTileBasic01;
		case MAP_END: 				return kTileEnd;
		case MAP_LASER_VERTICAL: 	return kTileLaserVertical;
		case MAP_LASER_HORIZONTAL: 	return kTileLaserHorizontal;
		case MAP_DEFORM_BLOCK: 		return kTileDeformBlock01;
		default: return kTileBasic01;
	}
}

void DrawMapProgressBar() {
	u8 i = 0;
	u8 *pscr;

	for (i = 0; i < SCR_WIDTH; i++) {
		pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 8);
		cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_BLACK, COLOR_RED), 1, 1);
		pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 7);
		cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_RED, COLOR_BLACK), 1, 1);

		if (i == 0 || i == SCR_WIDTH - 1) {
			pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 6);
			cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_BLACK, COLOR_RED), 1, 1);
			pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 5);
			cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_RED, COLOR_BLACK), 1, 1);
			pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 4);
			cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_BLACK, COLOR_RED), 1, 1);
			pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 3);
			cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_RED, COLOR_BLACK), 1, 1);
		} else {
			if (i <= currentMap + 1) {
				pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 6);
				cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_BRIGHT_YELLOW, COLOR_BLACK), 1, 1);
				pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 5);
				cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_BRIGHT_YELLOW, COLOR_BLACK), 1, 1);
				pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 4);
				cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_ORANGE, COLOR_BLACK), 1, 1);
				pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 3);
				cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_BRIGHT_RED, COLOR_BLACK), 1, 1);
			}
		}

		pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 2);
		cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_BLACK, COLOR_RED), 1, 1);
		pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 1);
		cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_RED, COLOR_BLACK), 1, 1);
	}

}

void UpdateMapProgressBar() {
	u8 *pscr;

	if (currentMap > 0 && currentMap < SCR_WIDTH - 1) {
		pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, SCR_HEIGHT - 6);
		cpct_drawSolidBox(pscr + currentMap, cpct_px2byteM0(COLOR_BRIGHT_GREEN, COLOR_BLACK), 1, 4);
	}
}

u8 CheckDeath() {
	if (deadCount > 0) {
		if (deadCount == DEATH_TIME) {
			cpct_akp_SFXPlay(2, 13, 35, 0, 0, AY_CHANNEL_B);
			totalDefeats++;
		}

		deadCount--;
		winCount = 0;	// Death is stronger than winning.

		if (deadCount == 0) {
			ResetPlayer();
			if (block.active) {
				EraseEntity(&block.entity);
				ResetBlock();
			}
			ResetDeformBlocks();
			DrawTotalDefeats();
		}

		return 1;
	} else {
		return 0;
	}
}

u8 CheckWin() {
	if (deadCount > 0) {	// Death is stronger than winning.
		return 0;
	}

	if (winCount > 0) {
		if (winCount == WIN_TIME) {
			cpct_akp_SFXPlay(3, 13, 50, 4, 0, AY_CHANNEL_B);
		}

		winCount--;

		if (winCount == 0) {
			ChangeMap();
		}

		return 1;
	} else {
		return 0;
	}
}

void UpdateBlock(u8 *ticks) {
	block.entity.prevX = block.entity.x;
	block.entity.prevY = block.entity.y;

	if ((*ticks & 1) == 0) {
		if (block.xSpeed > 0) {
			if (!CheckMapCollisionBasic(block.entity.right, block.entity.y)
				&& !CheckMapCollisionBasic(block.entity.right, block.entity.bottom - 1)) {
				block.entity.x += block.xSpeed;
				block.entity.right += block.xSpeed;
			} else {
				block.xSpeed *= -1;
			}
		} else if (block.xSpeed < 0) {
			if (!CheckMapCollisionBasic(block.entity.x - 1, block.entity.y)
				&& !CheckMapCollisionBasic(block.entity.x - 1, block.entity.bottom - 1)) {
				block.entity.x += block.xSpeed;
				block.entity.right += block.xSpeed;
			} else {
				block.xSpeed *= -1;
			}
		}

		if (block.ySpeed < 0) {
			if (!CheckMapCollisionBasic(block.entity.x, block.entity.y - 1 + block.ySpeed)
				&& !CheckMapCollisionBasic(block.entity.right - 1, block.entity.y - 1 + block.ySpeed)) {
				block.entity.y += block.ySpeed;
				block.entity.bottom += block.ySpeed;
			} else {
				block.ySpeed *= -1;
			}
		} else if (block.ySpeed > 0) {
			if (!CheckMapCollisionBasic(block.entity.x, block.entity.bottom + block.ySpeed)
				&& !CheckMapCollisionBasic(block.entity.right - 1, block.entity.bottom + block.ySpeed)) {
				block.entity.y += block.ySpeed;
				block.entity.bottom += block.ySpeed;
			} else {
				block.ySpeed *= -1;
			}
		}
	}

	if (CheckEntitiesCollisionBB(&player.entity, &block.entity)) {
		deadCount = DEATH_TIME;
	}
}

void ResetBlock() {
	block.entity.x = block.startX;
	block.entity.y = block.startY;
	block.entity.prevX = block.entity.x;
	block.entity.prevY = block.entity.y;
	block.entity.right = block.entity.x + TILE_WIDTH;
	block.entity.bottom = block.entity.y + TILE_HEIGHT;
	SetAnimationFrame(&block.entity, kBlock, 1);
}

u8 CheckMapCollisionBasic(u8 x, u8 y) {
	x = x >> 2;
	y = y >> 4;

	return map[x][y];
}

void ActivateDeformBlock(u8 x, u8 y) {
	u8 i = 0;

	for (i = 0; i < numDeformBlocks; i++) {
		if (deformBlocks[i].x == x && deformBlocks[i].y == y) {
			if (deformBlocks[i].time == 0) {
				cpct_akp_SFXPlay(4, 13, 40, 0, 0, AY_CHANNEL_B);
				deformBlocks[i].time = DEFORM_BLOCK_TIME;
				cpct_drawTileAligned4x8_f(kTileDeformBlock02, CPCT_VMEM_START + x * 4 + SCR_WIDTH * y * 2);
				cpct_drawTileAligned4x8_f(kTileDeformBlock02 + 32, CPCT_VMEM_START + x * 4 + SCR_WIDTH * y * 2 + SCR_WIDTH);
			}
		}
	}
}

void UpdateDeformBlocks() {
	u8 i = 0;

	for (i = 0; i < numDeformBlocks; i++) {
		if (deformBlocks[i].time > 0) {
			deformBlocks[i].time--;

			if (deformBlocks[i].time == 0) {
				cpct_akp_SFXPlay(5, 13, 56, 0, 0, AY_CHANNEL_B);
				cpct_drawSolidBox(CPCT_VMEM_START + deformBlocks[i].x * 4 + SCR_WIDTH * deformBlocks[i].y * 2, cpct_px2byteM0(COLOR_BLACK, COLOR_BLACK), TILE_WIDTH, TILE_HEIGHT);
				map[deformBlocks[i].x][deformBlocks[i].y] = MAP_VOID;
			}
		}
	}
}

void ResetDeformBlocks() {
	u8 i = 0;

	for (i = 0; i < numDeformBlocks; i++) {
		cpct_drawTileAligned4x8_f(kTileDeformBlock01, CPCT_VMEM_START + deformBlocks[i].x * 4 + SCR_WIDTH * deformBlocks[i].y * 2);
		cpct_drawTileAligned4x8_f(kTileDeformBlock01 + 32, CPCT_VMEM_START + deformBlocks[i].x * 4 + SCR_WIDTH * deformBlocks[i].y * 2 + SCR_WIDTH);
		map[deformBlocks[i].x][deformBlocks[i].y] = MAP_DEFORM_BLOCK;
		deformBlocks[i].time = 0;
	}
}

void DrawHUDBackground() {
	u8 i = 0;
	u8 *pscr;

	pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, 0);
	cpct_drawSolidBox(pscr, cpct_px2byteM0(COLOR_BLACK, COLOR_BLACK), 29, 9);

	pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, 0);
	cpct_drawTileAligned2x8_f(kDefeatsIcon, pscr);

	for (i = 0; i < 14; i++) {
		pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, 9);
		cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_RED, COLOR_RED), 1, 1);
	}

	pscr = cpct_getScreenPtr(CPCT_VMEM_START, 14, 0);
	cpct_drawTileAligned2x8_f(kTimeIcon, pscr);

	for (i = 14; i < 30; i++) {
		pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, 9);
		cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_RED, COLOR_RED), 1, 1);

		if (i == 29) {
			pscr = cpct_getScreenPtr(CPCT_VMEM_START, 0, 0);
			cpct_drawSolidBox(pscr + i, cpct_px2byteM0(COLOR_BLACK, COLOR_RED), 1, 9);
		}
	}
}

void DrawTotalDefeats() {
	DrawComposedNumber(totalDefeats, 11, 0, 1);
}

void UpdateTotalTime() {
	timeCount--;

	if (timeCount == 0) {
		timeCount = 50;
		seconds++;

		if (seconds == 60) {
			seconds = 0;
			minutes++;
		}
	}
}

void DrawTotalTime(u8 forceDraw) {
	if (timeCount == 50 || forceDraw) {
		u8 digits = 0;
		u8 *pscr;
		u8 x = 27;

		if (seconds < 10) {
			digits = DrawComposedNumber(seconds, x, 0, 1);
			x = x - 2;

			pscr = cpct_getScreenPtr(CPCT_VMEM_START, x, 0);
			DrawNumber(0, pscr);
		} else {
			digits = DrawComposedNumber(seconds, x, 0, 1);
			x = x - 2;
		}

		x = x - 2;
		pscr = cpct_getScreenPtr(CPCT_VMEM_START, x, 0);
		DrawDots(pscr);

		x = x - 2;
		digits = DrawComposedNumber(minutes, x, 0, 1);
	}
}
