#ifndef VERBATIM_COMPILED_MESH_HPP
#define VERBATIM_COMPILED_MESH_HPP

#include "verbatim_cap.hpp"
#include "verbatim_edge.hpp"
#include "verbatim_face.hpp"
#include "verbatim_mesh.hpp"

/// Compiled mesh.
///
/// Generated by precalculation from logical mesh.
class CompiledMesh
{
  private:
    /// Vertex array.
    seq<Vertex> m_vertices;

    /// Face array.
    seq<Face> m_faces;

    /// Edge array.
    seq<Edge> m_edges;

    /// Cap faces.
    seq<Cap> m_caps;

    /// Number of distinct object ids in this compiled mesh.
    unsigned m_block_count;

  public:
    /// Constructor
    ///
    /// \param object_count Object count for vertices within this mesh.
    CompiledMesh(unsigned object_count) :
      m_block_count(object_count) { }

  private:
    /// Append caps to geometry buffer.
    ///
    /// \param buffer Target buffer.
    /// \param object_id Object id to use.
    /// \param exists Add real or unreal edges?
    void appendCaps(GeometryBuffer &buffer, unsigned object_id, bool exists) const
    {
      for(const Cap &vv : m_caps)
      {
        if((vv.getBlockId() == object_id) && (vv.isReal() == exists))
        {
          buffer.addCap(vv.getCorner(0), vv.getCorner(1), vv.getCorner(2), vv.getNormal());
        }
      }
    }

    /// Append edges to geometry buffer.
    ///
    /// \param buffer Target buffer.
    /// \param object_id Object id to use.
    /// \param exists Add real or unreal edges?
    void appendEdges(GeometryBuffer &buffer, unsigned object_id, bool exists) const
    {
      for(const Edge &vv : m_edges)
      {
        if((vv.getBlockId() == object_id) && (vv.isReal() == exists))
        {
          buffer.addEdge(vv);
        }
      }
    }

  public:
    /// Add cap.
    ///
    /// \param c1 First corner.
    /// \param c2 Second corner.
    /// \param c3 Third corner.
    /// \param nor Face normal.
    /// \param elem Element settings.
    void addCap(const vec3 &c1, const vec3 &c2, const vec3 &c3, const vec3 &nor, const Element &elem)
    {
      m_caps.emplace_back(c1, c2, c3, nor, elem);
    }

    /// Add edge.
    ///
    /// \param c1 First corner.
    /// \param c2 Second corner.
    /// \param n1 First normal.
    /// \param n2 Second normal.
    /// \param elem Element settings.
    void addEdge(const vec3 &c1, const vec3 &c2, const vec3 &n1, const vec3 &n2, const Element &elem)
    {
      m_edges.emplace_back(c1, c2, n1, n2, elem);
    }

    /// Add face.
    ///
    /// \param c1 First corner.
    /// \param c2 Second corner.
    /// \param c3 Third corner.
    /// \param nor Face normal.
    /// \param elem Element settings.
    void addFace(unsigned c1, unsigned c2, unsigned c3, const vec3 &nor, const Element &elem)
    {
      m_faces.emplace_back(c1, c2, c3, nor, elem);
    }

    /// Add vertex.
    ///
    /// \param pos Position.
    /// \param texcoord_offset Texture coordinate offset.
    /// \param texcoord_direction Texture coordinate direction.
    /// \param nor Normal
    /// \param col Color.
    void addVertex(const vec3 &pos, const vec2 &texcoord_offset, const vec2 &texcoord_direction,
        const vec3 &nor, const uvec4 &col, const uvec4 &wt, const uvec4 &rf)
    {
      m_vertices.emplace_back(pos, texcoord_offset, texcoord_direction, nor, col, wt, rf);
    }

    /// Accessor.
    ///
    /// \return Face count.
    unsigned getFaceCount() const
    {
      return m_faces.size();
    }

    /// Accessor.
    ///
    /// \return Vertex count.
    unsigned getVertexCount() const
    {
      return m_vertices.size();
    }

    /// Insert object into a vertex buffer.
    ///
    /// \param op Geometry buffer to insert to.
    /// \return Compiled mesh as result of insertion.
    MeshUptr insert(GeometryBuffer &op) const
    {
      if(!op.fitsVertices(m_vertices.size()) || !op.fitsEdgeVertices(m_caps.size() * 3))
      {
        return MeshUptr();
      }
      MeshUptr ret(new Mesh(op));

      // Vertices first.
      unsigned vertex_base = op.getVertexCount();
      for(const Vertex &vv : m_vertices)
      {
        op.addVertex(vv);
      }

      // Add objects, one object at a time.
      for(unsigned ii = 0; (m_block_count > ii); ++ii)
      {
        unsigned face_base = op.getIndexCount();
        unsigned face_count = 0;
        unsigned face_count_full = 0;

        // Real faces.
        for(const Face &vv : m_faces)
        {
          if(vv.getBlockId() == ii)
          {
            face_count_full += 3;
            if(vv.isReal())
            {
              face_count += 3;
#if defined(USE_LD)
              if(face_count != face_count_full)
              {
                std::ostringstream sstr;
                sstr << "face order problem for object " << ii;
                BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
              }
#endif
            }
            op.addIndex(vv.getConvertedIndex(vertex_base, 0));
            op.addIndex(vv.getConvertedIndex(vertex_base, 1));
            op.addIndex(vv.getConvertedIndex(vertex_base, 2));
          }
        }

        unsigned edge_base = op.getEdgeIndexCount();
        // Real edges.
        appendEdges(op, ii, true);
        unsigned edge_count = op.getEdgeIndexCount() - edge_base;
        // Non-real edges.
        appendEdges(op, ii, false);
        unsigned edge_count_full = op.getEdgeIndexCount() - edge_base;

        unsigned cap_base = op.getEdgeIndexCount();
        // Real caps.
        appendCaps(op, ii, true);
        unsigned cap_count = op.getEdgeIndexCount() - cap_base;
        // Non-real caps.
        appendCaps(op, ii, false);
        unsigned cap_count_full = op.getEdgeIndexCount() - cap_base;
  
        // Append to mesh.
        if(0 < face_count_full)
        {
          IndexRun face_run(face_base, face_count, face_count_full);
          IndexRun edge_run(edge_base, edge_count, edge_count_full);
          IndexRun cap_run(cap_base, cap_count, cap_count_full);
          ret->addIndexBlock(face_run, edge_run, cap_run);
        }
#if defined(USE_LD)
        else if(0 < edge_count_full)
        {
          std::ostringstream sstr;
          sstr << "detected edge block of size " << edge_count_full << " with no corresponding faces";
          BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
        }
#endif
      }

      return ret;
    }

    /// Optimize a compiled mesh.
    ///
    /// Removes redundant vertices and faces.
    void optimize()
    {
#if defined(USE_LD)
      // Not going to test illegal faces in release build.
      for(unsigned ii = 0; (m_faces.size() > ii);)
      {
        Face &vv = m_faces[ii];

        if(!vv.isValid())
        {
          std::cerr << "WARNING: invalid face: " << vv << std::endl;
          if(m_faces.size() > ii + 1)
          {
            m_faces[ii] = m_faces.back();
            m_faces.pop_back();
          }
        }
        else
        {
          ++ii;
        }
      }
#endif

      dnload_qsort(m_faces.getData(), m_faces.size(), sizeof(Face), Face::qsort_cmp_face);
      dnload_qsort(m_edges.getData(), m_edges.size(), sizeof(Edge), Edge::qsort_cmp_edge);
      dnload_qsort(m_caps.getData(), m_caps.size(), sizeof(Cap), Cap::qsort_cmp_cap);
    }

#if defined(USE_LD)
  public:
    /// Output to stream.
    ///
    /// \param ostr Output stream.
    /// \return Output stream.
    std::ostream& put(std::ostream &ostr) const
    {
      return ostr << "CompiledMesh(" << this << ")";
    }
#endif
};

/// Smart pointer type.
typedef uptr<CompiledMesh> CompiledMeshUptr;

#if defined(USE_LD)
/// Stream output operator.
///
/// \param lhs Left-hand-side operand.
/// \param rhs Right-hand-side operand.
/// \return Output stream.
inline std::ostream& operator<<(std::ostream &lhs, const CompiledMesh &rhs)
{
  return rhs.put(lhs);
}
#endif

#endif
