/*****************************************************************************/
/*                             Application description                       */
/*****************************************************************************/
/*                                                                           */
/*  The kart AI operates in multiple passes as such:                         */
/*  -Receive and parse map outline into a checkpoint list                    */
/*  -Use presets to adjust checkpoint list according to track layout         */
/*  -Analyze frame differences to interpolate missing data                   */
/*  -Brute force in realtime a set of predefined simulated paths             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                             Common libs                                   */
/*****************************************************************************/

#include <string>
#include <cstring>
#include <vector>

/*****************************************************************************/
/*                             Logging                                       */
/*****************************************************************************/

#include <iostream>

#define Log(msg) InternalLog(__FILE__, __LINE__, msg)
void InternalLog(const char* filename, const int line, std::string input)
{
    std::cout << filename << ":" << line << ": " << input << "\n";
}
void InternalLog(const char* filename, const int line, const char* input)
{
    std::cout << filename << ":" << line << ": " << input << "\n";
}

/*****************************************************************************/
/*                             String utilities                              */
/*****************************************************************************/

#include <sstream>

template <class T>
std::string ToString(const T value)
{
    std::ostringstream oss;
    oss << value;
    return oss.str();
}

template <class T>
T ToValue(const std::string str)
{
    T value;
    std::istringstream iss(str);
    iss >> value;
    return value;
}

std::vector<std::string> Split(const std::string& value, const char delimiter)
{
    std::vector<std::string> result;
    std::stringstream ss(value);
    std::string split;
    while (std::getline(ss, split, delimiter))
        result.push_back(split);
    return result;
}

/*****************************************************************************/
/*                             Timer utility                                 */
/*****************************************************************************/

#include <chrono>
struct Clock
{
    typedef std::chrono::time_point<std::chrono::high_resolution_clock> Timestamp;
    Timestamp start;
    Clock()
    {
        Restart();
    }
    void Restart()
    {
        start = Now();
    }
    Timestamp Now()
    {
        return std::chrono::high_resolution_clock::now();
    }
    int GetTime()
    {
        std::chrono::duration<double, std::micro> elapsed = Now() - start;
        return (int)elapsed.count();
    }
};

/*****************************************************************************/
/*                             Mathematics                                   */
/*****************************************************************************/

#include <cmath>

#define PI 3.14159265359f

template <class T>
struct vec2
{
    T x, y;
    vec2(){}
    vec2(T x, T y) : x(x), y(y) {}
    vec2 operator +(const vec2& r) const {return vec2(x+r.x, y+r.y);}
    vec2 operator -(const vec2& r) const {return vec2(x-r.x, y-r.y);}
    vec2 operator *(const vec2& r) const {return vec2(x*r.x, y*r.y);}
    vec2 operator /(const vec2& r) const {return vec2(x/r.x, y/r.y);}
    vec2 operator +(const T& r) const {return vec2(x+r, y+r);}
    vec2 operator -(const T& r) const {return vec2(x-r, y-r);}
    vec2 operator *(const T& r) const {return vec2(x*r, y*r);}
    vec2 operator /(const T& r) const {return vec2(x/r, y/r);}
    vec2& operator +=(const vec2& r) {x+=r.x; y+=r.y; return *this;}
    vec2& operator -=(const vec2& r) {x-=r.x; y-=r.y; return *this;}
    vec2& operator *=(const vec2& r) {x*=r.x; y*=r.y; return *this;}
    vec2& operator /=(const vec2& r) {x/=r.x; y/=r.y; return *this;}
    vec2& operator +=(const T& r) {x+=r; y+=r; return *this;}
    vec2& operator -=(const T& r) {x-=r; y-=r; return *this;}
    vec2& operator *=(const T& r) {x*=r; y*=r; return *this;}
    vec2& operator /=(const T& r) {x/=r; y/=r; return *this;}
    T LengthSquared() const {return x*x+y*y;}
    T Length() const {return sqrt(LengthSquared());}
    vec2 Normalized() const {return *this / Length();}
    T Dot(const vec2& r) const {return x*r.x+y*r.y;}
    T AngleBetween(const vec2& r) const {return acos(Dot(r)/(Length()*r.Length()));}
    T ToAngle() const {return atan2(y, x);}
    vec2 Rotate(T angle)
    {
        float c = cos(angle);
        float s = sin(angle);
        return vec2(
            x*c-y*s,
            x*s+y*c
        );
    }
    T Cross(const vec2& a, const vec2& b)
    {
        return
            (b.x-a.x)*(y-a.y) -
            (b.y-a.y)*(x-a.x);
    }
    template <class R>
    vec2<R> Conv() const {return vec2<R>(x,y);}
    std::string Str() const {return ToString(x) + " - " + ToString(y);}
};
typedef vec2<float> vec2f;
typedef vec2<int> vec2i;

//in a segment parallel to origin-end, how far along are said point
float LineProgress(vec2f origin, vec2f end, vec2f point)
{
    return
        (
            (point.x - origin.x)*(end.x - origin.x) +
            (point.y - origin.y)*(end.y - origin.y)
        ) / (
            (end.x   - origin.x)*(end.x - origin.x) +
            (end.y   - origin.y)*(end.y - origin.y)
        );
}

bool Collide(vec2f upperleft, vec2f size, vec2f point)
{
    if (point.x <= upperleft.x) return false;
    if (point.y <= upperleft.y) return false;
    if (point.x >= upperleft.x + size.x) return false; 
    if (point.y >= upperleft.y + size.y) return false; 
    return true;
}

//modulus operator on radian angles
float AngleMod(float a, float b)
{
    float diff = b - a;
    float modd = fmod(fabs(diff)+PI, PI*2.0f) - PI;
    return (diff < 0.0f) ? modd : -modd;
}

/*****************************************************************************/
/*                             Networking                                    */
/*****************************************************************************/

#ifdef PLEB
#include <winsock2.h>
#else
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#define closesocket close
#endif

class Network
{
private:
    int tcp;
    std::string tcpbuffer;
public:
    void Create(unsigned int ip, unsigned short port)
    {
        //setup
#ifdef PLEB
        WSADATA wsadata;
        WSAStartup(MAKEWORD(2,2), &wsadata);
#endif
        Log("Network started");
        
        //target
        struct sockaddr_in addr;
        addr.sin_family = AF_INET;
        addr.sin_port = htons(port);
        addr.sin_addr.s_addr = htonl(ip);
        
        //socket
        tcpbuffer = "";
        tcp = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
        int result = connect(tcp, (struct sockaddr*)&addr, sizeof(addr));
        if (result < 0)
        {
            Log("Connect error");
            std::cout << errno  << "\n";
            }
        else
            Log("Connected");
    }
    void Destroy()
    {
        //cleanup
        closesocket(tcp);
#ifdef PLEB
        WSACleanup();
#endif
        Log("Network stopped");
    }
    void Send(std::string msg)
    {
        //Log("Sending " + msg);
        msg = msg + "\n";
        send(tcp, msg.data(), msg.size(), 0);
    }
    std::string Receive()
    {
        const int buffersize = 4096;
        char buffer[buffersize];
        while (true)
        {
            const unsigned int newline = tcpbuffer.find('\n');
            if (newline > 0 &&
                newline <= 1024*1024*1024)
            {
                const std::string line = tcpbuffer.substr(0, newline);
                tcpbuffer = tcpbuffer.substr(newline + 1);
                return line;
            }
            
            int numbytes = recv(tcp, buffer, buffersize, 0);
            if (numbytes <= 0)
            {
                Log("Disconnected");
                return "DISCONNECTED";
            }
            
            const std::string received(buffer, numbytes);
            tcpbuffer += received;
        }
    }
};

unsigned int ParseIP(const char* input)
{
    std::string str(input);
    std::vector<std::string> split = Split(str, '.');
    if (split.size() != 4)
        return 0x7F000001;
    unsigned int ip = 0;
    ip |= ToValue<unsigned int>(split[0]) << 8*3;
    ip |= ToValue<unsigned int>(split[1]) << 8*2;
    ip |= ToValue<unsigned int>(split[2]) << 8*1;
    ip |= ToValue<unsigned int>(split[3]) << 8*0;
    return ip;
};

/*****************************************************************************/
/*                             Types                                         */
/*****************************************************************************/

static const int PowerupSize = 32;
enum Powerup {
    PowerupNone = 0,
    PowerupBanana,
    PowerupGreenShell,
    PowerupRedShell,
    PowerupBlueShell,
    PowerupOil,
    PowerupMushroom,
    PowerupBigMushroom,
    PowerupLightning,
    PowerupStar,
};

const char* PowerupMap[] = {
    "none",
    "banana",
    "greenshell",
    "redshell",
    "blueshell",
    "oil",
    "mushroom",
    "bigmushroom",
    "lightning",
    "star",
};

enum ShellType {
    ShellGreen,
    ShellRed,
    ShellBlue,
};

const char* ShellMap[] = {
    "green",
    "red",
    "blue",
};

enum Area {
    AreaTrack = 0,
    AreaGrass,
    AreaWall,
    AreaIce,
    AreaOil,
    AreaBoost,
    AreaStart,
    AreaMud,
    AreaWater,
    AreaBanana,
};

const char* AreaMap[] = {
    "track",
    "grass",
    "wall",
    "ice",
    "oil",
    "booster",
    "startarea",
    "mud",
    "water",
    "banana"
};

enum Tile {
    TileHorizonal = 0,
    TileVertical,
    TileUpperleft,
    TileUpperright,
    TileBottomleft,
    TileBottomright,
    TileCross,
    TileNone,
};

const char TileMap[] = {
    '-',
    '|',
    '/',
    '`',
    '\\',
    ',',
    '+',
    '.'
};

enum Action {
    ActionAccel = (1<<0),
    ActionBreak = (1<<1),
    ActionTurnl = (1<<2),
    ActionTurnr = (1<<3),
    ActionDrift = (1<<4),
    ActionPower = (1<<5),
};

void PrintAction(int action)
{
    std::cout << ((action & ActionAccel) ? "A" : "-");
    std::cout << ((action & ActionBreak) ? "B" : "-");
    std::cout << ((action & ActionTurnl) ? "L" : "-");
    std::cout << ((action & ActionTurnr) ? "R" : "-");
    std::cout << ((action & ActionDrift) ? "D" : "-");
    std::cout << ((action & ActionPower) ? "P" : "-");
    std::cout << "\n";
}

//inverses turning and accel when driving backwards
int InverseAction(int action, bool inverse)
{
    int out = action;
    if (inverse)
    {
        if (action & ActionAccel)
        {
            out ^= ActionAccel;
            out ^= ActionBreak;
        }
        if (action & ActionBreak)
        {
            out ^= ActionAccel;
            out ^= ActionBreak;
        }
    }
    return out;
}

/*****************************************************************************/
/*                             Type converters                               */
/*****************************************************************************/

Powerup ParsePowerup(const char* str)
{
    for (int i = 0; i < sizeof(PowerupMap); ++i)
        if (strcmp(str, PowerupMap[i]) == 0)
            return (Powerup)i;
    return PowerupNone;
}

ShellType ParseShell(const char* str)
{
    for (int i = 0; i < sizeof(ShellMap); ++i)
        if (strcmp(str, ShellMap[i]) == 0)
            return (ShellType)i;
    return ShellGreen;
}

Area ParseArea(const char* str)
{
    for (int i = 0; i < sizeof(AreaMap); ++i)
        if (strcmp(str, AreaMap[i]) == 0)
            return (Area)i;
    return AreaGrass;
}

Tile ParseTile(const char* str)
{
    for (int i = 0; i < sizeof(TileMap); ++i)
        if (str[0] == TileMap[i])
            return (Tile)i;
    return TileNone;
}

/*****************************************************************************/
/*                             Data structures                               */
/*****************************************************************************/

struct Modifier
{
    Area type;
    vec2i pos;
    vec2i size;
};

struct Map
{
    vec2i tsize;
    vec2i msize;
    std::vector<Tile> map;
    std::vector<Modifier> mods;
    std::vector<vec2i> path;
    
    Area GetModifier(vec2f position) const
    {
        for (int i = 0; i < mods.size(); ++i)
        {
            if (Collide(mods[i].pos.Conv<float>(), mods[i].size.Conv<float>(), position))
                return mods[i].type;
        }
        return AreaTrack; //none
    }
};

struct Car
{
    int id;
    vec2f dir;
    vec2f vel;
    vec2f pos;
    int drift;
    vec2i size;
    Powerup power;
    int boost_frame;
    int small_frame;
};

struct Shell
{
    ShellType type;
    vec2f pos;
    vec2f vel;
};

struct Box
{
    vec2f pos;
    vec2f size;
};

struct State
{
    std::vector<Car> cars;
    std::vector<Shell> shells;
    std::vector<Box> boxes;
};

/*****************************************************************************/
/*                             JSON Parsing                                  */
/*****************************************************************************/

#include "cJSON.cpp"

Map ParseMap(cJSON* json)
{
    //init struct
    Map map;
    
    //parse size
    map.tsize.x = cJSON_GetObjectItem(json, "tile_width")->valueint;
    map.tsize.y = cJSON_GetObjectItem(json, "tile_height")->valueint;
    
    //parse tiles
    cJSON* tiles = cJSON_GetObjectItem(json, "tiles");
    map.msize.y = cJSON_GetArraySize(tiles);
    for (int y = 0; y < map.msize.y; ++y)
    {
        cJSON* row = cJSON_GetArrayItem(tiles, y);
        map.msize.x = cJSON_GetArraySize(row);
        map.map.resize(map.msize.x*map.msize.y);
        for (int x = 0; x < map.msize.x; ++x)
        {
            cJSON* cell = cJSON_GetArrayItem(row, x);
            char* str = cell->valuestring;
            map.map[x+y*map.msize.x] = ParseTile(str);
        }
    }
    
    //parse modifiers
    cJSON* mods = cJSON_GetObjectItem(json, "modifiers");
    int nummods = cJSON_GetArraySize(mods);
    map.mods.resize(nummods);
    for (int i = 0; i < nummods; ++i)
    {
        cJSON* mod = cJSON_GetArrayItem(mods, i);
        map.mods[i].type    = ParseArea(cJSON_GetObjectItem(mod, "type")->valuestring);
        map.mods[i].pos.x   = cJSON_GetObjectItem(mod, "x")->valueint;
        map.mods[i].pos.y   = cJSON_GetObjectItem(mod, "y")->valueint;
        map.mods[i].size.x  = cJSON_GetObjectItem(mod, "width")->valueint;
        map.mods[i].size.y  = cJSON_GetObjectItem(mod, "height")->valueint;
    }
    
    //parse path
    cJSON* paths = cJSON_GetObjectItem(json, "path");
    int numpaths = cJSON_GetArraySize(paths);
    map.path.resize(numpaths);
    for (int i = 0; i < numpaths; ++i)
    {
        cJSON* path = cJSON_GetArrayItem(paths, i);
        map.path[i].x = cJSON_GetObjectItem(path, "tile_x")->valueint;
        map.path[i].y = cJSON_GetObjectItem(path, "tile_y")->valueint;
    }
    
    //return
    return map;
}

State ParseState(cJSON* json)
{
    //init struct
    State state;
    
    //parse cars
    cJSON* cars = cJSON_GetObjectItem(json, "cars");
    int numcars = cJSON_GetArraySize(cars);
    state.cars.resize(numcars);
    for (int i = 0; i < numcars; ++i)
    {
        cJSON* car = cJSON_GetArrayItem(cars, i);
        state.cars[i].id = cJSON_GetObjectItem(car, "id")->valueint;
        state.cars[i].dir.x = cJSON_GetObjectItem(cJSON_GetObjectItem(car, "direction"), "x")->valuedouble;
        state.cars[i].dir.y = cJSON_GetObjectItem(cJSON_GetObjectItem(car, "direction"), "y")->valuedouble;
        state.cars[i].vel.x = cJSON_GetObjectItem(cJSON_GetObjectItem(car, "velocity"), "x")->valuedouble;
        state.cars[i].vel.y = cJSON_GetObjectItem(cJSON_GetObjectItem(car, "velocity"), "y")->valuedouble;
        state.cars[i].pos.x = cJSON_GetObjectItem(cJSON_GetObjectItem(car, "pos"), "x")->valuedouble;
        state.cars[i].pos.y = cJSON_GetObjectItem(cJSON_GetObjectItem(car, "pos"), "y")->valuedouble;
        state.cars[i].drift = cJSON_GetObjectItem(car, "drift")->valueint;
        state.cars[i].size.x = cJSON_GetObjectItem(car, "width")->valueint;
        state.cars[i].size.y = cJSON_GetObjectItem(car, "height")->valueint;
        state.cars[i].power  = ParsePowerup(cJSON_GetObjectItem(car, "powerup")->valuestring);
        state.cars[i].boost_frame = -9999;
        state.cars[i].small_frame = -9999;
    }
    
    //parse shells
    cJSON* shells = cJSON_GetObjectItem(json, "shells");
    int numshells = cJSON_GetArraySize(shells);
    state.shells.resize(numshells);
    for (int i = 0; i < numshells; ++i)
    {
        cJSON* shell = cJSON_GetArrayItem(shells, i);
        state.shells[i].type = ParseShell(cJSON_GetObjectItem(shell, "type")->valuestring);
        state.shells[i].pos.x = cJSON_GetObjectItem(shell, "x")->valuedouble;
        state.shells[i].pos.y = cJSON_GetObjectItem(shell, "y")->valuedouble;
        state.shells[i].vel.x = cJSON_GetObjectItem(shell, "dx")->valuedouble;
        state.shells[i].vel.y = cJSON_GetObjectItem(shell, "dy")->valuedouble;
    }
    
    //parse boxes
    cJSON* boxes = cJSON_GetObjectItem(json, "boxes");
    int numboxes = cJSON_GetArraySize(boxes);
    state.boxes.resize(numboxes);
    for (int i = 0; i < numboxes; ++i)
    {
        cJSON* box = cJSON_GetArrayItem(boxes, i);
        state.boxes[i].pos.x = cJSON_GetObjectItem(box, "x")->valuedouble;
        state.boxes[i].pos.y = cJSON_GetObjectItem(box, "y")->valuedouble;
        state.boxes[i].size.x = cJSON_GetObjectItem(box, "width")->valuedouble;
        state.boxes[i].size.y = cJSON_GetObjectItem(box, "height")->valuedouble;
    }
    
    //return
    return state;
}

/*****************************************************************************/
/*                             Precalculation                                */
/*****************************************************************************/

//struct to contain precalculated values
struct Precompute
{
    std::vector<vec2i> path;
    std::vector<int> dirarray;
    std::vector<vec2f> checkpoints;
    int carindex;
    int checkpoint;
    int frame;
    vec2f GetCheck(int index) const {return checkpoints[(index+checkpoints.size())%checkpoints.size()];}
};

/*****************************************************************************/
/*                             Track checkpoints                             */
/*****************************************************************************/

/* in/out   Left    Right   Top     Bottom */
/* Left     NULL    HORIZ   TURN    TURN   */
/* Right    HORIZ   NULL    TURN    TURN   */
/* Top      TURN    TURN    NULL    VERT   */
/* Bottom   TURN    TURN    VERT    NULL   */

#define TILE 1.0f
#define HALF TILE/2.0f
#define BORD (16.0f+4.0f)/(128.0f)
#define OFFS HALF/2.0f
#define EDGE (HALF-BORD)*1.0f/4.0f
#define WALL (OFFS-EDGE)
#define NONE 0.0f

//default offsets from center of the track path
const vec2f check_center[] = {
    vec2f( NONE, NONE), vec2f( OFFS, NONE), vec2f(-OFFS,-OFFS), vec2f(-OFFS, OFFS),
    vec2f(-OFFS, NONE), vec2f( NONE, NONE), vec2f( OFFS,-OFFS), vec2f( OFFS, OFFS),
    vec2f(-OFFS,-OFFS), vec2f( OFFS,-OFFS), vec2f( NONE, NONE), vec2f( NONE, OFFS),
    vec2f(-OFFS, OFFS), vec2f( OFFS, OFFS), vec2f( NONE,-OFFS), vec2f( NONE, NONE),
};

//offsets for leaning hints
const vec2f check_side[] = { /* leans right*/
    vec2f( NONE, NONE), vec2f( NONE, EDGE), vec2f( EDGE, EDGE), vec2f(-EDGE, EDGE),
    vec2f( NONE,-EDGE), vec2f( NONE, NONE), vec2f( EDGE,-EDGE), vec2f(-EDGE,-EDGE),
    vec2f(-EDGE,-EDGE), vec2f(-EDGE, EDGE), vec2f( NONE, NONE), vec2f(-EDGE, NONE),
    vec2f( EDGE,-EDGE), vec2f( EDGE, EDGE), vec2f( EDGE, NONE), vec2f( NONE, NONE),
};

//local direction enums use for path analysis
enum Lean {
    LeanW = -9999,
    LeanL = -1,
    LeanN =  0,
    LeanR =  1
};

//conversion between tile types to local directions
const Lean forward_table[] = {
    LeanW, LeanN, LeanL, LeanR,
    LeanN, LeanW, LeanR, LeanL,
    LeanR, LeanL, LeanW, LeanN,
    LeanL, LeanR, LeanN, LeanW,
};

//manual table containing hints about optimal track path
//Order: LT,FW,RT
const Lean lean_table[] = {
    /*LT [LT] */
    LeanL, LeanR, LeanL,
    LeanL, LeanL, LeanL,
    LeanR, LeanL, LeanL,
    
    /*LT [FW]*/
    LeanR, LeanR, LeanL,
    LeanN, LeanN, LeanN,
    LeanR, LeanN, LeanL,
    
    /*LT [RT]*/
    LeanR, LeanR, LeanN,
    LeanR, LeanR, LeanL,
    LeanL, LeanL, LeanL,
    
    
    
    /*FW [LT]*/
    LeanR, LeanR, LeanR,
    LeanL, LeanL, LeanL,
    LeanR, LeanL, LeanL,
    
    /*FW [Fw]*/
    LeanR, LeanR, LeanL,
    LeanR, LeanN, LeanL,
    LeanL, LeanL, LeanR,
    
    /*FW [RT]*/
    LeanR, LeanR, LeanL,
    LeanR, LeanR, LeanR,
    LeanL, LeanL, LeanL,
    
    
    
    /*RT [LT]*/
    LeanR, LeanL, LeanL,
    LeanR, LeanL, LeanL,
    LeanN, LeanL, LeanL,
    
    /*RT [Fw]*/
    LeanR, LeanN, LeanL,
    LeanN, LeanN, LeanN,
    LeanL, LeanL, LeanL,
    
    /*RT [RT]*/
    LeanR, LeanR, LeanR,
    LeanR, LeanR, LeanR,
    LeanL, LeanL, LeanR,
};

//direction enums used for path analysis
enum Dir {
    DirNone = -1,
    DirLeft = 0,
    DirRight,
    DirUp,
    DirDown,
};

//inverse direction value for tile type lookup
Dir Inverse(Dir dir)
{
    if (dir == DirLeft ) return DirRight;
    if (dir == DirRight) return DirLeft;
    if (dir == DirUp   ) return DirDown;
    if (dir == DirDown ) return DirUp;
    return DirNone;
}

struct Exits
{
    bool u,d,l,r;
};

Exits GetExits(Tile tile)
{
    switch (tile)
    {
        case TileHorizonal:     return {0,0,1,1};
        case TileVertical:      return {1,1,0,0};
        case TileUpperleft:     return {0,1,0,1};
        case TileUpperright:    return {0,1,1,0};
        case TileBottomleft:    return {1,0,0,1};
        case TileBottomright:   return {1,0,1,0};
        case TileCross:         return {1,1,1,1};
        case TileNone:          return {0,0,0,0};
        default:                return {0,0,0,0};
    }
    return {0,0,0,0};
}

struct Astar
{
    std::vector<unsigned int> costs;
    std::vector<vec2i> costdirs;
    std::vector<vec2i> sources;
    std::vector<vec2i> process;
    unsigned int Process(const Map& map, vec2i source, vec2i pos, vec2i minus)
    {
        if (pos.x < 0) return 2147483648;
        if (pos.y < 0) return 2147483648;
        if (pos.x >= map.msize.x) return 2147483648;
        if (pos.y >= map.msize.y) return 2147483648;
        int sindex = source.x+source.y*map.msize.x;
        int pindex = pos.x   +pos.y   *map.msize.x;
        
        Exits sexit = GetExits(map.map[sindex]); //hah
        Exits pexit = GetExits(map.map[pindex]);
        
        unsigned int cost = costs[sindex] + 2;
        if (pos.x > source.x)
        {
            if (!sexit.r) cost += 8;
            if (!pexit.l) cost += 8;
        }
        if (pos.x < source.x)
        {
            if (!sexit.l) cost += 8;
            if (!pexit.r) cost += 8;
        }
        if (pos.y < source.y)
        {
            if (!sexit.u) cost += 8;
            if (!pexit.d) cost += 8;
        }
        if (pos.y > source.y)
        {
            if (!sexit.d) cost += 8;
            if (!pexit.u) cost += 8;
        }
        if (!pexit.u &&
            !pexit.d &&
            !pexit.l &&
            !pexit.r)
            cost += 16;
        if (pos.x == minus.x &&
            pos.y == minus.y)
            cost += 1;
        
        
        return cost;
        
        
    }
    void Calculate(const Map& map, vec2i pos, vec2i minus)
    {
        costs.resize(map.msize.x*map.msize.y);
        for (int i = 0; i < costs.size(); ++i)
            costs[i] = 2147483648;
        costdirs.resize(map.msize.x*map.msize.y);
        
        costs[pos.x+pos.y*map.msize.x] = 0;
        
        sources.push_back(pos);
        sources.push_back(pos);
        sources.push_back(pos);
        sources.push_back(pos);
        process.push_back(pos + vec2i(0,-1));
        process.push_back(pos + vec2i(0, 1));
        process.push_back(pos + vec2i(-1,0));
        process.push_back(pos + vec2i( 1,0));
        
        int itr = 0;
        while (itr < process.size())
        {
            unsigned int cost = Process(map, sources[itr], process[itr], minus);
            
            if (cost != 2147483648)
            {
                
                vec2i pos = process[itr];
                int pindex = pos.x   +pos.y   *map.msize.x;
                
                if (costs[pindex] > cost)
                {
                    costs[pindex] = cost;
                    costdirs[pindex] = sources[itr];
                    
                    sources.push_back(pos);
                    sources.push_back(pos);
                    sources.push_back(pos);
                    sources.push_back(pos);
                    process.push_back(pos + vec2i(0,-1));
                    process.push_back(pos + vec2i(0, 1));
                    process.push_back(pos + vec2i(-1,0));
                    process.push_back(pos + vec2i( 1,0));
                }
            }
            ++itr;
        }
    }
    unsigned int GetCost(const Map& map, vec2i pos)
    {
        if (pos.x < 0) return 2147483648;
        if (pos.y < 0) return 2147483648;
        if (pos.x >= map.msize.x) return 2147483648;
        if (pos.y >= map.msize.y) return 2147483648;
        int index = pos.x + pos.y * map.msize.x;
        unsigned int i = costs[index];
        return i;
    }
    std::vector<vec2i> GetPathTo(const Map& map, vec2i pos)
    {
        std::vector<vec2i> rpath;
        vec2i walker = pos;
        while (GetCost(map, walker) > 0)
        {
            rpath.push_back(walker);
            walker = costdirs[walker.x+walker.y*map.msize.x];
            continue;
            
            unsigned int cu = GetCost(map, walker+vec2i(0,-1));
            unsigned int cd = GetCost(map, walker+vec2i(0, 1));
            unsigned int cl = GetCost(map, walker+vec2i(-1,0));
            unsigned int cr = GetCost(map, walker+vec2i( 1,0));
            int li = 0;
            unsigned int lc = cu;
            if (lc > cd) {lc = cd; li = 1;}
            if (lc > cl) {lc = cl; li = 2;}
            if (lc > cr) {lc = cr; li = 3;}
            if (li == 0) walker += vec2i(0,-1);
            if (li == 1) walker += vec2i(0, 1);
            if (li == 2) walker += vec2i(-1,0);
            if (li == 3) walker += vec2i( 1,0);
        }
        return rpath;
    }
};

//convert jump points to path
std::vector<vec2i> GetPath(const Map& map)
{
    std::vector<vec2i> path;
    path.push_back(map.path[0]);
    for (int i = 1; i < map.path.size(); ++i)
    {
        vec2i prevjump = map.path[i-1];
        vec2i currjump = map.path[i-0];
        
        vec2i minus = vec2i(-1,-1);
        if (path.size() > 1)
            minus = path[path.size()-2];
        
        Astar astar;
        astar.Calculate(map, prevjump, minus);
        
        
        
        std::vector<vec2i> rpath = astar.GetPathTo(map, currjump);
        for (int i = rpath.size() - 1; i >= 0; --i)
        {
            path.push_back(rpath[i]);
        }
    }
    
    {
        vec2i minus = vec2i(-1,-1);
        if (path.size() > 1)
            minus = path[path.size()-2];
        
        vec2i prevjump = map.path[map.path.size()-1];
        vec2i currjump = map.path[0];
        Astar astar;
        astar.Calculate(map, prevjump, minus);
        
        
        std::vector<vec2i> rpath = astar.GetPathTo(map, currjump);
        for (int i = rpath.size() - 1; i > 0; --i)
        {
            path.push_back(rpath[i]);
        }
    }
    return path;
}

//convert path coordinates to tile types
std::vector<int> GetDirectionArray(const Precompute& pre, const Map& map)
{
    //size
    const int size = pre.path.size();
    
    //world directions
    std::vector<Dir> directions;
    directions.resize(size);
    for (int i = 0; i < size; ++i)
    {
        Dir dir = DirNone;
        vec2i currpos = pre.path[( i )%size];
        vec2i nextpos = pre.path[(i+1)%size];
        if (nextpos.x < currpos.x) dir = DirLeft;
        if (nextpos.x > currpos.x) dir = DirRight;
        if (nextpos.y < currpos.y) dir = DirUp;
        if (nextpos.y > currpos.y) dir = DirDown;
        directions[i] = dir;
    }
    
    //direction indexes
    std::vector<int> dirarray;
    dirarray.resize(size);
    for (int i = 0; i < size; ++i)
    {
        Dir prev = directions[(i-1+size)%size];
        Dir curr = directions[( i )%size];
        int dirindex = (int)curr + (int)Inverse(prev) * 4;
        dirarray[i] = dirindex;
        if (dirindex < 0) Log("Error");
        if (dirindex >= 16) Log("Error");
        //if (dirindex/4 == dirindex%4) Log("Error");
    }
    
    //return
    return dirarray;
}

//calculate optimal track checkpoints
std::vector<vec2f> CalculateCheckpoints(const Map& map, const Precompute& pre, int version)
{
    //size
    const int size = pre.path.size();
    
    //forward directions
    std::vector<Lean> forwards;
    forwards.resize(size);
    for (int i = 0; i < size; ++i)
        forwards[i] = forward_table[pre.dirarray[i]];
    
    //lean analysis
    std::vector<Lean> leans;
    leans.resize(size);
    for (int i = 0; i < size; ++i)
    {
        Lean prev = forwards[(i-1+size)%size];
        Lean curr = forwards[( i )%size];
        Lean next = forwards[(i+1)%size];
        Lean futu = forwards[(i+2)%size];
        int index =
            ((int)prev+1) * 3 * 3 * 3 +
            ((int)curr+1) * 3 * 3 +
            ((int)next+1) * 3 +
            ((int)futu+1);
        if (index < 0 ||
            index >= 3*3*3*3)
        {
            leans[i] = LeanN;
            //Log("Error");
        }
        else
            leans[i] = lean_table[index];
    }
    
    //scale to map coordinates
    std::vector<vec2f> coords;
    coords.resize(size);
    const vec2f scale = map.tsize.Conv<float>();
    for (int i = 0; i < size; ++i)
    {
        vec2f mappos = pre.path[i].Conv<float>();
        vec2f center = vec2f(0.5f, 0.5f);
        vec2f offset = check_center[pre.dirarray[i]];
        vec2f leanof = check_side[pre.dirarray[i]] * leans[i];
        if (version <= 1) //90degree angles
            coords[i] = (mappos + center) * scale;
        if (version == 2) //more optimal path
            coords[i] = (mappos + center + offset) * scale;
        if (version >= 3) //use hints to optimize path further
            coords[i] = (mappos + center + offset + leanof) * scale;
    }
    
    //return
    return coords;
}

/*****************************************************************************/
/*                             Track collisions                              */
/*****************************************************************************/

//wall collision lines
const vec2f walls[] = {
    //input=left
    vec2f(-HALF, -HALF), vec2f(HALF, HALF), vec2f(-HALF, HALF), vec2f(HALF, -HALF), //invalid
    vec2f(-HALF, BORD-HALF), vec2f( HALF, BORD-HALF), vec2f( HALF, HALF-BORD), vec2f(-HALF, HALF-BORD),
    vec2f(-HALF, BORD-HALF), vec2f(BORD-HALF, -HALF), vec2f(HALF-BORD, -HALF), vec2f(-HALF, HALF-BORD),
    vec2f(-HALF, BORD-HALF), vec2f(HALF-BORD,  HALF), vec2f(BORD-HALF,  HALF), vec2f(-HALF, HALF-BORD),
    //input right
    vec2f( HALF, HALF-BORD), vec2f(-HALF, HALF-BORD), vec2f(-HALF, BORD-HALF), vec2f( HALF, BORD-HALF),
    vec2f(-HALF, -HALF), vec2f(HALF, HALF), vec2f(-HALF, HALF), vec2f(HALF, -HALF), //invalid
    vec2f( HALF, HALF-BORD), vec2f(BORD-HALF, -HALF), vec2f(HALF-BORD, -HALF), vec2f( HALF, BORD-HALF),
    vec2f( HALF, HALF-BORD), vec2f(HALF-BORD,  HALF), vec2f(BORD-HALF,  HALF), vec2f( HALF, BORD-HALF),
    //input top
    vec2f(HALF-BORD, -HALF), vec2f(-HALF, HALF-BORD), vec2f(-HALF, BORD-HALF), vec2f(BORD-HALF, -HALF),
    vec2f(HALF-BORD, -HALF), vec2f( HALF, BORD-HALF), vec2f( HALF, HALF-BORD), vec2f(BORD-HALF, -HALF),
    vec2f(-HALF, -HALF), vec2f(HALF, HALF), vec2f(-HALF, HALF), vec2f(HALF, -HALF), //invalid
    vec2f(HALF-BORD, -HALF), vec2f(HALF-BORD,  HALF), vec2f(BORD-HALF,  HALF), vec2f(BORD-HALF, -HALF),
    //input bottom
    vec2f(BORD-HALF,  HALF), vec2f(-HALF, HALF-BORD), vec2f(-HALF, BORD-HALF), vec2f(HALF-BORD,  HALF),
    vec2f(BORD-HALF,  HALF), vec2f( HALF, BORD-HALF), vec2f( HALF, HALF-BORD), vec2f(HALF-BORD,  HALF),
    vec2f(BORD-HALF,  HALF), vec2f(BORD-HALF, -HALF), vec2f(HALF-BORD, -HALF), vec2f(HALF-BORD,  HALF),
    vec2f(-HALF, -HALF), vec2f(HALF, HALF), vec2f(-HALF, HALF), vec2f(HALF, -HALF), //invalid
};

//check if the car is within the expected square
//server code doesnt offset position when checking track progress
bool IsInCheckpoint(const Map& map, const Precompute& pre, const Car& car, int checkpoint, bool useoffset)
{
    vec2i chkpos = pre.path[checkpoint%pre.path.size()];
    vec2i carpos = (car.pos / map.tsize.Conv<float>()).Conv<int>();
    return
        chkpos.x == carpos.x &&
        chkpos.y == carpos.y;
}

//assuming the car is in the right square, is it within the walls
bool IsOnTrack(const Map& map, const Precompute& pre, const Car& car, int checkpoint)
{
    //check track tiles
    if (!IsInCheckpoint(map, pre, car, checkpoint, true))
        return false;
    vec2i chkpos = pre.path[checkpoint%pre.path.size()];
    
    //calculate local offset
    vec2f scale = map.tsize.Conv<float>();
    vec2f carpos = car.pos / scale;
    vec2f mappos = chkpos.Conv<float>();
    vec2f center = vec2f(0.5f, 0.5f);
    vec2f offset = carpos - mappos - center;
    
    //test walls
    //int index = chkpos.x+chkpos.y*map.msize.x;
    int index = pre.dirarray[(checkpoint+pre.path.size())%pre.path.size()];
    vec2f p1a = walls[index*4+0];
    vec2f p1b = walls[index*4+1];
    vec2f p2a = walls[index*4+2];
    vec2f p2b = walls[index*4+3];
    float test1 = offset.Cross(p1a, p1b);
    float test2 = offset.Cross(p2a, p2b);
    
    //return result
    return
        (test1 > 0) &&
        (test2 > 0);
}

bool IsOnTrackNew(const Map& map, const Precompute& pre, const Car& car, int checkpoint)
{
    //check track tiles
    if (!IsInCheckpoint(map, pre, car, checkpoint, true))
        return false;
    vec2i chkpos = pre.path[checkpoint%pre.path.size()];
    
    //calculate local offset
    vec2f scale = map.tsize.Conv<float>();
    vec2f carpos = car.pos;
    vec2f mappos = chkpos.Conv<float>() * scale;
    vec2f offset = carpos - mappos;
    
    const float border = 16+4;
    const float border2 = border*border;
    const float borderi = (scale.x-border)*(scale.x-border);
    int index = pre.dirarray[(checkpoint+pre.path.size())%pre.path.size()];
    if (index ==  1 || index ==  4)
        return (offset.y > border && offset.y < scale.y - border);
    if (index == 11 || index == 14)
        return (offset.x > border && offset.x < scale.x - border);
    if (index ==  2 || index ==  8)
    {
        float dist = (vec2f(0,0)*scale - offset).LengthSquared();
        return dist > border2 && dist < borderi;
    }
    if (index ==  6 || index ==  9)
    {
        float dist = (vec2f(1,0)*scale - offset).LengthSquared();
        return dist > border2 && dist < borderi;
    }
    if (index ==  3 || index == 12)
    {
        float dist = (vec2f(0,1)*scale - offset).LengthSquared();
        return dist > border2 && dist < borderi;
    }
    if (index ==  7 || index == 13)
    {
        float dist = (vec2f(1,1)*scale - offset).LengthSquared();
        return dist > border2 && dist < borderi;
    }
    return false;
}

bool IsOnTrack2(const Map& map, const Precompute& pre, const Car& car, int checkpoint)
{
    bool prevtrack = IsOnTrackNew(map, pre, car, checkpoint-1);
    bool currtrack = IsOnTrackNew(map, pre, car, checkpoint+0);
    bool nexttrack = IsOnTrackNew(map, pre, car, checkpoint+1);
    return prevtrack || currtrack || nexttrack;
}

/*****************************************************************************/
/*                             Car simulation                                */
/*****************************************************************************/

//simulate a step in the physics simulation
void Simulate(const Map& map, const Precompute& pre, Car& car, int checkpoint, int action, int iteration)
{
    //initial values
    const float fps = 30.0f; //server calls this frametime, but thats just wrong
    const float dt = 1.0f/fps;
    const float accelforce = 2500;
    vec2f force = vec2f(0,0);
    float mass = 20.0f;
    float drag_coeff = 0.1f;
    float roll_coeff = 3.0f;
    
    //going offroad
    bool ontrack = IsOnTrack2(map, pre, car, checkpoint);
    if (!ontrack)
    {
        drag_coeff *= 10.0f;
        roll_coeff *= 10.0f;
    }
    
    //environmental effects
    Area area = map.GetModifier(car.pos);
    if (area == AreaBoost)
    {
        drag_coeff = 0.0f;
        roll_coeff = 0.0f;
        car.vel *= 1.2f;
    }
    if (area == AreaMud)
    {
        drag_coeff *= 7.0f;
        roll_coeff *= 7.0f;
    }
    if (area == AreaBanana)
    {
        //todo fix this
        car.vel *= 0.1f;
    }
    if (area == AreaOil)
    {
        car.dir = car.dir.Rotate( 3.0f * PI / 180.0f);
    }
    if (area == AreaIce)
    {
        //todo fix this
        if (rand()%2 == 1)
            car.dir = car.dir.Rotate( 4.0f * PI / 180.0f);
        else
            car.dir = car.dir.Rotate(-4.0f * PI / 180.0f);
    }
    
    //parse actions
    if (action & ActionAccel)
        force += car.dir * accelforce;
    if (action & ActionBreak)
        force -= car.dir * accelforce;
    if (action & ActionTurnl)
        car.dir = car.dir.Rotate(-3.0f * PI / 180.0f);
    if (action & ActionTurnr)
        car.dir = car.dir.Rotate( 3.0f * PI / 180.0f);
    if (action & ActionDrift)
        car.drift = 1;
    
    //friction
    force -= car.vel * drag_coeff * car.vel.Length();
    force -= car.vel * roll_coeff;
    
    //apply forces
    vec2f accel = force / mass;
    car.vel += accel * dt;
    
    //power effects
    const float stunduration = 2500; //ms
    const float boostduration = 2500; //ms
    const float invincibleduration = 2500; //ms
    const float tippedduration = 500; //ms
    if (car.small_frame * dt + stunduration  * 0.001f > (pre.frame+iteration) * dt)
        car.vel *= 0.9f;
    if (car.boost_frame * dt + boostduration * 0.001f > (pre.frame+iteration) * dt)
        car.vel *= 1.05f;
    
    //kill sideways velocity
    float drift = (car.drift == 1) ? 0.97f : 0.9f;
    car.drift = 0;
    vec2f fw_vec = car.dir.Normalized();
    vec2f sw_vec = fw_vec.Rotate(PI/2.0f);
    vec2f fw_vel = fw_vec * car.vel.Dot(fw_vec);
    vec2f sw_vel = sw_vec * car.vel.Dot(sw_vec);
    car.vel = fw_vel + sw_vel * drift;
    
    //move
    car.pos += car.vel * dt;
    
    //check powerup collision
    //todo
}

//logic to figure out when we enter the next checkpoint
bool EnteredNextCheckpoint(const Map& map, const Precompute& pre, const Car& car, int checkpoint)
{
    int nextcheckpoint = (checkpoint + 1) % pre.path.size();
    return IsInCheckpoint(map, pre, car, nextcheckpoint, false);
}

/*****************************************************************************/
/*                             Brute forcing                                 */
/*****************************************************************************/

enum Instruction {
    InstNone = 0,
    InstFull,
    InstEase,
    InstFade,
};

struct Instructions
{
    Instruction iaccel;
    Instruction ibreak;
    Instruction iturnl;
    Instruction iturnr;
    Instruction idrift;
};

//preset list of behaveour patterns
const Instructions patterns[] = {
    //straight accel    ->          straight break
    {InstFade, InstEase, InstNone, InstNone, InstNone},
    //turn glide        ->          turn accel
    {InstEase, InstNone, InstFull, InstNone, InstNone},
    {InstEase, InstNone, InstNone, InstFull, InstNone},
    //turn glide        ->          turn break
    {InstNone, InstEase, InstFull, InstNone, InstNone},
    {InstNone, InstEase, InstNone, InstFull, InstNone},
    //turn accel drift  ->          turn accel
    {InstFull, InstNone, InstFull, InstNone, InstFade},
    {InstFull, InstNone, InstNone, InstFull, InstFade},
    //turn accel drift  ->          straight accel
    {InstFull, InstNone, InstFade, InstNone, InstFade},
    {InstFull, InstNone, InstNone, InstFade, InstFade},
    //straight break    ->          turn accel
    {InstEase, InstFade, InstEase, InstNone, InstNone},
    {InstEase, InstFade, InstNone, InstEase, InstNone},
    //turn break        ->          turn accel
    {InstEase, InstFade, InstFull, InstNone, InstNone},
    {InstEase, InstFade, InstNone, InstFull, InstNone},
    //turn accel        ->          straight accel
    {InstFull, InstNone, InstFade, InstNone, InstNone},
    {InstFull, InstNone, InstNone, InstFade, InstNone},
    //turn accel        ->          inverse accel
    {InstFull, InstNone, InstFade, InstEase, InstNone},
    {InstFull, InstNone, InstEase, InstFade, InstNone}
};

const int numiterations = 60;
const int numpatterns = sizeof(patterns) / sizeof(Instructions) * (numiterations+1);

int UseInstruction(Instruction instruction, int mutation, int iteration)
{
    if (instruction == InstNone) return 0;
    if (instruction == InstFull) return 1;
    if (instruction == InstEase) return numiterations - iteration - mutation <= 0;
    if (instruction == InstFade) return numiterations - iteration - mutation  > 0;
    return 0;
}

//converts a pattern and frame number into an action command
int GetAction(int pattern, int iteration)
{
    int index    = pattern / (numiterations+1);
    int mutation = pattern % (numiterations+1);
    const Instructions& inst = patterns[index];
    int action = 0;
    action |= ActionAccel * UseInstruction(inst.iaccel, mutation, iteration);
    action |= ActionBreak * UseInstruction(inst.ibreak, mutation, iteration);
    action |= ActionTurnl * UseInstruction(inst.iturnl, mutation, iteration);
    action |= ActionTurnr * UseInstruction(inst.iturnr, mutation, iteration);
    action |= ActionDrift * UseInstruction(inst.idrift, mutation, iteration);
    return action;
}

//measures in pixels how far along the checkpoint path we've gone
float ScoreNextCheckpoint(const Precompute& pre, const Car& car, int checkpoint)
{
    vec2f prev = pre.GetCheck(checkpoint-1);
    vec2f curr = pre.GetCheck(checkpoint+0);
    vec2f next = pre.GetCheck(checkpoint+1);
    vec2f offs = curr + (next - prev);
    float p1 = LineProgress(prev, curr, car.pos);
    float p2 = LineProgress(curr, next, car.pos);
    float p3 = LineProgress(curr, offs, car.pos);
    float t1 = (float)(checkpoint+0) * 128.0f - (curr - car.pos).Length();
    float t2 = (float)(checkpoint+1) * 128.0f - (next - car.pos).Length();
    float t3 = (p3 < 0) ? t1 : t2;
    return t3;
}

float Score(const Map& map, const Precompute& pre, Car car, int checkpoint, bool inverse, int instruction, int t)
{
    //store initial position
    float score = 0.0f;
    float currpos = ScoreNextCheckpoint(pre, car, checkpoint);
    
    //bonus for simply driving
    int action = GetAction(instruction, 0);
    action = InverseAction(action, inverse);
    if ((action & ActionBreak) &&
        car.vel.LengthSquared() < 2.0f)
        score -= 1.0f;
    
    //do the simulation
    for (int i = 0; i < numiterations; ++i)
    {
        //simulate
        int action = GetAction(instruction, i);
        action = InverseAction(action, inverse);
        Simulate(map, pre, car, checkpoint, action, i);
        
        //advance checkpoint
        if (EnteredNextCheckpoint(map, pre, car, checkpoint))
            ++checkpoint;
    }
    
    //analyze end result
    float nextpos = ScoreNextCheckpoint(pre, car, checkpoint);
    score += (nextpos - currpos);
    return score;
}

//brute force a preset list of bahaveour patterns
int BestActionFromInstructions(const Map& map, const Precompute& pre, const Car& car, bool inverse)
{
    //brute force all presets
    int bestpattern = 0;
    float bestscore = -99999999999.0f;
    for (int i = 0; i < numpatterns; ++i)
    {
        float score = Score(map, pre, car, pre.checkpoint, inverse, i, 1);
        if (bestscore < score)
        {
            bestscore = score;
            bestpattern = i;
        }
    }
    
    //return the action from the best result
    int i1 = bestpattern / numiterations;
    int i2 = bestpattern % numiterations;
    //std::cout << i1 << ":" << i2 << ":    ";
    
    //redraw the best action
    Score(map, pre, car, pre.checkpoint, inverse, bestpattern, 0);
    int bestaction = GetAction(bestpattern, 0);
    bestaction = InverseAction(bestaction, inverse);
    
    //return
    return bestaction;
}

/*****************************************************************************/
/*                             Bot logic                                     */
/*****************************************************************************/

//find index in the array corresponding to the right player id
//todo, might not be needed
int GetCarIndex(int id, const State& state)
{
    for (int i = 0; i < state.cars.size(); ++i)
        if (id == state.cars[i].id)
            return i;
    return -1;
}

//helper function to enable reverse driving
float InverseAngle(float angle, bool& inverse)
{
    if (angle < -PI/2.0f) {angle += PI; inverse = !inverse;}
    if (angle >  PI/2.0f) {angle -= PI; inverse = !inverse;}
    return angle;
}

//heart of the AI, using logic to determine the best action
int GetBestAction(const Map& map, const Precompute& pre, const State& state, int version)
{
    //default action is none
    int action = 0;
    const Car& car = state.cars[pre.carindex];
    
    //check if we're already driving backwards
    float dirangle = car.dir.ToAngle();
    float velangle = car.vel.ToAngle();
    float difangle = fabs(AngleMod(dirangle, velangle));
    bool inverse = difangle > PI/2.0f;
    float inverseangle = inverse ? PI : 0.0f;
    
    //brute force a solution from a preset list
    action = BestActionFromInstructions(map, pre, car, inverse);
    //PrintAction(action);
    
    //determine the use of items
    if (car.power == PowerupBanana)
        action |= ActionPower;
    if (car.power == PowerupGreenShell)
        action |= ActionPower;
    if (car.power == PowerupRedShell)
        action |= ActionPower;
    if (car.power == PowerupBlueShell)
        action |= ActionPower;
    if (car.power == PowerupOil)
        action |= ActionPower;
    if (car.power == PowerupMushroom)
        action |= ActionPower;
    if (car.power == PowerupBigMushroom)
        action |= ActionPower;
    if (car.power == PowerupLightning)
        action |= ActionPower;
    if (car.power == PowerupStar)
        action |= ActionPower;
    
    //return
    return action;
}

/*****************************************************************************/
/*                             Bot loop                                      */
/*****************************************************************************/

//this function instances an AI, connects to the server and executes bot logic
void Bot(unsigned int ip, unsigned short port, std::string nick, int version)
{
    //setup network
    Network net;
    net.Create(ip, port);
    net.Send(nick);
    
    //parse map
    int id;
    Map map;
    {
        //await network
        
        
        //parse json
        cJSON* root = NULL;
        while (root == NULL)
        {
            std::string line = net.Receive();
            if (line == "DISCONNECTED") return;
            root = cJSON_Parse(line.c_str());
        }
        //std::cout << cJSON_Print(root) << "\n";
        id  = cJSON_GetObjectItem(root, "id")->valueint;
        map = ParseMap(cJSON_GetObjectItem(root, "map"));
        cJSON_Delete(root);
        
        //print
        Log("Player id: " + ToString(id));
    }
    
    //compute checkpoints
    Precompute pre;
    pre.path = GetPath(map);
    
    pre.dirarray = GetDirectionArray(pre, map);
    pre.checkpoints = CalculateCheckpoints(map, pre, version);
    pre.carindex = -1;
    pre.checkpoint = -1;
    pre.frame = 0;
    
    //send default command
    net.Send(ToString(ActionAccel));
    
    std::vector<vec2f> history1;
    
    //begin ai loop
    State initstate;
    State prevstate;
    while (true)
    {
        //await network
        std::string line = net.Receive();
        if (line == "DISCONNECTED") break;
        
        //parse json
        cJSON* root = cJSON_Parse(line.c_str());
        if (root == NULL) continue; //Bug: may sometimes be unparsable due to -1.IND
        State state = ParseState(root);
        //std::cout << cJSON_Print(root) << "\n";
        cJSON_Delete(root);
        
        //store initial state
        if (pre.frame == 0)
            initstate = state;
        
        //advance checkpoint
        pre.carindex = GetCarIndex(id, state);
        if (EnteredNextCheckpoint(map, pre, state.cars[pre.carindex], pre.checkpoint))
            ++pre.checkpoint;
        
        //analyse state difference
        ++pre.frame;
        if (pre.frame > 1)
        {
            //copy last state
            for (int i = 0; i < state.cars.size(); ++i)
            {
                state.cars[i].boost_frame = prevstate.cars[i].boost_frame;
                state.cars[i].small_frame = prevstate.cars[i].small_frame;
            }
            
            //todo minify thing
            
            //check if used boost
            for (int i = 0; i < state.cars.size(); ++i)
            {
                if (prevstate.cars[i].power == PowerupMushroom &&
                        state.cars[i].power != PowerupMushroom)
                        state.cars[i].boost_frame = pre.frame;
                if (prevstate.cars[i].power == PowerupBigMushroom &&
                        state.cars[i].power != PowerupBigMushroom)
                        state.cars[i].boost_frame = pre.frame;
                if (prevstate.cars[i].power == PowerupStar &&
                        state.cars[i].power != PowerupStar)
                        state.cars[i].boost_frame = pre.frame;
            }
        }
        
        
        //send best action
        int action = ActionAccel;
        if (version >= 1)
            action = GetBestAction(map, pre, state, version);
        net.Send(ToString(action));
        
        //store history
        prevstate = state;
    }
}

/*****************************************************************************/
/*                             Main                                          */
/*****************************************************************************/

#include <thread>
int main(int argc, char* argv[])
{
    //read arguments
    unsigned int ip = ParseIP("127.0.0.1");
    unsigned short port = 31337;
    int bots = 1;
    if (argc >= 2) ip = ParseIP(argv[1]);
    if (argc >= 3) port = ToValue<unsigned short>(std::string(argv[2]));
    Bot(ip, port, "WiggleBot", 99);
    
    return 0;
}



















