import processing.core.*; 
import processing.data.*; 
import processing.event.*; 
import processing.opengl.*; 

import controlP5.*; 
import processing.net.*; 

import java.util.HashMap; 
import java.util.ArrayList; 
import java.io.File; 
import java.io.BufferedReader; 
import java.io.PrintWriter; 
import java.io.InputStream; 
import java.io.OutputStream; 
import java.io.IOException; 

public class sketch_150911b extends PApplet {




final int PORT = 2015;

Scene currentScene = null;
PFont smallFont = null;
PFont bigFont = null;

public void setup() {
  
  noFill();
  stroke(255);
  strokeWeight(1);
  background(0);
  frameRate(120);
  
  smallFont = loadFont("Hack-Bold-16.vlw");
  bigFont = loadFont("Hack-Bold-48.vlw");
  
  currentScene = new GameModeScene(this);
  currentScene.setup();
}

public void draw() {
  currentScene.draw();
}

public void controlEvent(ControlEvent cevent) {
  currentScene.controlEvent(cevent);
}

public void serverEvent(Server server, Client client) {
  currentScene.serverEvent(server, client);
}

public void disconnectEvent(Client client) {
  currentScene.disconnectEvent(client);
}

public void mousePressed() {
  currentScene.mousePressed();
}

public void keyReleased() {
  currentScene.keyReleased();
}

public void keyPressed() {
  currentScene.keyPressed();
}
class Blur {
  PImage buffer;
  int sum = 16;
  int[] kernel = { 
    2, 4, 2, 
    4, 8, 4, 
    2, 4, 2 
  };
  int[] offset;
  
  public void init() {
    buffer = createImage(width, height, RGB);
    offset = new int[] { 
        -(width + 1), -(width), -(width - 1),
        -1, 0, +1,
        +(width - 1), +(width), +(width + 1)
    };
  }
  
  public void applyEffect() {
    loadPixels();

    int i = 0;
    int alpha = 255 << 24;
    for( int y = 0; y < height; ++y) {
      for( int x = 0; x < width; ++x) {
    
        int r = 0, g = 0, b = 0;
        
        for( int o = 0; o < offset.length; ++o) {
          
          int a = i + offset[o];
          if ( a < 0 || a >= pixels.length) {
            continue;
          }
          
          int c = pixels[a];
          r += ((c >> 16) & 0xFF) * kernel[o];
          g += ((c >> 8) & 0xFF) * kernel[o];
          b += (c & 0xFF) * kernel[o];
        }
        
        r = (r >> 4); if (r < 0) r = 0; else if (r > 255) r = 255;
        g = (g >> 4); if (g < 0) g = 0; else if (g > 255) g = 255;
        b = (b >> 4); if (b < 0) b = 0; else if (b > 255) b = 255;
                
        buffer.pixels[i++] = (alpha | (r << 16) | (g << 8) | (b));
      }
    }
    
    buffer.updatePixels();
    image(buffer, 0, 0);
  }
}
class GameClient {
  
  Client remoteClient;
  Ship ship;
  Universe universe;    
  float lastShoot = 3.0f;
  boolean shooting = false;
  int life = 3;
  int points = 0;
  
  public GameClient(Universe universe, Ship ship, Client remoteClient) {
    this.ship = ship;
    this.universe = universe;
    this.remoteClient = remoteClient;
  }
  
  public void accleration(boolean value) {
    this.ship.accleration = value;
    this.universe.updateObject(this.ship);
  }
  
  public void turnLeft(boolean value) {
    this.ship.turnLeft = value;
    this.universe.updateObject(this.ship);
  }
  
  public void turnRight(boolean value) {
    this.ship.turnRight = value;
    this.universe.updateObject(this.ship);
  }
  
  public void shoot(boolean value) {
    this.shooting = value;
  }
  
  public void update(float elapsed) {
    lastShoot += elapsed;
    if (shooting && this.ship.visible) {
      if (lastShoot >= 0.5f) {
        
        float[] shift = x_rotate(this.ship.model1[0], this.ship.direction);
        
        this.universe.updateObject(this.universe.bullet(
          this.ship.position[0] + shift[0], 
          this.ship.position[1] + shift[1], 
          (cos(this.ship.direction - PI / 2) * Bullet.BULLET_SPEED),
          (sin(this.ship.direction - PI / 2) * Bullet.BULLET_SPEED),
          Bullet.BULLET_LIFE,
          this.ship.entityColor,
          this.ship.id
          ));
        
        lastShoot = 0.0f;
      }
    }
  }
}
class GameModeScene extends Scene {
  
  ControlP5 cp5;
  
  public GameModeScene(PApplet applet) {
    super(applet);
  }
  
  public void clean() {
    if (cp5 != null) {
      cp5.hide();
      cp5 = null;
    }
  }
  
  public void setup() {
    background(0);
    
    int w = width / 2;
    int h = 25;
    int ox = (width - w) / 2;
    
    textFont(bigFont);
    float fw = textWidth("Asteroids");
    text("Asteroids", (width - fw) / 2, 75);
    
    cp5 = new ControlP5(this.applet);
    cp5.addButton("server")
       .setSize(w, h)
       .setPosition(ox, 200);
    cp5.addButton("client")
       .setSize(w, h)
       .setPosition(ox, 250);
  }
  
  public void draw() {
  }
  
  public void controlEvent(ControlEvent cevent) {
    if (cevent.getName().equals("server")) {
      currentScene.clean();
      currentScene = new LocalGameScene(this.applet);
      currentScene.setup();
    } else if (cevent.getName().equals("client")) {      
      currentScene.clean();
      currentScene = new JoinScene(this.applet);
      currentScene.setup();
    }
  }
}
abstract class GameObject {
  
  public static final int TYPE_ASTEROID = 0;
  public static final int TYPE_SHIP = 1;
  public static final int TYPE_BULLET = 2;
  public static final int TYPE_EXPLOSION = 3;
  
  int id;
  int type;
  boolean visible = false;
  
  public GameObject(int type) {
    this.type = type;
  }
  
  public abstract boolean collide(float[][] polygon);
  public abstract void update(float fraction);
  public abstract void draw();
  public abstract String serialize();
}

class Explosion extends GameObject {
  
  float scale = 0.0f;
  float force = 4.0f;
  float[] position = new float[2];
  float[][] lines = new float[10][2];
  
  public Explosion() {
    super(GameObject.TYPE_EXPLOSION);
  }
  
  public String serialize() {
    return "e:" + str(id) + ":" + str(position[0]) + ":" + str(position[1]) + ":" + str(scale) + ":" + str(force); 
  }
  
  public void sync(float px, float py, float f) {
    this.position[0] = px;
    this.position[1] = py;
    this.force = f;
    this.scale = 0.0f;
    
    for (int i = 0; i < lines.length; i+= 2) {
      float rad = 2 * PI * random(1.0f);
      px = cos(rad);
      py = sin(rad);
      
      lines[i + 0][0] = px;
      lines[i + 0][1] = py;
      lines[i + 1][0] = px * 2;
      lines[i + 1][1] = py * 2;
    }
  }
  
  public boolean collide(float[][] polygon) {
    return false;
  }
  
  public void update(float fraction) {
    if (this.scale >= this.force) {
      this.visible = false;
      return;
    }
    
    this.scale += this.force * fraction;
  }
  
  public void draw() {
    stroke(255);
    for (int i = 0; i < lines.length; i+= 2) {
      line(position[0] + lines[i + 0][0] * scale, position[1] + lines[i + 0][1] * scale, position[0] + lines[i + 1][0] * scale, position[1] + lines[i + 1][1] * scale); 
    }
  }
}

class Bullet extends GameObject {
  
  static final float BULLET_SPEED = 250.0f;
  static final float BULLET_LIFE = 2.0f;
  
  float[] position = new float[2];
  float[] force = new float[2];
  float[] direction = new float[2];
  float size = 10.0f;
  float life = 0.0f;
  int entityColor = color(255);
  int shipId;
  
  public Bullet() {
    super(GameObject.TYPE_BULLET);
  }
  
  public String serialize() {
    return "b:" + str(id) + ":" + str(position[0]) + ":" + str(position[1]) + ":" + str(force[0]) + ":" + str(force[1]) + ":" + str(life) + ":" + str(entityColor) + ":" + str(shipId); 
  }
  
  public void sync(float px, float py, float fx, float fy, float life, int ec, int sid) {
    this.position[0] = px;
    this.position[1] = py;
    this.force[0] = fx;
    this.force[1] = fy;    
    float fl = sqrt( fx * fx + fy * fy);
    this.direction[0] = fx / fl * this.size;
    this.direction[1] = fy / fl * this.size;
    this.life = life;
    this.entityColor = ec;
    this.shipId = sid;
  }
  
  public boolean collide(float[][] polygon) {
    return pointInPolygon(position, polygon) || pointInPolygon(position[0] - direction[0], position[1] - direction[1], polygon);
  }
  
  public void update(float elapsed) {
    
    this.life -= elapsed;
    this.position[0] += this.force[0] * elapsed;
    this.position[1] += this.force[1] * elapsed;
    
    wrap(this.position, this.size);
  }
  
  public void draw() {    
    stroke(entityColor);
    line(position[0], position[1], position[0] - direction[0], position[1] - direction[1]);
  }
}

class Asteroid extends Entity {
  float[][] model = new float[][] {
    new float[] { -5.0f, -7.5f },
    new float[] { +2.5f, -7.5f },
    new float[] { +7.5f, -2.5f },
    new float[] { +7.5f, +2.5f },
    new float[] { +2.5f, +7.5f },
    new float[] { -2.5f, +7.5f },
    new float[] { -7.5f, +2.5f },
    new float[] { -7.5f, -2.5f },
  };
  float[][] modelTranslated = new float[8][2];
  
  public Asteroid() {
    super(GameObject.TYPE_ASTEROID);
    
    this.entityColor = color(130);
  }
  
  public boolean collide(float[][] polygon) {    
    for (int p = 0; p < modelTranslated.length; ++p) {
      if (pointInPolygon(modelTranslated[p], polygon)) {
        return true;
      }
    }
    return false;
  }
  
  public void update(float fraction) {
    super.update(fraction);
    
    for (int i = 0; i < model.length; ++i) {
      to_world(modelTranslated[i], direction, size, position, model[i]);
    }
  }
  
  public void draw() {
    stroke(entityColor);
    drawPolygon(modelTranslated);
  }
}

class Ship extends Entity {
  float[][] model1 = new float[][] {
    new float[] { 7 -7.32448f, 0 -14.8034f },
    new float[] { 0 -7.32448f, 20-14.8034f },
    new float[] { 3 -7.32448f, 17-14.8034f },
    new float[] { 12-7.32448f, 17-14.8034f },
    new float[] { 15-7.32448f, 20-14.8034f }
  };
  float[][] model2 = new float[][] {
    new float[] { 4 -7.32448f, 19-14.8034f },
    new float[] { 11-7.32448f, 19-14.8034f }
  };
  float[][] model1Translated = new float[5][2];
  float[][] model2Translated = new float[2][2];
  
  public Ship() {
    super(GameObject.TYPE_SHIP);
  }
  
  public boolean collide(float[][] polygon) {
    for(int p = 0; p < model1Translated.length; ++p) {
      if (pointInPolygon(model1Translated[p], polygon)) {
        return true;
      }
    }
    return false;
  }
  
  public void update(float fraction) {
    super.update(fraction);
    
    for (int i = 0; i < model1.length; ++i) {
      to_world(model1Translated[i], direction, 1.0f, position, model1[i]);
    }
    
    for (int i = 0; i < model2.length; ++i) {
      to_world(model2Translated[i], direction, 1.0f, position, model2[i]);
    }
  }
  
  public void draw() {
    stroke(entityColor);
    drawPolygon(model1Translated);
    
    if (this.accleration) {
      stroke(128, 255, 255);
      drawPolygon(model2Translated);
    }
  }
}

abstract class Entity extends GameObject {
  
  final float TURN_SPEED = 3.0f;
  final float ACCLERATION = 100.0f;
  
  int entityColor = color(255);
  float size = 1.0f;
  float hsize = 0.5f;
  float direction = 0.0f;
  float rotation = 0.0f;
  float[] position = new float[] { 0, 0 };
  float[] force = new float[] { 0, 0 };
  boolean accleration = false;
  boolean turnLeft = false;
  boolean turnRight = false;
  
  public Entity(int type) {
    super(type);
  }
  
  public String serialize() {    
    return (type == TYPE_SHIP ? "s:" : "a:") + str(id) + ":" + str(position[0]) + ":" + str(position[1]) + ":" + str(force[0]) + ":" + str(force[1]) + ":" + str(size) + ":" + str(direction) + ":" + str(rotation) + ":" + str(accleration) + ":" + str(turnLeft) + ":" + str(turnRight) + ":" + str(entityColor); 
  }
  
  public void sync(float px, float py, float fx, float fy, float size, float d, float r, boolean a, boolean tl, boolean tr, int ec) {
    this.position[0] = px;
    this.position[1] = py;
    this.force[0] = fx;
    this.force[1] = fy;
    this.size = size;
    this.hsize = size / 2.0f;
    this.direction = d;
    this.rotation = r;
    this.accleration = a;
    this.turnLeft = tl;
    this.turnRight = tr;
    this.entityColor = ec;
  }
  
  public void update(float fraction) {
    
    if (accleration) {
      force[0] += ACCLERATION * fraction * cos(direction - PI / 2);
      force[1] += ACCLERATION * fraction * sin(direction - PI / 2);
      
      forceLimit(force, ACCLERATION);
    }
    
    if (turnLeft) {
      direction -= TURN_SPEED * fraction;
    }
    
    if (turnRight) {
      direction += TURN_SPEED * fraction;
    }
    
    direction += rotation * fraction;
    
    position[0] += force[0] * fraction;
    position[1] += force[1] * fraction;
    
    wrap(position, size);
  }
}
class GameServer {
  
  Server serverInstance;
  LocalUniverse universe;
  
  public GameServer(PApplet applet, LocalUniverse universe, int port) {
    this.universe = universe;
    this.serverInstance = new Server(applet, port);
  }
  
  public void update(float elapsed) {
    Client remoteClient = this.serverInstance.available();
    if (remoteClient != null) {
      GameClient gc = universe.getGameClient(remoteClient);
      if (gc == null) {
        gc = universe.connect(remoteClient);
        if (gc == null) {
          remoteClient.stop();
          return;
        }
      }
      
      processMessage(gc);
    }
  }
  
  public void destroy(int type, int id) {
    this.serverInstance.write("x:" + str(type) + ":" + str(id) + "|");
  }
  
  public void sync(Client client, Universe u) {
    String message = "y";
    
      for( int b = 0; b < u.bullets.length; ++b) {
        if (u.bullets[b].visible) {
          message += ":" + u.bullets[b].serialize();
        }
      }
      for( int a = 0; a < u.asteroids.length; ++a) {
        if (u.asteroids[a].visible) {
          message += ":" + u.asteroids[a].serialize();
        }
      }
      for( int s = 0; s < u.ships.length; ++s) {
        if (u.ships[s].visible) {
          message += ":" + u.ships[s].serialize();
        }
      }
      for( int e = 0; e < u.explosions.length; ++e) {
        if (u.explosions[e].visible) {
          message += ":" + u.explosions[e].serialize();
        }
      }
      
      message += "|";
      client.write(message);
  }
  
  public void write(String message) {
    this.serverInstance.write(message + "|");
  }
  
  public void processMessage(GameClient gameClient) {
    char r = gameClient.remoteClient.readChar();
    if (r == 'a') {
      gameClient.accleration(true);
    } else if (r == 'b') {
      gameClient.turnLeft(true);
    } else if (r == 'c') {
      gameClient.turnRight(true);
    } else if (r == 'd') {
      gameClient.shoot(true);
    } else if (r == 'e') {
      gameClient.accleration(false);
    } else if (r == 'f') {
      gameClient.turnLeft(false);
    } else if (r == 'g') {
      gameClient.turnRight(false);
    } else if (r == 'h') {
      gameClient.shoot(false);
    }
  }
}
class JoinScene extends Scene {
  
  ControlP5 cp5;

  public JoinScene(PApplet applet) {
    super(applet);
  }
  
  public void clean() {
    if (cp5 != null) {
      cp5.hide();
      cp5 = null;
    }
  }
  
  public void setup() {
    background(0);
    
    int w = width / 2;
    int h = 25;
    int ox = (width - w) / 2;
    
    textFont(bigFont);
    float fw = textWidth("Join Game");
    text("Join Game", (width - fw) / 2, 75);    
    
    cp5 = new ControlP5(this.applet);    
    cp5.addTextfield("remoteIP")
       .setSize(w, h)
       .setPosition(ox, 150)
       .setFocus(true)
       .setAutoClear(false)
       .setText("127.0.0.1")
       .setLabel("Remote address");
    cp5.addButton("connect")
       .setSize(w, h)
       .setPosition(ox, 200);
    cp5.addButton("back")
       .setSize(w, h)
       .setPosition(ox, 300);
  }
  
  public void draw() {
  }
    
  public void controlEvent(ControlEvent cevent) {    
    if (cevent.getName().equals("back")) {
      currentScene.clean();
      currentScene = new GameModeScene(this.applet);
      currentScene.setup();
    } else if (cevent.getName().equals("connect")) {
      String address = ("" + ((controlP5.Textfield)cp5.get("remoteIP")).getText()).trim(); 
      
      if (address == null || address.length() == 0) {
        return;
      }
      
      Client client = new Client(this.applet, address, PORT);
      if (client.active()) {
        currentScene.clean();
        currentScene = new RemoteGameScene(this.applet, client);
        currentScene.setup();
      }
    }
  }
}
class LocalGameScene extends Scene {
  
  LocalUniverse universe;
  GameClient client;
  int lastFrame = 0;
  boolean accleration = false, turnRight = false, turnLeft = false, shooting = false;
  Blur blur = null;
  
  public LocalGameScene(PApplet applet) {
    super(applet);
  }
  
  public void clean() {
    
  }
  
  public void setup() {    
    universe = new LocalUniverse(this.applet, true);
    client = universe.connect(null);
    blur = new Blur();
    blur.init();
    
    lastFrame = millis();
  }
  
  public void draw() {  
    int currentFrame = millis();
    float elapsed = (float)(currentFrame - lastFrame) / 1000.0f;
    lastFrame = currentFrame;
      
    universe.update(elapsed);
    universe.draw();
    
    blur.applyEffect();
  }
  
  public void keyReleased() {
    if (keyCode == 0x20) {
      if (shooting) {
        shooting = false;
        client.shoot(shooting);
      }
    }
    
    if (key == CODED) {
      
      if (keyCode == UP) {
        if (accleration) {
          accleration = false;
          client.accleration(accleration);
        }
      }
      
      if (keyCode == LEFT) {
        if (turnLeft) {
          turnLeft = false;
          client.turnLeft(turnLeft);
        }
      }
      
      if (keyCode == RIGHT) {
        if (turnRight) {
          turnRight = false;
          client.turnRight(turnRight);
        }
      }
    }
  }
    
  public void keyPressed() {
    if (keyCode == 82 || keyCode == 112) {
      if (universe != null) {
        universe.reset();
      }
    }
    
    if (keyCode == 0x20) {
      if (!shooting) {
        shooting = true;
        client.shoot(shooting);
      }
    }
    
    if (key == CODED) {
      if (keyCode == UP) {
        if (!accleration) {
          accleration = true;
          client.accleration(accleration);
        }
      }
      if (keyCode == LEFT) {
        if (!turnLeft) {
          turnLeft = true;
          client.turnLeft(turnLeft);
        }
      }
      if (keyCode == RIGHT) {
        if (!turnRight) {
          turnRight = true;
          client.turnRight(turnRight);
        }
      }
    }
  }
  
  public void serverEvent(Server server, Client client) {
    if (universe != null) {
      universe.connect(client);
    }
  }
  
  public void disconnectEvent(Client client) {
    if (universe != null) {
      universe.disconnect(client);
    }
  }
}
class LocalUniverse extends Universe {
  
  GameClient[] clients = new GameClient[MAX_SHIP];
  GameServer server = null;  
  int[] colors = new int[] {
    color(255, 255, 255),
    color(255, 0, 0),
    color(255, 255, 0),
    color(0, 255, 255)
  };
  
  public LocalUniverse(PApplet applet, boolean acceptRemote) {
    if (acceptRemote) {
      this.server = new GameServer(applet, this, PORT);
    }
  }
  
  public void destroy(int type, int id) {
    super.destroy(type, id);
    
    if (server != null) {
      server.destroy(type, id);
    }
  }
  
  public void updateObject(GameObject go) {
    if (server != null) {
      server.write(go.serialize());
    }
  }
  
  public void reset() {
    for( int b = 0; b < bullets.length; ++b) {
      bullets[b].visible = false;
    }
    for( int a = 0; a < asteroids.length; ++a) {
      asteroids[a].visible = false;
    }
    for( int s = 0; s < ships.length; ++s) {
      ships[s].visible = false;
    }
    for( int e = 0; e < explosions.length; ++e) {
      explosions[e].visible = false;
    }
    for( int c = 0; c < clients.length; ++c) {
      if (clients[c] == null) continue;      
      shipReset(clients[c].ship);
    }
  }
  
  public void shipReset(Ship s) {
    s.sync(width / 2, height / 2, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, false, false, false, s.entityColor);
    s.visible = true;
    updateObject(s);
  }
  
  public void onDamage(int indexOfShip, Ship s) {
    --clients[indexOfShip].life;    
    if (clients[indexOfShip].life > 0) {
      shipReset(s);
    } else {
      destroy(s.type, s.id);
    }
  }
  
  public void update(float elapsed) {
    super.update(elapsed);
    
    // check bullet -> asteroid, ship, life
    for(int b = 0; b < bullets.length; ++b) {
      if (bullets[b].visible) {
        
        if (bullets[b].life <= 0.0f) {
          destroy(bullets[b].type, bullets[b].id);
          continue;
        }
        
        for (int a = 0; a < asteroids.length; ++a) {
          if (asteroids[a].visible) {
            if (bullets[b].collide(asteroids[a].modelTranslated)) {
              
              clients[bullets[b].shipId].points += 1;
              updateObject(explosion(asteroids[a].position[0], asteroids[a].position[1], 10.0f));
              
              destroy(bullets[b].type, bullets[b].id);
              destroy(asteroids[a].type, asteroids[a].id);
              break;
            }
          }
        }
        
        for (int s = 0; s < ships.length; ++s) {
          if (ships[s].visible && ships[s].entityColor != bullets[b].entityColor) {
            if (bullets[b].collide(ships[s].model1Translated)) {
              
              clients[bullets[b].shipId].points += 10;
              updateObject(explosion(ships[s].position[0], ships[s].position[1], 10.0f));
              
              destroy(bullets[b].type, bullets[b].id);
              onDamage(s, ships[s]);
              break;
            }
          }
        }
      }
    }
  
    // check asteroids -> ship
    for (int a = 0; a < asteroids.length; ++a) {
      if (asteroids[a].visible) {
        for (int s = 0; s < ships.length; ++s) {
          if (asteroids[a].collide(ships[s].model1Translated)) {
            
              clients[s].points += 2;
              updateObject(explosion(asteroids[a].position[0], asteroids[a].position[1], 10.0f));
              destroy(asteroids[a].type, asteroids[a].id);
              onDamage(s, ships[s]);
              break;
          }
        }
      }
    }
    
    // shooting
    for(int c = 0; c < clients.length; ++c) {
      if (clients[c] != null) {
        clients[c].update(elapsed);
      }
    }
    
    if (this.server != null) {
      this.server.update(elapsed);
    }
    
    // asteroids respawn
    for (int a = 0; a < asteroids.length; ++a) {
      if (!asteroids[a].visible) {
        asteroids[a].sync(random(width), random(height), 10.5f, 0.2f, 2.0f, 0.0f, 1.0f, false, false, false, color(130)); 
        asteroids[a].visible = true;
        updateObject(asteroids[a]);
        
        break;
      }
    }
  }
  
  public GameClient getGameClient(Client remoteClient) {
    if (remoteClient == null) return null;
    
    for(int c = 0; c < clients.length; ++c) {
      if (clients[c] != null && clients[c].remoteClient == remoteClient) {
        return clients[c];
      }
    }
    
    return null;
  }
  
  public GameClient connect(Client remoteClient) {
    
    int i = 0;
    for (int s = 0; s < ships.length; ++s) {
      if (!ships[s].visible) {
        i = s;
        break;
      }
    }
    
    Ship s = ship(width / 2, height / 2, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, false, false, false, colors[i]);
    if (s != null) {
      clients[s.id] = new GameClient(this, s, remoteClient);
      
      if (remoteClient != null) {
        server.sync(remoteClient, this);
      }
      
      return clients[s.id]; 
    }
    return null;
  }
  
  public void disconnect(Client remoteClient) {
    for(int c = 0; c < clients.length; ++c) {
      if (clients[c] != null && clients[c].remoteClient == remoteClient) {
        disconnect(clients[c]);
        break;
      }
    }
  }
  
  public void disconnect(GameClient client) {    
    destroy(client.ship.type, client.ship.id);
    clients[client.ship.id] = null;
  }
}
class RemoteGameScene extends Scene {
  
  int lastFrame = 0;
  boolean accleration = false, turnRight = false, turnLeft = false, shooting = false;
  Blur blur = null;
  RemoteGameClient client;
  Universe universe;
  
  public RemoteGameScene(PApplet applet, Client networkClient) {
    super(applet);
  
    this.universe = new Universe();
    this.client = new RemoteGameClient(this.universe, networkClient);
  }
  
  public void clean() {
  }
  
  public void setup() {
    background(0);
    
    blur = new Blur();
    blur.init();
    
    lastFrame = millis();
  }
  
  public void draw() {  
    int currentFrame = millis();
    float elapsed = (float)(currentFrame - lastFrame) / 1000.0f;
    lastFrame = currentFrame;
    
    client.update(elapsed);
    
    universe.update(elapsed);
    universe.draw();
    
    blur.applyEffect();
  }
  
  public void keyReleased() {
    if (keyCode == 0x20) {
      if (shooting) {
        shooting = false;
        client.shoot(shooting);
      }
    }
    
    if (key == CODED) {
      if (keyCode == UP) {
        if (accleration) {
          accleration = false;
          client.accleration(accleration);
        }
      }
      
      if (keyCode == LEFT) {
        if (turnLeft) {
          turnLeft = false;
          client.turnLeft(turnLeft);
        }
      }
      
      if (keyCode == RIGHT) {
        if (turnRight) {
          turnRight = false;
          client.turnRight(turnRight);
        }
      }
    }
  }
    
  public void keyPressed() {
    if (keyCode == 0x20) {
      if (!shooting) {
        shooting = true;
        client.shoot(shooting);
      }
    }
    
    if (key == CODED) {
      if (keyCode == UP) {
        if (!accleration) {
          accleration = true;
          client.accleration(accleration);
        }
      }
      if (keyCode == LEFT) {
        if (!turnLeft) {
          turnLeft = true;
          client.turnLeft(turnLeft);
        }
      }
      if (keyCode == RIGHT) {
        if (!turnRight) {
          turnRight = true;
          client.turnRight(turnRight);
        }
      }
    }
  }
  
  public void disconnectEvent(Client client) {
    currentScene.clean();
    currentScene = new GameModeScene(this.applet);
    currentScene.setup();
  }
}

class RemoteGameClient {
  
  Client remoteClient;
  Universe universe;
  
  public RemoteGameClient(Universe universe, Client remoteClient) {
    this.universe = universe;
    this.remoteClient = remoteClient;
  }
  
  public void accleration(boolean value) {
    if (value) {
      remoteClient.write('a');
    } else {      
      remoteClient.write('e');
    }
  }
  
  public void turnLeft(boolean value) {
    if (value) {
      remoteClient.write('b');
    } else {      
      remoteClient.write('f');
    }
  }
  
  public void turnRight(boolean value) {
    if (value) {
      remoteClient.write('c');
    } else {      
      remoteClient.write('g');
    }
  }
  
  public void shoot(boolean value) {
    if (value) {
      remoteClient.write('d');
    } else {      
      remoteClient.write('h');
    }
  }
  
  public void update(float elapsed) {
    if (!remoteClient.active()) {
      return;
    }
    
    if (remoteClient.available() > 0) {
      parseMessage();
    } 
  }
  
  public void parseMessage() {
    String t = remoteClient.readStringUntil((int)'|');

    if (t == null) return;
    
    String[] tokens = split(t.substring(0, t.length() - 1), ':');  
    int c = 0;
    
    String command = tokens[c++]; 
    if ("x".equals(command)) {
      int type = PApplet.parseInt(tokens[c++]);
      int id = PApplet.parseInt(tokens[c++]);
      
      universe.destroy(type, id);
    } else if ("y".equals(command)) {      
      while(c < tokens.length) {
        command = tokens[c++];
        if ("a".equals(command)) {
          universe.asteroid(
            PApplet.parseInt(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseBoolean(tokens[c++]),
            PApplet.parseBoolean(tokens[c++]),
            PApplet.parseBoolean(tokens[c++]),
            PApplet.parseInt(tokens[c++])
          );
        } else if ("s".equals(command)) {
          universe.ship(
            PApplet.parseInt(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseBoolean(tokens[c++]),
            PApplet.parseBoolean(tokens[c++]),
            PApplet.parseBoolean(tokens[c++]),
            PApplet.parseInt(tokens[c++])
          );
        } else if ("b".equals(command)) {
          universe.bullet(
            PApplet.parseInt(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseInt(tokens[c++]),
            PApplet.parseInt(tokens[c++])
          );
        } else if ("e".equals(command)) {
          universe.explosion(
            PApplet.parseInt(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++])
          );
        }
      }
    } else if ("a".equals(command)) {       
      universe.asteroid(
          PApplet.parseInt(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseBoolean(tokens[c++]),
          PApplet.parseBoolean(tokens[c++]),
          PApplet.parseBoolean(tokens[c++]),
          PApplet.parseInt(tokens[c++])
        );
    } else if ("s".equals(command)) {
        universe.ship(
          PApplet.parseInt(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseBoolean(tokens[c++]),
          PApplet.parseBoolean(tokens[c++]),
          PApplet.parseBoolean(tokens[c++]),
          PApplet.parseInt(tokens[c++])
        );
      } else if ("b".equals(command)) {
        universe.bullet(
          PApplet.parseInt(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseFloat(tokens[c++]),
          PApplet.parseInt(tokens[c++]),
          PApplet.parseInt(tokens[c++])
        );
      } else if ("e".equals(command)) {
          universe.explosion(
            PApplet.parseInt(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++]),
            PApplet.parseFloat(tokens[c++])
          );
        }
  } 
}
abstract class Scene {
  protected PApplet applet;
  
  public Scene(PApplet applet) {
    this.applet = applet;
  }
  
  public abstract void clean();
  public abstract void setup();
  public abstract void draw();
  public void mousePressed() {}
  public void keyReleased() {}
  public void keyPressed() {}
  public void controlEvent(ControlEvent cevent) { }
  public void serverEvent(Server server, Client client) { }
  public void disconnectEvent(Client client) { }
}
class Universe {
  
  final int MAX_SHIP = 4;
  
  Ship[] ships = new Ship[MAX_SHIP];
  Bullet[] bullets = new Bullet[MAX_SHIP * 4];
  Explosion[] explosions = new Explosion[MAX_SHIP * 4];
  Asteroid[] asteroids = new Asteroid[10];

  public Universe() {
    for( int b = 0; b < bullets.length; ++b) {
      bullets[b] = new Bullet();
      bullets[b].id = b;
    }
    for( int a = 0; a < asteroids.length; ++a) {
      asteroids[a] = new Asteroid();
      asteroids[a].id = a;
    }
    for( int e = 0; e < explosions.length; ++e) {
      explosions[e] = new Explosion();
      explosions[e].id = e;
    }
    for( int s = 0; s < ships.length; ++s) {
      ships[s] = new Ship();
      ships[s].id = s;
    }
  }
  
  public void update(float elapsed) {
    for( int b = 0; b < bullets.length; ++b) {
      if (bullets[b].visible) bullets[b].update(elapsed);
    }
    for( int a = 0; a < asteroids.length; ++a) {
      if (asteroids[a].visible) asteroids[a].update(elapsed);
    }
    for( int s = 0; s < ships.length; ++s) {
      if (ships[s].visible) ships[s].update(elapsed);
    }
    for( int e = 0; e < explosions.length; ++e) {
      if (explosions[e].visible) explosions[e].update(elapsed);
    }
  }
  
  public void draw() {
    background(0);  
    for( int b = 0; b < bullets.length; ++b) {
      if (bullets[b].visible) bullets[b].draw();
    }
    for( int a = 0; a < asteroids.length; ++a) {
      if (asteroids[a].visible) asteroids[a].draw();
    }
    for( int s = 0; s < ships.length; ++s) {
      if (ships[s].visible) ships[s].draw();
    }
    for( int e = 0; e < explosions.length; ++e) {
      if (explosions[e].visible) explosions[e].draw();
    }
  }
  
  public void destroy(int type, int id) {
    if (type == GameObject.TYPE_BULLET) {
      bullets[id].visible = false;
    }
    
    if (type == GameObject.TYPE_EXPLOSION) {
      explosions[id].visible = false;
    }
    
    if (type == GameObject.TYPE_ASTEROID) {
      asteroids[id].visible = false;
    }
    
    if (type == GameObject.TYPE_SHIP) {
      ships[id].visible = false;
    }
  }
  
  public Bullet bullet(int id, float px, float py, float fx, float fy, float life, int ec, int sid) {
    bullets[id].sync(px, py, fx, fy, life, ec, sid);
    bullets[id].visible = true;
    return bullets[id];
  }
  
  public Bullet bullet(float px, float py, float fx, float fy, float life, int ec, int sid) {
    for( int b = 0; b < bullets.length; ++b) {
      if (!bullets[b].visible) {
        bullets[b].sync(px, py, fx, fy, life, ec, sid);
        bullets[b].visible = true;
        return bullets[b];
      }
    }
    
    return null;
  }
  
  public Explosion explosion(int id, float px, float py, float f) {
    explosions[id].sync(px, py, f);
    explosions[id].visible = true;
    return explosions[id];
  }
  
  public Explosion explosion(float px, float py, float f) {
    for( int e = 0; e < explosions.length; ++e) {
      if (!explosions[e].visible) {
        explosions[e].sync(px, py, f);
        explosions[e].visible = true;
        return explosions[e];
      }
    }
    
    return null;
  }
  
  public Ship ship(int id, float px, float py, float fx, float fy, float size, float d, float r, boolean acc, boolean tl, boolean tr, int ec) {
    ships[id].sync(px, py, fx, fy, size, d, r, acc, tl, tr, ec);
    ships[id].visible = true;
    return ships[id]; 
  }
  
  public Ship ship(float px, float py, float fx, float fy, float size, float d, float r, boolean a, boolean tl, boolean tr, int ec) {
    for( int s = 0; s < ships.length; ++s) {
      if (!ships[s].visible) {
        ships[s].sync(px, py, fx, fy, size, d, r, a, tl, tr, ec);
        ships[s].visible = true;
        return ships[s];
      }
    }
    
    return null;
  }
  
  public Asteroid asteroid(int id, float px, float py, float fx, float fy, float size, float d, float r, boolean acc, boolean tl, boolean tr, int ec) {
    asteroids[id].sync(px, py, fx, fy, size, d, r, acc, tl, tr, ec);
    asteroids[id].visible = true;
    return asteroids[id]; 
  }
  
  public Asteroid asteroid(float px, float py, float fx, float fy, float size, float d, float r, boolean acc, boolean tl, boolean tr, int ec) {
    for( int a = 0; a < asteroids.length; ++a) {
      if (!asteroids[a].visible) {
        asteroids[a].sync(px, py, fx, fy, size, d, r, acc, tl, tr, ec);
        asteroids[a].visible = true;
        return asteroids[a];
      }
    }
    
    return null;
  }
  
  public void updateObject(GameObject go) {
  }
}
public void forceLimit(float[] force, float limit) {
  float fl = sqrt(force[0] * force[0] + force[1] * force[1]); 
  if (fl > limit) {
    force[0] = (force[0] / fl) * limit;
    force[1] = (force[1] / fl) * limit;
  }
}

public void wrap(float[] vector, float size) {
  if (vector[0] - size <= 0) vector[0] = width - size - 1;
  if (vector[1] - size <= 0) vector[1] = height - size - 1;
  if (vector[0] + size >= width) vector[0] = size + 1;
  if (vector[1] + size >= height) vector[1] = size + 1;
}

public void drawPolygon(float[][] polygon) {
  int pl = polygon.length;
  
  if (pl == 0) {
    return;
  }
  
  if (polygon.length == 1) {
    point(polygon[0][0], polygon[0][1]);
    return;
  } 
  
  for(int i = 1; i < pl; ++i) {
    line(polygon[i - 1][0], polygon[i - 1][1], polygon[i][0], polygon[i][1]);
  }
  
  if (pl > 2) {
    line(polygon[pl - 1][0], polygon[pl - 1][1], polygon[0][0], polygon[0][1]);
  }
}

public boolean pointInPolygon(float[] point, float[][] polygon) {
  return pointInPolygon(point[0], point[1], polygon);
}

public boolean pointInPolygon(float px, float py, float[][] polygon) {
  // http://stackoverflow.com/questions/11716268/point-in-polygon-algorithm
  
  int i, j, nvert = polygon.length;
  boolean c = false;

  for(i = 0, j = nvert - 1; i < nvert; j = i++) {
    if( ( (polygon[i][1] >= py ) != (polygon[j][1] >= py) ) &&
        (px <= (polygon[j][0] - polygon[i][0]) * (py - polygon[i][1]) / (polygon[j][1] - polygon[i][1]) + polygon[i][0])
      )
      c = !c;
  }

  return c;
}

public void to_world(float[] result, float rotate, float scale, float[] position, float[] point) {
  result[0] = (point[0] * cos(rotate) - point[1] * sin(rotate)) * scale + position[0];
  result[1] = (point[1] * cos(rotate) + point[0] * sin(rotate)) * scale + position[1];
}

public float[] x_rotate(float[] point, float angle)
{
  float[] result = new float[2];
  
  result[0] = point[0] * cos(angle) - point[1] * sin(angle);
  result[1] = point[1] * cos(angle) + point[0] * sin(angle);
  
  return result;
}
  public void settings() {  size(640, 360); }
  static public void main(String[] passedArgs) {
    String[] appletArgs = new String[] { "sketch_150911b" };
    if (passedArgs != null) {
      PApplet.main(concat(appletArgs, passedArgs));
    } else {
      PApplet.main(appletArgs);
    }
  }
}
