#include "Map.h"
#include "GameManager.h"
#include "MapEvent.h"
#include "OdinAI/AStarGraphSearch.h"
#include "PathValidation.h"
#include "ManhattanFunction.h"
#include <algorithm>

void Map::Init(int sizeX, int sizeY, char *map)
{
    m_sizeX = sizeX;
    m_sizeY = sizeY;

    // Be sure to delete old map data.
    m_map.clear();
    m_map.resize(sizeX);
    for(int i = 0;i < m_map.size();++i)
    {
        m_map[i].resize(sizeY);
    }
    
    m_graph.Clear();
    for(int y = 0;y < m_sizeY;++y)
    {
        for(int x = 0;x < m_sizeX;++x)
        {
            m_map[x][y] = map[y * m_sizeX + x];
            if(m_map[x][y] == static_cast<char>(Tile::Stone))
            {
                ++m_stoneCount;
            }

            int nodeIdx = GetNodeIdx(iVec2(x, y));
            m_graph.AddNode(DNBNode(nodeIdx, iVec2(x, y), m_map[x][y]));
        }
    }

    for(int i = 0;i < m_sizeX * m_sizeY;++i)
    {
        UpdateGraphNode(i);
    }
}

void Map::Update(char *map)
{
    auto &playerMap = gGameMgr.GetPlayerMgr().GetPlayers();
    for(auto iter = playerMap.begin();iter != playerMap.end();++iter)
    {
        if(!iter->second.IsDead())
        {
            const iVec2 &pos = iter->second.GetPosition();
            map[pos.y * m_sizeX + pos.x] = static_cast<char>(Tile::Player);
        }
    }

    for(int y = 0;y < m_sizeY;++y)
    {
        for(int x = 0;x < m_sizeX;++x)
        {
            if(m_map[x][y] != map[y * m_sizeX + x])
            {
                if(m_map[x][y] == static_cast<char>(Tile::Stone))
                {
                    --m_stoneCount;
                }
                m_map[x][y] = map[y * m_sizeX + x];
            }
        }
    }

    gGameMgr.GetBombMgr().FillMap(&m_map);
    auto &bombs = gGameMgr.GetBombMgr().GetBombs();
    for(auto iter = bombs.begin();
        iter != bombs.end();++iter)
    {
        if(gGameMgr.GetPlayerMgr().GetOurPlayer()->GetPosition() != iter->position)
        {
            m_map[iter->position.x][iter->position.y] = static_cast<char>(Tile::Bomb);
        }
    }

    for(int y = 0;y < m_sizeY;++y)
    {
        for(int x = 0;x < m_sizeX;++x)
        {
            int nodeIdx = GetNodeIdx(iVec2(x, y));
            m_graph.GetNode(nodeIdx).SetTile(m_map[x][y]);
        }
    }

    /*for(int y = 0;y < m_sizeY;++y)
    {
        for(int x = 0;x < m_sizeX;++x)
        {
            if(m_map[x][y] < 32)
                std::cout << (int)m_map[x][y];
            else
                std::cout << m_map[x][y];
        }
        std::cout << std::endl;
    }*/

    auto mapChangedEvt = MapChangedEvent::Create();
    mapChangedEvt.get()->Init(&m_map);
    gGameMgr.GetEventMgr().FireEvent(mapChangedEvt);
}

void Map::UpdateGraphNode(int nodeIdx)
{
    iVec2 gridPos = GetPosFromNodeIdx(nodeIdx);

    int tileCost = GetTileCost(m_map[gridPos.x][gridPos.y]);
    if(m_map[gridPos.x][gridPos.y] != static_cast<char>(Tile::Wall))
    {
        int nextTileCost;
        if(gridPos.x > 0 && m_map[gridPos.x-1][gridPos.y] != static_cast<char>(Tile::Wall))
        {
            nextTileCost = GetTileCost(m_map[gridPos.x-1][gridPos.y]);
            OdinAI::GraphEdge edge(nodeIdx, GetNodeIdx(iVec2(gridPos.x-1, gridPos.y)), std::max<int>(nextTileCost, tileCost));
            m_graph.AddEdge(edge);
        }

        if(gridPos.y > 0 && m_map[gridPos.x][gridPos.y-1] != static_cast<char>(Tile::Wall))
        {
            nextTileCost = GetTileCost(m_map[gridPos.x][gridPos.y-1]);
            OdinAI::GraphEdge edge(nodeIdx, GetNodeIdx(iVec2(gridPos.x, gridPos.y-1)), std::max<int>(nextTileCost, tileCost));
            m_graph.AddEdge(edge);
        }

        if(gridPos.x + 1 < m_sizeX && m_map[gridPos.x+1][gridPos.y] != static_cast<char>(Tile::Wall))
        {
            nextTileCost = GetTileCost(m_map[gridPos.x+1][gridPos.y]);
            OdinAI::GraphEdge edge(nodeIdx, GetNodeIdx(iVec2(gridPos.x+1, gridPos.y)), std::max<int>(nextTileCost, tileCost));
            m_graph.AddEdge(edge);
        }

        if(gridPos.y + 1 < m_sizeY && m_map[gridPos.x][gridPos.y+1] != static_cast<char>(Tile::Wall))
        {
            nextTileCost = GetTileCost(m_map[gridPos.x][gridPos.y+1]);
            OdinAI::GraphEdge edge(nodeIdx, GetNodeIdx(iVec2(gridPos.x, gridPos.y + 1)), std::max<int>(nextTileCost, tileCost));
            m_graph.AddEdge(edge);
        }
    }
}

std::list<int> Map::GetPath(const iVec2 &from, const iVec2 &to) const
{
    std::list<int> path;
    OdinAI::AStarGraphSearch<DNBGraph, ManhattanFunction, SafePathValidation> search(m_graph, GetNodeIdx(from), GetNodeIdx(to));
    search.GetPath(path);

    return path;
}

const std::vector< std::vector<char> > &Map::GetMap() const
{
    return m_map;
}

bool Map::IsTileStandAble(char tile) const
{
    return tile == static_cast<char>(Tile::Grass) || tile == static_cast<char>(Tile::Debris);
}

int Map::GetNumStones() const
{
    return m_stoneCount;
}

bool Map::CanTileBeDestroyed(char tile) const
{
    return tile == static_cast<char>(Tile::Stone);
}

int Map::GetTileCost(char tile) const
{
    return 1;
}

char Map::GetTile(int nodeIdx) const
{
    iVec2 tilePos = GetPosFromNodeIdx(nodeIdx);
    return m_map[tilePos.x][tilePos.y];
}

int Map::GetSizeX() const
{
    return m_sizeX;
}

int Map::GetSizeY() const
{
    return m_sizeY;
}

int Map::GetNodeIdx(const iVec2 &position) const
{
    return (position.y * m_sizeX) + position.x;
}

iVec2 Map::GetPosFromNodeIdx(int nodeIdx) const
{
    return iVec2(nodeIdx % m_sizeX, nodeIdx / m_sizeX);
}

const OdinAI::SparseGraph<DNBNode, OdinAI::GraphEdge> &Map::GetGraph() const
{
    return m_graph;
}