#ifndef DNB_MAP_H_
#define DNB_MAP_H_

#include <vector>
#include "Vector2.h"
#include "OdinAI/SparseGraph.h"
#include "DNBNode.h"
#include "OdinAI/GraphEdges.h"

enum class Tile : char {
   Grass = '.',
   Floor = '_',
   Wall = '+',
   Stone = '#',
   Debris = ' ',
   Bomb = 'B',
   Player = 'P',
   Invalid = 'x'
};

typedef OdinAI::SparseGraph<DNBNode, OdinAI::GraphEdge> DNBGraph;

/**
 * Map manages what the AI can see, and where the AI can move.
 */
class Map
{
public:
    Map() : m_graph(false), m_stoneCount(0), m_sizeX(0), m_sizeY(0) {}

    /**
     * Initialize the map instance.
     * @param sizeX The map extent on the x axe.
     * @param sizeY The map extent on the y axe.
     */
    void Init(int sizeX, int sizeY, char *map);

    /**
     * Update map data
     * @param map Map data stored in a big string.
     */
    void Update(char *map);

    /**
     * @return Get node index from position.
     */
    int GetNodeIdx(const iVec2 &position) const;

    /**
     * @return Get grid position from node index.
     */
    iVec2 GetPosFromNodeIdx(int nodeIdx) const;

    /**
     * Checks if a tile is standable.
     */
    bool IsTileStandAble(char tile) const;

    /**
     * Checks if a tile can be destroyed, even if it is not standable.
     */
    bool CanTileBeDestroyed(char tile) const;

    /**
     * Get a path between two positions. Empty list returned if no path found.
     */
    std::list<int> GetPath(const iVec2 &from, const iVec2 &to) const;

    /**
     * @return The cost of traversing to this tile. Returns -1 if it is impossible to traverse this type of tile.
     */
    int GetTileCost(char tile) const;

    char GetTile(int nodeIdx) const;

    /**
     * @return Map extent on the x axe.
     */
    int GetSizeX() const;

    /**
     * @return Map extent on the y axe.
     */
    int GetSizeY() const;

    /**
     * @return The number of stones.
     */
    int GetNumStones() const;

    /**
     * @return The sparse graph for the map.
     */
    const DNBGraph &GetGraph() const;

    const std::vector< std::vector<char> > &GetMap() const;
private:
    /**
     * Update our graph.
     */
    void UpdateGraphNode(int nodeIdx);

    std::vector< std::vector<char> >m_map;

    DNBGraph m_graph; //!< Our graph system.

    int m_stoneCount;
    int m_sizeX;
    int m_sizeY;
};

#endif