/*******************************************************************************
 * ________       .___.__           _____  .___ 
 * \_____  \    __| _/|__| ____    /  _  \ |   |
 *  /   |   \  / __ | |  |/    \  /  /_\  \|   |
 * /    |    \/ /_/ | |  |   |  \/    |    \   |
 * \_______  /\____ | |__|___|  /\____|__  /___|
 *         \/      \/         \/         \/     
 *
 * Copyright (c) Emil Sandst 2012
 *******************************************************************************/
#ifndef ODINAI_GOALS_H_
#define ODINAI_GOALS_H_

#ifdef _WINDOWS
#pragma once
#endif

#include <stack>

namespace OdinAI
{
	enum Goal_Status
	{
		GOAL_FAILED,
		GOAL_COMPLETE,
		GOAL_ACTIVE,
		GOAL_INACTIVE,
	};

	/**
	 * Abstract class for a AI goal.
	 */
	class BaseGoal
	{
	public:
		BaseGoal() {}

		virtual ~BaseGoal() {}

		/**
		 * Adds a goal to a composite goal. 
		 * If this function is called on a atomic goal, it will cause an assertion.
		 */
		virtual void AddGoal(BaseGoal *goal) {}

		/**
		 * Function called before a goal is processed.
		 */
		virtual void Activate() {m_status = GOAL_ACTIVE;}

		/**
		 * Function used to process an goal.
		 */
		virtual int Process() {return GOAL_FAILED;}

		/**
		 * Function used to clean up an goal.
		 */
		virtual void End() {}

		/**
		 * Is goal active?
		 */
		bool IsActive() const {return m_status == GOAL_ACTIVE;}

		/**
		 * Activate goal if inactive.
		 * It will also activate the goal if the status is ready or failed.
		 */
		void ActivateIfInactive() {if(m_status != GOAL_ACTIVE)Activate();}

		/**
		 * Returns true if goal is complete.
		 */
		bool IsComplete() const {return m_status == GOAL_COMPLETE;}

		/**
		 * Returns true if goal has failed.
		 */
		bool HasFailed() const {return m_status == GOAL_FAILED;}

		/**
		 * Returns the status of the goal.
		 */
		int GetStatus() const {return m_status;}

	protected:
		int m_status;//!< The status of the goal.

        friend class CompositeGoal;
	};

	/**
	 * Atomic goal consist of one goal, which can be processed.
	 */
	class AtomicGoal : public BaseGoal
	{
	public:
		virtual ~AtomicGoal() {}

		/**
		 * This function is not implemented an will cause an assertion.
		 */
		void AddGoal(BaseGoal *goal);

		/**
		 * Function called before a goal is processed.
		 */
		virtual void Activate() {}

		/**
		 * Function used to process an goal.
		 */
		virtual int Process() {return GOAL_FAILED;}

		/**
		 * Function used to clean up an goal.
		 */
		virtual void End() {}
	};

	/**
	 * Composite goal consist of more than one goal.
	 */
	class CompositeGoal : public BaseGoal
	{
	public:
		CompositeGoal() {}

		virtual ~CompositeGoal();

		/**
		 * Adds a goal to a composite goal. 
		 */
		virtual void AddGoal(BaseGoal *goal);

		/**
		 * Function called before a goal is processed.
		 */
		virtual void Activate() {}

		/**
		 * Function used to process goal.
		 */
		virtual int Process() {return GOAL_FAILED;}

		/**
		 * Function used to process the subgoals
		 */
		virtual int ProcessSubgoals();

		/**
		 * Function used to clean up an goal.
		 */
		virtual void End() {}

		void Clear();
	protected:
		std::stack<BaseGoal*> m_stackOfGoals;
	};

}

#endif