#include <arpa/inet.h>
#include <assert.h>
#include <fcntl.h>
#include <jansson.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>

#include "ai.h"
#include "main.h"
#include "map.h"
#include "movement.h"
#include "player.h"
#include "weapon.h"
#include "minimax.h"

#define BUFFER_SIZE 65536

int state;
int our_turn;

int sock;
FILE *stream;

void send_handshake()
{
    fprintf(stream, "{"
        "\"message\":\"connect\","
        "\"revision\": 1,"
        "\"name\": \""NAME"\""
        "}\n");
    fflush(stream);
}

void send_loadout()
{
    int weapons = get_weapon_selection();
    int primary_weapon = weapons >> 8,
        secondary_weapon = weapons & 0xff;
    fprintf(stream, "{"
        "\"message\":\"loadout\","
        "\"primary-weapon\": \"%s\","
        "\"secondary-weapon\": \"%s\""
        "}\n",
        weapon_to_string(primary_weapon),
        weapon_to_string(secondary_weapon));
    fflush(stream);
}

void send_action()
{
    alphabeta(0, INT_MIN, INT_MAX);
    action_send(selected_act, selected_param);
#if 0
    player_t *enemy = players;
    if (enemy == me) {
        enemy = enemy->next;
    }
    if (rand() % 5 == 0) {
        /* Fire. */
        dir_t dir = laser_direction(
            me->weapons[0],
            map_coords(me->j, me->k),
            map_coords(enemy->j, enemy->k));
        debug("Firing laser at %s\n", dir_to_string(dir));
        fprintf(stream, "{"
            "\"message\":\"action\","
            "\"type\": \"laser\","
            "\"direction\": \"%s\""
            "}\n",
            dir_to_string(dir));
        fflush(stream);
    } else {
        /* Move. */
        movement_t movement = movement_towards(
            map_coords(me->j, me->k),
            map_coords(enemy->j, enemy->k));
        debug("Directions to enemy is %s %s %s\n",
            dir_to_string(movement.moves[0]),
            dir_to_string(movement.moves[1]),
            dir_to_string(movement.moves[2]));
        if (movement.moves[0] == dir_none) {
            movement.moves[0] = (dir_t)(rand() % 6);
        }
        for (int i = 0; i < 3; i++) {
            dir_t dir = movement.moves[i];
            if (dir == dir_none) {
                break;
            }
            debug("Moving to %s\n", dir_to_string(dir));
            fprintf(stream, "{"
                "\"message\":\"action\","
                "\"type\": \"move\","
                "\"direction\": \"%s\""
                "}\n",
                dir_to_string(dir));
            fflush(stream);
        }
    }
#endif
}

void update_gamestate_from_json(int initial, json_t *json)
{
    json_t *map_json = json_object_get(json, "map");
    json_t *players_json = json_object_get(json, "players");
    update_map_from_json(initial, map_json);

    our_turn = 0;

    for (int i = 0; i < json_array_size(players_json); i++) {
        json_t *player_json = json_array_get(players_json, i);
        int player_is_us = update_player_from_json(initial, player_json);

        if (i == 0 && player_is_us) {
            our_turn = 1;
        }
    }
}

void fsm()
{
    switch (state) {
        case STATE_SEND_HANDSHAKE:
            send_handshake();
            state = STATE_WAIT_CONNECT;
            return;

        case STATE_SEND_LOADOUT:
            send_loadout();
            state = STATE_WAIT_GAMESTATE;
            return;

        case STATE_SEND_ACTION:
            send_action();
            state = STATE_WAIT_GAMESTATE;
            return;
    }

    char *buffer = malloc(sizeof(char) * BUFFER_SIZE);

    int cur = 0;
    while (read(sock, &buffer[cur], 1) == 1) {
        if (buffer[cur++] == '\n') {
            break;
        }
    }
    buffer[cur] = 0;
    printf("%s", buffer);

    json_error_t json_error;
    json_t *message = json_loads(buffer,
        JSON_DISABLE_EOF_CHECK, &json_error);
    assert(message);
    assert(json_is_object(message));

    json_t *message_type_json = json_object_get(message, "message");
    if (message_type_json == 0) {
        /* Error message -- ignore by returning early. */
        debug("Ignoring error\n");
        return;
    }
    assert(json_is_string(message_type_json));

    const char *message_type = json_string_value(message_type_json);
    debug("%s\n", message_type);

    switch (state) {
        case STATE_WAIT_CONNECT:
            assert(strcmp(message_type, "connect") == 0);
            debug("Received connect\n");
            state = STATE_WAIT_GAMESTART;
            break;

        case STATE_WAIT_GAMESTART:
            assert(strcmp(message_type, "gamestate") == 0);
            debug("Received gamestart\n");
            update_gamestate_from_json(1, message);
            state = STATE_SEND_LOADOUT;
            break;

        case STATE_WAIT_GAMESTATE:
            if (strcmp(message_type, "action") == 0 ||
                strcmp(message_type, "endturn") == 0) {
                debug("Ignoring %s\n", message_type);
                break;
            }
            assert(strcmp(message_type, "gamestate") == 0);
            debug("Received gamestate\n");
            update_gamestate_from_json(0, message);
            if (our_turn) {
                state = STATE_SEND_ACTION;
            }
            break;

        default:
            fprintf(stderr, "Unknown state, quitting\n");
            state = STATE_QUIT;
            break;
    }

    /* TODO: Free buffer and json data. */
}

int main(int argc, char **argv)
{
    /*map_j = 100; map_k = 100;
    printf("%d %d %f\n", 1, 0,   laser_damage(1, map_coords(5, 5), map_coords( 10,  5)));
    printf("%d %d %f\n", -1, 0,  laser_damage(1, map_coords(5, 5), map_coords(0,  5)));
    printf("%d %d %f\n", 0, 1,   laser_damage(2, map_coords(5, 5), map_coords( 5,  10)));
    printf("%d %d %f\n", 0, -1,  laser_damage(2, map_coords(5, 5), map_coords( 5, 0)));
    printf("%d %d %f\n", 1, 1,   laser_damage(3, map_coords(5, 5), map_coords( 10,  10)));
    printf("%d %d %f\n", -1, -1, laser_damage(3, map_coords(5, 5), map_coords(0, 0)));*/

    struct sockaddr_in sa;
    sa.sin_family = AF_INET;
    sa.sin_port = htons(54321);
    inet_pton(AF_INET, "127.0.0.1", &sa.sin_addr);

    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (connect(sock, (struct sockaddr *)&sa,
            sizeof(struct sockaddr_in)) != 0) {
        fprintf(stderr, "Couldn't connect\n");
        return 1;
    }

    stream = fdopen(sock, "w+");
    state = STATE_SEND_HANDSHAKE;

    while (state != STATE_QUIT) {
        fsm();
    }

    close(sock);
    return 0;
}
