/*
 * Functions for reading and converting various restricted
 * C64/etc and/or indexed/paletted graphics formats.
 * Programmed and designed by Matti 'ccr' Hamalainen
 * (C) Copyright 2012 Tecnic Software productions (TNSP)
 *
 * Please read file 'COPYING' for information on license and distribution.
 */
#ifndef LIB64GFX_H
#define LIB64GFX_H 1

#include "libgfx.h"

#ifdef __cplusplus
extern "C" {
#endif


// Bitmap constants
#define C64_SCR_WIDTH          320
#define C64_SCR_HEIGHT         200
#define C64_SCR_CH_WIDTH       (C64_SCR_WIDTH/8)
#define C64_SCR_CH_HEIGHT      (C64_SCR_HEIGHT/8)
#define C64_SCR_COLOR_SIZE     (C64_SCR_CH_WIDTH * C64_SCR_CH_HEIGHT)
#define C64_SCR_SCREEN_SIZE    (C64_SCR_CH_WIDTH * C64_SCR_CH_HEIGHT)
#define C64_SCR_BITMAP_SIZE    (C64_SCR_WIDTH * C64_SCR_HEIGHT/8)
#define C64_SCR_EXTRADATA      1024
#define C64_SCR_MAX_BANK       8

// C64 video screen pixel aspect ratio on PAL
#define C64_SCR_PAR_XY         (0.9365f)

// Sprite constants
#define C64_SPR_WIDTH          3 // bytes
#define C64_SPR_HEIGHT         21 // lines
#define C64_SPR_WIDTH_PX       (8 * C64_SPR_WIDTH)
#define C64_SPR_SIZE           ((C64_SPR_WIDTH * C64_SPR_HEIGHT) + 1)

// Character constants
#define C64_CHR_WIDTH          1 // bytes
#define C64_CHR_HEIGHT         8 // lines
#define C64_CHR_WIDTH_PX       (8 * C64_CHR_WIDTH)
#define C64_CHR_SIZE           (C64_CHR_WIDTH * C64_CHR_HEIGHT)

// Etc.
#define C64_RAM_SIZE           (64*1024)
#define C64_NCOLORS            16
#define C64_MAX_COLORS         16
#define C64_VIDBANK_SIZE       (16*1024)
#define C64_MAX_SPRITES        (C64_VIDBANK_SIZE / C64_SPR_SIZE)
#define C64_MAX_CHARS          256

// Different supported C64 bitmap "modes"
enum
{
    D64_FMT_HIRES        = 0x0000,
    D64_FMT_MC           = 0x0001,
    D64_FMT_ILACE        = 0x0002,
    D64_FMT_FLI          = 0x0004,

    D64_FMT_MODE_MASK    = 0x000f,

};

enum
{
    D64_FLI_2BANK,
    D64_FLI_4BANK,
    D64_FLI_8BANK,
};

enum
{
    D64_ILACE_COLOR,
    D64_ILACE_RES,
};

typedef struct
{
    BOOL multicolor, xexpand, yexpand;
    int color, xc, yc;
    Uint8 data[C64_SPR_HEIGHT][C64_SPR_WIDTH];
} DMC64Sprite;

enum
{
    D64_CHR_GLOBAL,		// use font-global setting
    D64_CHR_MULTICOLOR,	// character is multicolor
    D64_CHR_HIRES,
};

typedef struct
{
    int mode, color;
    Uint8 data[C64_CHR_HEIGHT];
} DMC64Char;


typedef struct
{
    BOOL multicolor;
    int colbg, color, col1, col2;
    int nglyphs;
    DMC64Char *glyphs;
} DMC64Font;



typedef struct
{
    int type,     // Image type (D64_FMT_*)
        fliType,  // FLI type (if FLI used)
        laceType, // Interlace type (D64_ILACE_*)
        laceBank1,
        laceBank2;

    Uint8 color[C64_SCR_MAX_BANK][C64_SCR_COLOR_SIZE],
            bitmap[C64_SCR_MAX_BANK][C64_SCR_BITMAP_SIZE],
            screen[C64_SCR_MAX_BANK][C64_SCR_SCREEN_SIZE],
            extradata[C64_SCR_EXTRADATA],
            d020, bgcolor, d022, d023, d024;

    Uint8 charset[C64_MAX_CHARS][C64_CHR_HEIGHT * C64_CHR_WIDTH];
    DMC64Sprite sprites[C64_MAX_SPRITES];
} DMC64Image;


enum
{
    DT_COLOR_RAM,
    DT_BITMAP,
    DT_SCREEN_RAM,
    DT_BGCOLOR,
    DT_EXTRADATA,
    
    DT_DEC_FUNCTION,
    DT_ENC_FUNCTION,
    
    DT_LAST,
};


typedef struct _DMC64EncDecOp
{
    int    type;
    size_t offs;
    int    bank;
    size_t size;
    BOOL   (*decfunction)(DMC64Image *img, const struct _DMC64EncDecOp *op, const Uint8 *buf, const size_t len);
    BOOL   (*encfunction)(const struct _DMC64EncDecOp *op, Uint8 **buf, size_t *len, const DMC64Image *img);
} DMC64EncDecOp;


typedef struct _DMC64ImageFormat
{
    int  type;
    char *fext;
    char *name;
    size_t addr; // Loading address (0 if no loading address)
    size_t size; // Size, including loading address. Only used in encoding, if even there (0 if no static size)
    int  (*probe)(const Uint8 *buf, const size_t len, const struct _DMC64ImageFormat *fmt);
    int  (*decode)(DMC64Image *img, const Uint8 *buf, const size_t len, const struct _DMC64ImageFormat *fmt);
    int  (*encode)(DMC64Image *img, Uint8 **buf, size_t *len, const struct _DMC64ImageFormat *fmt);
    int  (*convertFrom)(DMImage *, const DMC64Image *, const BOOL doubleMC);
    int  (*convertTo)(DMImage *, DMC64Image *);

    int nencdecOps;
    DMC64EncDecOp encdecOps[16];
} DMC64ImageFormat;


extern const size_t      dmC64DefaultSizes[DT_LAST];
extern DMColor           dmC64Palette[C64_NCOLORS];
extern const DMC64ImageFormat  dmC64ImageFormats[];
extern const int         ndmC64ImageFormats;


char *    dmC64GetImageTypeString(char *buf, const size_t len, const int type);
int       dmC64ConvertCSData(DMImage *img, int xoffs, int yoffs, const Uint8 *inBuf, int width, int height, BOOL multicolor, int *colors);

int       dmC64DecodeGenericBMP(DMC64Image *img, const Uint8 *buf, const size_t len, const DMC64ImageFormat *fmt);
int       dmC64EncodeGenericBMP(Uint8 **pbuf, size_t *plen, const DMC64Image *img, const DMC64ImageFormat *fmt);
int       dmC64ConvertGenericBMP2Image(DMImage *dst, const DMC64Image *src, const BOOL doubleMC);

int       dmC64ConvertBMP2Image(DMImage **pdst, const DMC64Image *src, const DMC64ImageFormat *fmt, const BOOL doubleMC);
int       dmC64ProbeBMP(const Uint8 *buf, const size_t len, const DMC64ImageFormat **fmt);
int       dmC64DecodeBMP(DMC64Image *img, const Uint8 *buf, const size_t len, const size_t probeOffs, const size_t loadOffs, const DMC64ImageFormat **fmt, const DMC64ImageFormat *forced);


#ifdef __cplusplus
}
#endif

#endif // LIB64GFX_H
