#define GL_GLEXT_PROTOTYPES
#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <GL/gl.h>
#include <math.h>

#include "pdb.h"

int loadPDB(PDB *pdb, const char *filename){
   FILE *pdbfile;
   char *line = NULL;
   size_t linelen = 0;
   char *ptr;
   int i;
   int counter = 0;
   ATOM atom;
   double meanx = 0, meany = 0, meanz = 0;

   pdbfile = fopen(filename, "r");
   if(pdbfile == NULL) goto fileerror;

   pdb->atoms = malloc(sizeof(ATOM) * 10000); /* Well, yes it is dangerous */
   if(pdb->atoms ==  NULL) goto memoerror;

   while( getline(&line, &linelen, pdbfile) != -1){
      if( (strncmp(line, "ATOM", 4) == 0) ||
          (strncmp(line, "HETATM", 6) == 0) ){
         strtok_r(line, " \n", &ptr);
         for(i = 0; i < 5; i++){
            strtok_r(NULL, " \n", &ptr);
         }
         atom.px = atof(strtok_r(NULL, " \n", &ptr));
         atom.py = atof(strtok_r(NULL, " \n", &ptr));
         atom.pz = atof(strtok_r(NULL, " \n", &ptr));

         meanx += atom.px;
         meany += atom.py;
         meanz += atom.pz;

         pdb->atoms[counter] = atom;
         counter++;
         pdb->atomnum = counter;
         if(counter == 10000) break;
      }
   }

   meanx = meanx / (double)counter;
   meany = meany / (double)counter;
   meanz = meanz / (double)counter;

   for(i = 0; i < counter; i++){
      pdb->atoms[i].px -= meanx;
      pdb->atoms[i].py -= meany;
      pdb->atoms[i].pz -= meanz;
   }

   pdb->vertices = NULL;
   pdb->normals  = NULL;
   pdb->colors   = NULL;
   pdb->indices  = NULL;
   pdb->vcount   = 0;
   pdb->ncount   = 0;
   pdb->ccount   = 0;
   pdb->icount   = 0;
   pdb->vmax     = 0;
   pdb->imax     = 0;

   fclose(pdbfile);
   free(line);
   return(PDBSUCCESS);

memoerror:
   if(line != NULL) free(line);
   return(PDBMEMOERROR);

fileerror:
   return(PDBFILEERROR);
}

void freePDB(PDB *pdb){
   pdb->atomnum = 0;
   free(pdb->atoms);
   free(pdb->vertices);
   free(pdb->normals);
   free(pdb->colors);
   pdb->atoms = NULL;
   glDeleteBuffers(4, pdb->vbo);
   glDeleteVertexArrays(1, &pdb->vao);
}

void lazyRealloc(PDB *pdb){
   pdb->vmax    += 300;
   pdb->vertices = realloc(pdb->vertices, sizeof(GLfloat) * pdb->vmax);
   pdb->normals  = realloc(pdb->normals, sizeof(GLfloat)  * pdb->vmax);
   pdb->colors   = realloc(pdb->colors, sizeof(GLfloat)   * pdb->vmax);
   pdb->imax    += 600;
   pdb->indices  = realloc(pdb->indices, sizeof(GLuint)   * pdb->imax);
}

void generateSpheres(PDB *pdb, float posx, float posy, float posz, float r, float step){
   float phi, theta;
   /* Vertices */
   for(theta = 0.0; theta < M_PI; theta += step){
      for(phi = M_PI * -1; phi < M_PI; phi += step){

         pdb->vertices[pdb->vcount++] = posx + r * sinf(theta) * cosf(phi);
         pdb->vertices[pdb->vcount++] = posy + r * sinf(theta) * sinf(phi);
         pdb->vertices[pdb->vcount++] = posz + r * cosf(theta);

         pdb->colors[pdb->ccount++] = 1.0;
         pdb->colors[pdb->ccount++] = 1.0;
         pdb->colors[pdb->ccount++] = 1.0;

         pdb->normals[pdb->ncount++] = sinf(theta) * cosf(phi);
         pdb->normals[pdb->ncount++] = sinf(theta) * sinf(phi);
         pdb->normals[pdb->ncount++] = cosf(theta);

         if(theta < M_PI - step){

            pdb->indices[pdb->icount++] = pdb->vcount / 3 - 1;
            pdb->indices[pdb->icount++] = pdb->vcount / 3;
            pdb->indices[pdb->icount++] = (pdb->vcount / 3) + trunc(2.0 * M_PI / step);

            pdb->indices[pdb->icount++] = pdb->vcount / 3 - 1;
            pdb->indices[pdb->icount++] = (pdb->vcount / 3) + trunc(2.0 * M_PI / step);
            pdb->indices[pdb->icount++] = (pdb->vcount / 3 - 1) + trunc(2.0 * M_PI / step);

         }
         if( (pdb->vcount + 1) > pdb->vmax){
            lazyRealloc(pdb);
         }
      }
   }
}

void generateMesh(PDB *pdb){
   unsigned int i;

   lazyRealloc(pdb);

   for(i = 0; i < pdb->atomnum; i++){
      generateSpheres(pdb, pdb->atoms[i].px, pdb->atoms[i].py, pdb->atoms[i].pz, 1.0, 0.5);
   }

   glGenVertexArrays(1, &pdb->vao);
   glBindVertexArray(pdb->vao);
   glGenBuffers(4, pdb->vbo);

   glBindBuffer(GL_ARRAY_BUFFER, pdb->vbo[0]);
   glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * pdb->vcount, pdb->vertices, GL_STATIC_DRAW);

   glBindBuffer(GL_ARRAY_BUFFER, pdb->vbo[1]);
   glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * pdb->ncount, pdb->normals, GL_STATIC_DRAW);

   glBindBuffer(GL_ARRAY_BUFFER, pdb->vbo[2]);
   glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * pdb->ccount, pdb->colors, GL_STATIC_DRAW);

   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, pdb->vbo[3]);
   glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(GLuint) * pdb->icount, pdb->indices, GL_STATIC_DRAW);

   glBindVertexArray(0);
}

void drawPDB(PDB pdb){
   glBindVertexArray(pdb.vao);
   glDrawElements(GL_TRIANGLES, pdb.icount, GL_UNSIGNED_INT, 0);
}

