#version 330 core

// Input
layout (location = 0) in vec4 in_Position;
layout (location = 1) in vec4 in_Color;
layout (location = 2) in vec3 in_Velocity;
layout (location = 3) in float in_Life;
layout (location = 4) in vec4 in_InitialPosition;

// Output
out vec4 out_Position;
out vec4 out_Color;
out vec3 out_Velocity;
out float out_Life;
out vec4 out_InitialPosition;

// Uniforms
uniform float dt;
uniform float t;

// Constants
const vec3 bbMin = vec3(-8.,  0., -8.);
const vec3 bbMax = vec3(+8., 16., +8.);

// Credit for noise/fbm functions goes to iq (http://www.iquilezles.org/)
float coolfFunc3d2( int n )
{
    n = (n << 13) ^ n;
    n = (n * (n * n * 15731 + 789221) + 1376312589) & 0x7fffffff;
    return float(n);
}

float noise3f( in vec3 p )
{
    ivec3 ip = ivec3(floor(p));
    vec3 u = fract(p);
    u = u*u*(3.0-2.0*u);

    int n = ip.x + ip.y*57 + ip.z*113;

    float res = mix(mix(mix(coolfFunc3d2(n+(0+57*0+113*0)),
                            coolfFunc3d2(n+(1+57*0+113*0)),u.x),
                        mix(coolfFunc3d2(n+(0+57*1+113*0)),
                            coolfFunc3d2(n+(1+57*1+113*0)),u.x),u.y),
                    mix(mix(coolfFunc3d2(n+(0+57*0+113*1)),
                            coolfFunc3d2(n+(1+57*0+113*1)),u.x),
                        mix(coolfFunc3d2(n+(0+57*1+113*1)),
                            coolfFunc3d2(n+(1+57*1+113*1)),u.x),u.y),u.z);

    return 1.0 - res*(1.0/1073741824.0);
}

float fbm( in vec3 p )
{
    return 0.5000*noise3f(p*1.0) +
           0.2500*noise3f(p*2.0) +
           0.1250*noise3f(p*4.0) +
           0.0625*noise3f(p*8.0);
}

vec3 getPotentialFieldValue(vec3 p)
{
	vec3 potential;

	//potential.x = fbm(p + (50. + 0.1 * t) * vec3(0.73));
	//potential.y = fbm(p + (50. + 0.1 * t) * vec3(0.65));
	//potential.z = fbm(p + (50. + 0.1 * t) * vec3(0.34));
	//potential.x = fbm(p + 41. + 0.1 * t);
	//potential.y = fbm(p + 79. + 0.1 * t);
	//potential.z = fbm(p + 17. + 0.1 * t);
	//potential.x = fbm(p + 79. * vec3(0.11) + 0.09 * vec3(t));
	//potential.y = fbm(p + 17. * vec3(0.65) + 0.09 * vec3(t));
	//potential.z = fbm(p + 49. * vec3(0.34) + 0.09 * vec3(t));
	potential.x = fbm(p + 50. * vec3(0.11) + 0.1 * vec3(t));
	potential.y = fbm(p + 50. * vec3(0.65) + 0.1 * vec3(t));
	potential.z = fbm(p + 50. * vec3(0.34) + 0.1 * vec3(t));

	return potential;
}

// Shift the hue by H radians using the YIQ space.
// See: http://beesbuzz.biz/code/hsv_color_transforms.php
vec3 shiftHue(vec3 color, const float H)
{
	const float U = cos(H);
	const float W = sin(H);

	const vec3 vR = vec3( .299 + .701 * U + .168 * W, .587 - .587 * U + .330 * W, .114 - .114 * U - .497 * W);
	const vec3 vG = vec3( .299 - .299 * U - .328 * W, .587 + .413 * U + .035 * W, .114 - .114 * U + .292 * W);
	const vec3 vB = vec3( .299 - .300 * U + 1.25 * W, .587 - .588 * U - 1.05 * W, .114 + .886 * U - .203 * W);

	return vec3(dot(color, vR), dot(color, vG), dot(color, vB));
}

void main()
{
	// sampling position
	vec3 samplePos = in_Position.xyz / 13.;

	// compute the velocity
	const float eps = 0.0001;
	const float oneOverTwoEps = 1. / (2. * eps);

	vec3 dPdx = getPotentialFieldValue(samplePos + vec3(eps, 0., 0.)) - getPotentialFieldValue(samplePos - vec3(eps, 0., 0.));
	vec3 dPdy = getPotentialFieldValue(samplePos + vec3(0., eps, 0.)) - getPotentialFieldValue(samplePos - vec3(0., eps, 0.));
	vec3 dPdz = getPotentialFieldValue(samplePos + vec3(0., 0., eps)) - getPotentialFieldValue(samplePos - vec3(0., 0., eps));

	dPdx *= oneOverTwoEps;
	dPdy *= oneOverTwoEps;
	dPdz *= oneOverTwoEps;

	vec3 velocity;
	velocity.x = dPdy.z - dPdz.y;
	velocity.y = dPdz.x - dPdx.z;
	velocity.z = dPdx.y - dPdy.x;

	// scale the magnitude
	velocity *= 3.;

	// compute the position
	vec3 oldPosition = in_Position.xyz;
	vec3 newPosition = clamp(oldPosition + dt * velocity, bbMin, bbMax);

	// boundaries conditions.
	//velocity = 
	//if (newPosition.x <= bbMin.x || newPosition.x >= bbMax.x) newPosition.x = oldPosition.x;
	//if (newPosition.y <= bbMin.y || newPosition.y >= bbMax.y) newPosition.y = oldPosition.y;
	//if (newPosition.z <= bbMin.z || newPosition.z >= bbMax.z) newPosition.z = oldPosition.z;

	// life
	float life = in_Life - .000001 * dt;

	if (life <= 0.)
	{
		life = 1.;
		// use the noise to determine the new spawn point
		newPosition = in_InitialPosition.xyz;//mix(bbMin, bbMax, n);
	}

	// shift the hue through time
	//vec4 shiftedColor = vec4(shiftHue(in_Color.rgb, 1. * dt), in_Color.a);
	vec4 shiftedColor = in_Color;

	out_Position = vec4(newPosition, 1.0);
	out_Color = shiftedColor;
	out_Velocity = velocity;
	out_Life = life;
	out_InitialPosition = in_InitialPosition;
}
